#include "Parser.h"

using namespace std;

Parser::Parser(string instructionFile, string csvFile, string registersFile, string traceFile, string registersFileFiltered, string graphFile) {
	this -> instructionFile = instructionFile;
	this -> csvFile = csvFile;
	this -> registersFile = registersFile;
	this -> traceFile = traceFile;
	this -> registersFileFiltered = registersFileFiltered;
	this -> graphFile = graphFile;
	this -> totalInstructions = 0;
}

//function used to create a map of instruction based on the instruction file
void Parser::buildInstructionMap(void) {
	if (this -> instructionMap.size() != 0)
		this -> instructionMap.clear();

	fstream instructionFile(this -> instructionFile);
	string line;

	while (getline(instructionFile, line)) {
		string name;
		string app;
		int pos;
		InstructionInfo instr;

		instr.type = line.substr(0, pos = line.find_first_of(" "));
		line.erase(0, (pos+1));

		name = line.substr(0, pos = line.find_first_of(" "));
		line.erase(0, (pos+1));

		app = line.substr(0, pos = line.find_first_of(","));
		line.erase(0, (pos+2));
		line.push_back(' ');

		while ((pos = app.find_first_of(" ")) != string::npos) {
			instr.dst.emplace_back(app.substr(0, pos));
			app.erase(0, (pos+1));
		}

		while ((pos = line.find_first_of(" ")) != string::npos) {
			instr.src.emplace_back(line.substr(0, pos));
			line.erase(0, (pos+1));
		}

		this -> instructionMap.emplace(name, instr);
	}

	instructionFile.close();
}

//function used to easily access a register value inside registers file
void Parser::buildRegisterPosMap(void) {
	if (this -> registerFilePosition.size() != 0)
		this -> registerFilePosition.clear();

	fstream registersFile(this -> registersFile);
	string line;

	if(getline(registersFile, line)) {
		int pos = 0;
		string reg;
		string::iterator it = line.begin();
		while (it != line.end()) {
			while (line[pos] != '=' && it != line.end()){
				reg.push_back(line[pos]);
				++pos;
				++it;
			}

			if (line[pos] != '=')
				return;

			++pos;
			++it;
			this -> registerFilePosition.emplace(reg, pos);
			if (reg.compare("PIR")) {
				pos += 9;
				it += 9;
			}
			else {
				pos += 8;
				it += 8;
			}

			reg = "";
		}
	}

	else {
		cout << "Error in register file" << endl;
		return;
	}

	registersFile.close();
}

//function used by automatic graph generator
void Parser::buildTypeMap(map<string, vector<string>>& mapType) {
	if (this -> instructionMap.size() == 0) {
		cerr << "Instruction map does not exist" << endl;
		return;
	}

	for (map<string, InstructionInfo>::iterator it = this -> instructionMap.begin(); it != this -> instructionMap.end(); ++it) {
		if (mapType.find(it -> second.type) == mapType.end()) {
			vector<string> a;
			mapType.emplace(it -> second.type, a);
		}

		if (!(it -> first.compare("e_stmw")) || !(it -> first.compare("e_lmw")) || !(it -> first.compare("lwarx")) || !(it -> first.compare("se_bc")) || regex_match(it -> first, regex("e_lmv.+")) || regex_match(it -> first, regex("e_stmv.+")) || regex_match(it -> first, regex("dcb.+")) || regex_match(it -> first, regex("icb.+")))
			continue;

		mapType[it -> second.type].emplace_back(it -> first);
	}
}

void Parser::createAddressesVector(string traceFileName) {
	fstream traceFile(traceFileName);
	string lineTrace;
	long long int IP = 0;
	int counter = 0;

	while (getline(traceFile, lineTrace)) {
		int pos;
		stringstream ss;
		string IPstring;
		string name;

		counter++;

		ss << hex << IP;
		for (int i = ss.str().size(); i < 8; ++i)
			IPstring += "0";
		IPstring += uppercase(ss.str());

		this -> addresses.emplace_back(IPstring);

		//not only instruction name
		if ((pos = lineTrace.find_first_of(" ")) != string::npos)
			name = lineTrace.substr(0, pos);
		//only instruction name
		else
			name = lineTrace.substr(0, string::npos);

		if (name[0] == 's')
			IP += 2;
		else
			IP += 4;
	}
	this -> totalInstructions = counter;
}

void Parser::createTraceFile(void) {
	fstream csvFile(this -> csvFile);
	fstream registersFile(this -> registersFile);
	fstream traceFile(this -> traceFile, ios::out);
	fstream registersFileFiltered(this -> registersFileFiltered, ios::out);
	fstream notFoundLog("Log.txt", ios::out);
	fstream debug_file("Debug_file.txt", ios::out);

	string lineCsv;
	string lineRegisters;
	this -> totalInstructions = 0;
	int lineNumber = 0;
	int countNotFound = 0;
	string linePrev = "";


	for (int numLine = 0; getline(csvFile,lineCsv); ++numLine) {

		string lineCsvCopy = lineCsv;
		lineNumber = (numLine == 3) ? lineNumber : (lineNumber + 1);

		switch (numLine) {
			case 0:
				//first line, nothing useful
				if (lineCsv.compare("\"B::ART.List\"\r") && lineCsv.compare("\"B::ART.List\"")) {
					cerr << "Error in the file at first position" << endl;
					return;
				}
				break;

			case 1:
				//second line, nothing useful
				if (lineCsv.substr(0, lineCsv.find_first_of(",")).compare("\"record\"")) {
					cerr << "Error in the file at second position" << endl;
					return;
				}
				break;

			case 2:
				//third line, control on address and save address 
				if (lineCsv.substr(0, lineCsv.find_first_of(",")).compare("\"T000000\"")) {
					cerr << "Error in the file at third position" << endl;
					return;
				}

				//add address to unordered set. It will be useful to check conditional branch addresses later
				else {
					while (getline(registersFile, lineRegisters)) {
						int pos = lineCsv.find_first_of(",");
						string address = lineCsv.substr(pos+5, 8);
						string IPvalue = lineRegisters.substr(this -> registerFilePosition["IP"], 8);
						if (!IPvalue.compare(address)) {
							this -> addresses.emplace_back(address);
							break;
						}
					}
				}
				break;

			case 3:
				//fourth line and next instructions --> loop while a real instruction name occurs
				do {

					lineCsvCopy = lineCsv;
					lineNumber++;

					//if line is empty continue
					if (!lineCsv.compare("\r")) {
						linePrev = lineCsvCopy;
						continue;
					}

					map<string,InstructionInfo>::iterator it;
					int pos = lineCsv.find_first_of("\"", 1);
					string name = lineCsv.substr(1,(pos-1));
					it = this -> instructionMap.find(name);

					//error in file, no istruction found
					if (!name.compare("B::ART.List")) {
						cerr << "Error in previous instruction at line --> " << lineNumber << endl;
						countNotFound++;
						if (!linePrev.compare(""))
							notFoundLog << "ERROR" << endl;
						else
							notFoundLog << linePrev << endl;

						name = "nop";
						it = this -> instructionMap.find(name);
					}

					//instruction not found in the map
					if (it == this -> instructionMap.end()) {
						linePrev = lineCsvCopy;
						continue;
					}

					//instruction found
					else {
						debug_file << lineCsv;

						if (!name.compare("nop"))
							numLine = 0;
						else
							numLine = -1;

						++(this -> totalInstructions);

						string operandsLine;
						vector<string> operands;

						//branch type instructions
						if (!(this -> instructionMap[name].type.compare("B")) || !(this -> instructionMap[name].type.compare("BC"))) {
							//check if there are explicit operands
							if (lineCsv.find(",") != string::npos) {
								lineCsv.erase(0, (pos+2));
								string address2jump;
								if (lineCsv.find(";") != string::npos)
									address2jump = lineCsv.substr(0, lineCsv.find_first_of(","));
								else
									address2jump = lineCsv.substr(0, lineCsv.find_first_of("\r"));
								operands.emplace_back(address2jump);
							}
						}

						
						//not branch type instructions
						else {
							//check if there are explicit operands
							if ((pos = lineCsv.find_first_of(",")) != string::npos) {
								//cmpl, wrteei and multiple volatile load/store without quotes (this should be not managed, TO BE REMOVED)
								if (!name.compare("cmpl") || !name.compare("cmpi") || !name.compare("mbar") || !name.compare("wrteei") || regex_match(name, regex("e_lmv.+")) || regex_match(name, regex("e_stmv.+"))) {
									lineCsv.erase(0, (pos+1));
									if (lineCsv.find(";") != string::npos)
										operandsLine = lineCsv.substr(0, lineCsv.find_first_of("\""));
									else
										operandsLine = lineCsv.substr(0, lineCsv.find_first_of("\r"));
								}
								
								else {
									lineCsv.erase(0, (pos+2));
									operandsLine = lineCsv.substr(0, lineCsv.find_first_of("\""));
									operandsLine.push_back(',');
								}

								while ((pos = operandsLine.find_first_of(",")) != string::npos) {

									//hex number
									if (regex_match(operandsLine.substr(0, pos), regex("-?0x[A-Za-z0-9]+"))) {
										operands.emplace_back(to_string(stoll(operandsLine.substr(0, pos), nullptr, 16)));
										operandsLine.erase(0, (pos+1));
									}

									//hex number(reg)
									else if (regex_match(operandsLine.substr(0, pos), regex("-?0x[A-Za-z0-9]+\\(.+\\)"))) {
										string app = operandsLine.substr(0, pos);
										string result;
										result += to_string(stoi(app.substr(0, app.find_first_of("(")), nullptr, 16));
										app.erase(0, app.find_first_of("("));
										result += app.substr(0, (app.find_first_of(")")+1));
										operands.emplace_back(result);
										operandsLine.erase(0, (pos+1));
									}

									//special register, take the one after ";"
									else if (regex_match(operandsLine.substr(0, pos), regex("spr.+"))) {
										string operandsLineSpecial = lineCsv.substr((lineCsv.find_first_of(";")+4), string::npos);
										operandsLineSpecial = operandsLineSpecial.substr(0, operandsLineSpecial.find_last_of("\""));
										operandsLineSpecial.push_back(',');
										operandsLine.erase(0, (pos+1));
										pos = operandsLineSpecial.find_first_of(",");
										for (int i = 0; i < operands.size(); ++i) {
											if (pos != string::npos) {
												operandsLineSpecial.erase(0, (pos+1));
												pos = operandsLineSpecial.find_first_of(",");
											}
										}
										operands.emplace_back(operandsLineSpecial.substr(0, pos));
									}

									//others
									else {
										operands.emplace_back(operandsLine.substr(0, pos));
										operandsLine.erase(0, (pos+1));
									}
								}
							}
						}

						//write one line of trace file
						traceFile << name;
						if (operands.size() != 0)
							traceFile << " ";

						//write one line of registers file
						registersFileFiltered << lineRegisters.substr(0, lineRegisters.find_first_of("\r")) << endl;

						for (int i = 0; i < operands.size(); ++i) {
							traceFile << operands[i];
							if (i < (operands.size()-1))
								traceFile << ",";
						}
						traceFile << endl;

						break;
					}
				}while (getline(csvFile, lineCsv));
				break;

			default:
				cerr << "Something went wrong, number of non negligible lines > 4";
				return;
		}
		linePrev = lineCsvCopy;
	}

	cout << "Number of total instructions --> " << this -> totalInstructions << endl;
	cout << "Number of instruction not identified --> " << countNotFound << endl;

	csvFile.close();
	registersFile.close();
	traceFile.close();
	registersFileFiltered.close();
	notFoundLog.close();
	debug_file.close();
}

void Parser::createGraphFile(void) {
	fstream traceFile(this -> traceFile);
	fstream registersFile(this -> registersFileFiltered);
	fstream graphFile(this -> graphFile, ios::out);


	if (this -> addresses.size() == 0) 
		this -> createAddressesVector(this -> traceFile);

	//write number of instructions
	graphFile << this -> totalInstructions << endl;

	string lineTrace;
	string lineRegisters;
	int index = -1;

	while (getline(traceFile, lineTrace)) {
		getline(registersFile, lineRegisters);

		index++;
		int pos;
		lineTrace.push_back(',');
		string name;

		//not only instruction name
		if ((pos = lineTrace.find_first_of(" ")) != string::npos)
			name = lineTrace.substr(0, pos);
		//only instruction name
		else
			name = lineTrace.substr(0, (pos = lineTrace.find_first_of(",")));

		map<string,InstructionInfo>::iterator it;
		vector<string> operands;

		//control if instruction is in the map
		it = this -> instructionMap.find(name);
		if (it == this -> instructionMap.end()) {
			cerr << "Error: instruction not found. Name --> " << name;
			continue;
		}

		//set operands
		lineTrace.erase(0, (pos+1));
		while ((pos = lineTrace.find_first_of(",")) != string::npos) {
			//not a memory operand
			if (lineTrace.substr(0,pos).find("(") == string::npos) {
				operands.emplace_back(lineTrace.substr(0, pos));
				lineTrace.erase(0, (pos+1));
			}

			//memory operand
			else {
				operands.emplace_back(lineTrace.substr(0, (pos = lineTrace.find_first_of("("))));
				lineTrace.erase(0, (pos+1));
				operands.emplace_back(lineTrace.substr(0, (pos = lineTrace.find_first_of(")"))));
				lineTrace.erase(0, (pos+2));
			}
		}

		//control if both jump addresses exists (conditional branches)
		if (!(this -> instructionMap[name].type.compare("BC"))) {
			stringstream ss;
			string IP = this -> addresses[index];
			long long int IPint = stoll(IP, nullptr, 16);
			string addressJump;
			bool found = false;

			for (int i = 1; i <= stoll(this -> instructionMap[name].src[0]); ++i) {
				if (regex_match(this -> instructionMap[name].src[i], regex("[0-9]{1}"))) {
					addressJump = operands[stoll(this -> instructionMap[name].src[i])-1];
					addressJump[1] = '0';
					break;
				}
			}

			//se_
			if (name[0] == 's')
				IPint += 2;
			//e_
			else
				IPint += 4;

			ss << hex << IPint;

			string addressSummed;
			for (int i = ss.str().size(); i < 8; ++i)
				addressSummed += ("0");
			addressSummed += ss.str();
			addressSummed = uppercase(addressSummed);

			if (!(this -> addresses[index+1].compare(addressSummed))) {
				for (int i = (index+2); i < this -> totalInstructions; ++i) {

					if (!(this -> addresses[i].compare(addressJump))) {
						graphFile << to_string(i-index) << " ";
						found = true;
						break;
					}
				}
			}

			else if (!(this -> addresses[index+1].compare(addressJump))) {
				for (int i = (index+2); i < this -> totalInstructions; ++i) {

					if (!(this -> addresses[i].compare(addressSummed))) {
						graphFile << to_string(i-index) << " ";
						found = true;
						break;
					}
				}
			}

			else
				cerr << "Error: instruction after conditional branch not valid " << index << endl;

			if (!found)
				graphFile << "DANGER ";
		}

		bool multipleWords = false;
		if (!name.compare("e_stmw") || !name.compare("e_lmw"))
			multipleWords = true;
		
		graphFile << "D ";

		//destinations
		for (int i = 1; i <= stoll(this -> instructionMap[name].dst[0]); ++i) {

			//implicit parameter
			if (regex_match(this -> instructionMap[name].dst[i], regex("[a-zA-Z]+"))) {
				//manage load multiple words
				if (!name.compare("e_lmw") && !instructionMap[name].dst[i].compare("GPR")) {
					long long int r;
					//register value
					if(operands[0][0] == 'r' || operands[0][0] == 'R')
						r = stoll(operands[0].substr(1, string::npos));
					//immediate value
					else
						r = stoll(operands[0]);

					while (r <= 31) {
						graphFile << "r" << r << " ";
						r++;
					}
				}

				//not load multiple word
				else 
					graphFile << this -> instructionMap[name].dst[i] << " ";
			}

			//explicit parameter
			else if (regex_match(this -> instructionMap[name].dst[i], regex("[0-9]{1}"))) {
				if (stoll(this -> instructionMap[name].dst[i]) > operands.size()) {
					cerr << "Error in instruction file: instruction " << name << " has too much destinations" << endl;
					return;
				}

				if (i != 1 && !(this -> instructionMap[name].dst[i].compare(this -> instructionMap[name].dst[i-1]))) {
					int curr_reg = stoll(operands[(stoll(this -> instructionMap[name].dst[i])-1)].substr(1, string::npos));
					graphFile << "r" << curr_reg << " ";
				}

				else
					graphFile << operands[(stoll(this -> instructionMap[name].dst[i])-1)] << " ";
			}

			//memory parameter
			else if (regex_match(this -> instructionMap[name].dst[i], regex("mem.+"))) {
				string addressMem = this -> resolveAddress(this -> instructionMap[name].dst[i], lineRegisters, operands, multipleWords);
				multipleWords = false;
				graphFile << addressMem << " ";
			}
		}

		graphFile << ", S ";

		//sources
		for (int i = 1; i <= stoll(this -> instructionMap[name].src[0]); ++i) {
	
			//implicit parameter
			if (regex_match(this -> instructionMap[name].src[i], regex("[a-zA-Z]+"))) {
				//manage store multiple words
				if (!name.compare("e_stmw") && !instructionMap[name].src[i].compare("GPR")) {
					long long int r;
					//register value
					if(operands[0][0] == 'r' || operands[0][0] == 'R')
						r = stoll(operands[0].substr(1, string::npos));
					//immediate value
					else
						r = stoll(operands[0]);

					while (r <= 31) {
						graphFile << "r" << r << " ";
						r++;
					}
				}

				//not store multiple word
				else 
					graphFile << this -> instructionMap[name].src[i] << " ";
			}

			//explicit parameter
			else if (regex_match(this -> instructionMap[name].src[i], regex("[0-9]{1}"))) {
				if (stoll(this -> instructionMap[name].src[i]) > operands.size()) {
					cerr << "Error in instruction file: instruction " << name << " has too much sources" << endl;
					return;
				}

				if (i != 1 && !(this -> instructionMap[name].src[i].compare(this -> instructionMap[name].src[i-1]))) {
					int curr_reg = stoll(operands[(stoll(this -> instructionMap[name].src[i])-1)].substr(1, string::npos));
					graphFile << "r" << curr_reg << " ";
				}
				else
					graphFile << operands[(stoll(this -> instructionMap[name].src[i])-1)] << " ";
			}

			//memory parameter
			else if (regex_match(this -> instructionMap[name].src[i], regex("mem.+"))) {
				string addressMem = this -> resolveAddress(this -> instructionMap[name].src[i], lineRegisters, operands, multipleWords);
				multipleWords = false;
				graphFile << addressMem << " ";
			}
		}

		graphFile << endl;
	}
	traceFile.close();
	registersFile.close();
	graphFile.close();
}

void Parser::makeStats(void) {

	map<string, int> instructionsFound;
	fstream trace(this -> traceFile);
	string instruction_txt;
	enum type {A, B, BC, L, S, SP}; 
	int type_counter[6] = {};
	int counter = 0;

	while(getline(trace, instruction_txt)) {
		instruction_txt.push_back(' ');
		int pos = instruction_txt.find(" ");
		string name;
		name = instruction_txt.substr(0,pos);

		map<string, int>::iterator it;
		it = instructionsFound.find(name);
		if (it != instructionsFound.end())
			instructionsFound[name]++;
		else
			instructionsFound.emplace(name, 1);

		//arithmetic
		if (!(this -> instructionMap[name].type.compare("A")))
			type_counter[A]++;

		//branch unconditional
		else if (!(this -> instructionMap[name].type.compare("B")))
			type_counter[B]++;

		// branch conditional
		else if (!(this -> instructionMap[name].type.compare("BC")))
			type_counter[BC]++;

		//load
		else if (!(this -> instructionMap[name].type.compare("L")))
			type_counter[L]++;

		//store
		else if (!(this -> instructionMap[name].type.compare("S")))
			type_counter[S]++;

		//special and compare
		else if (!(this -> instructionMap[name].type.compare("SP")) || (!this -> instructionMap[name].type.compare("C")))
			type_counter[SP]++;

		counter++;
	}

	trace.close();
	fstream statsFile("stats/stats.txt", ios::out);

	for(map<string, int>::iterator it = instructionsFound.begin(); it != instructionsFound.end(); ++it)
		statsFile << it -> first << " " << it -> second << " times" << endl;

	statsFile.close();


	cout << endl << "***STATISTIC RESULTS***" << endl << endl;
	for (int i = 0; i < 6; ++i) {
		string type_print;

		switch (i) {
			case 0:
				type_print = "Arithmeticals";
				break;
			case 1:
				type_print = "Unconditional Branches";
				break;
			case 2:
				type_print = "Conditional Branches";
				break;
			case 3:
				type_print = "Load";
				break;
			case 4:
				type_print = "Store";
				break;
			case 5:
				type_print = "Specials";
				break;
			default:
				cerr << "Error in make_stats switch" << endl;
				return;
		}

		cout << type_print << " --> " << (float)type_counter[i]/counter*100 << "%" << endl;
	}

	cout << "Total Number of Instructions --> " << counter << endl;
}


void Parser::setInstructionFile(string instructionFile) {
	this -> instructionFile = instructionFile;
}

void Parser::setCsvFile(string csvFile) {
	this -> csvFile = csvFile;
}

void Parser::setRegistersFile(string registersFile) {
	this -> registersFile = registersFile;
}

void Parser::setTraceFile(string traceFile) {
	this -> traceFile = traceFile;
}

void Parser::setRegistersFileFiltered(string registersFileFiltered) {
	this -> registersFileFiltered = registersFileFiltered;
}

void Parser::setGraphFile(string graphFile) {
	this -> graphFile = graphFile;
}

string Parser::getGraphFile(void) {
	return this -> graphFile;
}

string Parser::resolveAddress(string expression, string lineRegisters, vector<string> operands, bool multipleWords) {
	string result;
	string r1, r2;
	int pos;
	int totLocation = 1;
	int partial;

	//remove "mem("
	expression.erase(0, 4);

	//compute base address
	long long int baseAddress = 0;
	r1 = expression.substr(0, (pos = expression.find_first_of("+")));
	expression.erase(0, (pos+1));
	r2 = expression.substr(0, (pos = expression.find_first_of(")")));
	expression.erase(0, (pos+1));

	//register value r1
	if (operands[stoll(r1)-1][0] == 'r' || operands[stoll(r1)-1][0] == 'R')
		baseAddress += stoll(lineRegisters.substr(this -> registerFilePosition[uppercase(operands[stoll(r1)-1])], 8), nullptr, 16);
	//immediate value r1
	else
		baseAddress += stoll(operands[stoi(r1)-1]);

	//register value r2
	if (operands[stoll(r2)-1][0] == 'r' || operands[stoll(r2)-1][0] == 'R')
		baseAddress += stoll(lineRegisters.substr(this -> registerFilePosition[uppercase(operands[stoll(r2)-1])], 8), nullptr, 16);
	//immediate value r2
	else
		baseAddress += stoll(operands[stoi(r1)-1]);

	if (baseAddress < 0)
		baseAddress = baseAddress * -1;


	//append base address to result
	result += "mem(";
	result += to_string(baseAddress);
	result += ")";

	//compute number of locations
	if (expression.find("*") == string::npos)
		return result;

	else {
		expression.push_back('*');
		while ((pos = expression.find_first_of("*")) != string::npos) {
			expression.erase(0, (pos+1));

			string operand;
			partial = 0;

			if (!expression.compare("")) {
				result += to_string(totLocation);
				return result;
			}

			switch (expression[0]) {

				//immediate value
				case '$':
					partial = stoi(expression.substr(1, (pos = expression.find_first_of("*"))));
					expression.erase(0, pos);
					break;

				//inside bracket value
				case '(':
					//manage multiple words load/store
					if (multipleWords && regex_match(expression.substr(0, expression.find_first_of("*")), regex("[(][$]32[-][0-9]+[)]"))) {
						int index = stoi(expression.substr((expression.find_first_of("+-")+1), expression.find_first_of(")")));
						partial += (32 - stoi(operands[index-1].substr(1, string::npos)));
						break;
					}

					pos = expression.find_first_of("+-");
					operand = expression.substr(1, (pos-1));
					expression.erase(0, pos);

					//first immediate value
					if (operand[0] == '$')
						partial += stoi(operand.substr(1, string::npos));
					//first index value
					else {
						int index = stoi(operand);
						//register value
						if (operands[index-1][0] == 'r' || operands[index-1][0] == 'R') {
							int registerPosition = this -> registerFilePosition[uppercase(operands[index-1])];
							partial += stoi(lineRegisters.substr(registerPosition, 8), NULL, 16);
						}
						//immediate value
						else 
							partial += stoi(operands[index-1]);
					}

					operand = expression.substr(0, (pos = expression.find_first_of(")")));
					expression.erase(0, (pos+1));

					//second immediate value
					if (operand[1] == '$') {
						operand.erase(1);
						partial += stoi(operand);
					}

					//second index value
					else {
						int index = stoi(operand.substr(1, string::npos));
						string value;
						//register value
						if (operands[index-1][0] == 'r' || operands[index-1][0] == 'R') {
							int registerPosition = this -> registerFilePosition[uppercase(operands[index-1])];
							value = lineRegisters.substr(registerPosition, 8);
						}
						//immediate value
						else
							value = operands[index-1];

						if (operand[0] == '+')
							partial += stoi(value, nullptr, 16);
						else
							partial -= stoi(value, nullptr, 16);
					}
					break;

				default:
					cerr << "Error during the address resolving " << result <<  endl;
					return result;
			}
			totLocation = totLocation * partial;
		}
	}

	result += (to_string(totLocation));
	return result;
}

void Parser::printInstructionMap(void) {
	for (map<string,InstructionInfo>::iterator it = this -> instructionMap.begin(); it != instructionMap.end(); ++it) {
		cout << "Key --> " << it -> first << " Dst --> ";

		for (int i = 0; i < it -> second.dst.size(); ++i) 
			cout << it -> second.dst[i] << " ";

		cout << "Src --> ";
		for (int i = 0; i < it -> second.src.size(); ++i)
			cout << it -> second.src[i] << " ";

		cout << "Type --> " << it -> second.type << endl;
	}
}