% Questa function calcola i coefficienti aerodinamici del velivolo completo
% e le derivate aerodinamice della portanza (quelle di momento non vengono calcolate con questa function)
% 
% INPUT:
%
% DATI(struct) dalla function INPUT_DATI
% ALA_3D(struct) dalla function ala_3D
% DOWNWASH(struct) dalla function downwash
% alpha [rad] angolo (o vettore di angoli) di incidenza complessiva 
% delta_e [rad] angolo (o vettore di angoli) di deflessione
% dell'equilibratore
%
%
% OUTPUT:
%
% COEFF(struct) con campi:
%     .C_L [-] coefficiente di portanza del velivolo completo (dipendente da alpha e delta_e)
%     .C_M_G [-] coefficiente di momento del velivolo completo (dipendente da alpha e delta_e) rispetto al baricentro
% DERIV (struct) con campi:
%     .C_L_alpha [1/rad] derivata del coeff. di portanza del velivolo completo rispetto ad alpha
%     .C_L_delta_e [1/rad] derivata del coeff. di portanza del velivolo completo rispetto a delta_e
%     .C_M_G_alpha [1/rad] derivata del coeff. di momento del velivolo completo rispetto a delta_e rispetto a G
%     .C_M_G_delta_e [1/rad] derivata del coeff. di momento del velivolo completo rispetto a delta_e rispetto a G
%     .C_D_alpha [1/rad] derivata del coeff. di resistenza del velivolo completo rispetto ad alpha
%     .C_D_delta_e [1/rad] derivata del coeff. di resistenza del velivolo completo rispetto a delta_e 


function [COEFF, DERIV] = equaz_costitutive(DATI, ALA_3D, DOWNWASH, alpha, delta_e)

% INPUT:
i_w1 = DATI.velivolo.i_w1; % [rad] calettamento della w1; % calettamento della w1
i_w2 = DATI.velivolo.i_w2; % [rad] calettamento della w2
S_w1 = DATI.velivolo.S_w1; % [m2] Superficie della w_1
S_w2 = DATI.velivolo.S_w2  % [m2] Superficie della w2
l_w1 = ALA_3D.l_w1 	 % [m] apertura alare ala w1
l_w2 = ALA_3D.l_w2 	 % [m] apertura alare ala w2
AR_w1 = DATI.velivolo.AR_w1 	 % [-] Aspect ratio della w1
AR_w2 = DATI.velivolo.AR_w2 	 % [-] Aspect ratio della w2
x_G = DATI.velivolo.x_G 	 % [m] Coordinata longitud del baricentro nel sist di riferimento di prua
z_G = DATI.velivolo.z_G 	 % [m] Coordinata vert del baricentro nel sist di riferimento di prua
x_AC_w1 = DATI.velivolo.x_AC_w1 	 % [m] Coordinata longit. del centro aerodinamico della w1 nel sist di rif di prua
x_AC_w2 = DATI.velivolo.x_AC_w2 	 % [m] Coordinata longit. del centro aerodinamico della w2 nel sist di rif di prua
z_AC_w1 = DATI.velivolo.z_AC_w1 	 % [m] Coordinata vert. del centro aerodinamico della w1 nel sist di rif di prua
z_AC_w2 = DATI.velivolo.z_AC_w2 	 % [m] Coordinata vert. del centro aerodinamico della w2 nel sist di rif di prua
c_w1 = ALA_3D.c_w1 	 % [m] corda media geometrica di w1
c_w2 = ALA_3D.c_w2 	 % [m] corda media geometrica di w2

a_w1 = ALA_3D.a_w1 	 % [1/rad] pendenza CL_alpha dell'ala w1
a_w2 = ALA_3D.a_w2 	 % [1/rad] pendenza CL_alpha dell'ala w2
e_w1 = ALA_3D.e_w1 	 % [-] Fattore di Oswald dell'ala w1
e_w2 = ALA_3D.e_w2 	 % [-] Fattore di Oswald dell'ala w2
C_D_0_w1 = DATI.profili.ali.C_D_0 	% [-] Coefficiente di resistenza parassita del profilo dell'ala 
C_D_0_w2 = DATI.profili.ali.C_D_0 	% [-] Coefficiente di resistenza parassita del profilo dell'ala 
C_M_AC_0_w1 = DATI.profili.ali.C_M_AC_0 % [-] Coeff. di momento di trasporto del profilo rispetto al suo AC
C_M_AC_0_w2 = DATI.profili.ali.C_M_AC_0 % [-] Coeff. di momento di trasporto del profilo rispetto al suo AC
C_M_AC_delta_e_w2 = DATI.profili.ali.C_M_AC_delta_e     % [-] Derivata aerodinamica del momento focale rispetto a delta_e
le_lw2 = DATI.equilibratore.le_lw2; % [-] lunghezza dell'equilibratore (% della lungezza di l_w2)
tau_e_3D = ALA_3D.tau_e_3D; % [-] fattore tau dell'equilibratore 	 

C_D_fus = DATI.fusoliera.C_D_fus; % [-] Coefficiente di resistenza della fusoliera adimensionalizzato rispetto alla sua superficie frontale
S_fus = DATI.fusoliera.S_fus; % [-] Superficie frontale della fusoliera

eps_D_0 = DOWNWASH.eps_D_0 	 %[rad] angolo di downwash per alpha = 0
eps_D_alpha = DOWNWASH.eps_D_alpha 	 %[1/rad] pendenza eps/alpha downwash
eps_U_0 = DOWNWASH.eps_U_0 	 %[rad] angolo di upwash per alpha = 0
eps_U_alpha = DOWNWASH.eps_U_alpha 	 %[1/rad] pendenza eps/alpha upwash

% ------------------------------------------------------------------------------------------%
% ------------------------------------------------------------------------------------------%

K_w1 = ALA_3D.K_w1 	 % [-] fattore correttivo della reistenza indotta della box-wing per w1
K_w2 = ALA_3D.K_w2 	 % [-] fattore correttivo della reistenza indotta della box-wing per w2

eta = S_w2./S_w1; % rapporto tra S_w2 e S_w1
csi = c_w2./c_w1; % rapporto tra c_w2 e c_w1


% calettamento del velivolo complessivo
i = (a_w1.*(i_w1 + eps_U_0 + (i_w2 - eps_D_0).*eps_U_alpha) +a_w2.*(i_w2 - eps_D_0 - eps_D_alpha.*(i_w1 +eps_U_0)).*eta)...
    ./(a_w1 + a_w1.*eps_U_alpha +a_w2.*eta - a_w2.*eps_D_alpha.*eta);

b_w2 = a_w2.*tau_e_3D.*le_lw2;

h_alpha = - (-1 -eps_U_alpha)./(1 + eps_D_alpha.*eps_U_alpha); 
h_0 = - (i - i_w1 - eps_U_0 + i.*eps_U_alpha - i_w2.*eps_U_alpha + eps_D_0.*eps_U_alpha)./(1 + eps_D_alpha.*eps_U_alpha); 

k_alpha = - (-1 + eps_D_alpha)./(1 + eps_D_alpha.*eps_U_alpha); 
k_0 = - (i - i_w2 + eps_D_0 - i.*eps_D_alpha + i_w1.*eps_D_alpha + eps_D_alpha.*eps_U_0)./(1 + eps_D_alpha.*eps_U_alpha);

% DERIVATE AERODINAMICHE DELLA PORTANZA

C_L_alpha = a_w1.*h_alpha + eta.*a_w2.*k_alpha;
C_L_delta_e = eta.*b_w2;
C_L = C_L_alpha.*alpha + C_L_delta_e.*delta_e;


% DERIVATE DI RESISTENZA
C_D = S_fus./S_w1.*C_D_fus + C_D_0_w1 + (a_w1.^2.*K_w1.*(h_0+h_alpha.*alpha).^2)./(AR_w1.*e_w1.*pi) + eta.*(C_D_0_w2+  (K_w2.*(a_w2.*(k_0+k_alpha.*alpha)+b_w2.*delta_e).^2 )./(AR_w2.*e_w2.*pi));
C_D_alpha = (2.*a_w1.^2.*h_alpha.*K_w1.*(h_0+h_alpha.*alpha))./(AR_w1.*e_w1.*pi)  +  (2.*a_w2.*K_w2.*k_alpha.*(a_w2.*(k_0+k_alpha.*alpha) + b_w2.*delta_e).*eta )./(AR_w2.*e_w2.*pi);
C_D_delta_e = (2.*b_w2.*K_w2.*(a_w2.*(k_0+k_alpha.*alpha)+b_w2.*delta_e  ).*eta )./(AR_w2.*e_w2.*pi);


% Cambio di sistema di riferimento (da quello di prua a baricentrale)
X_AC_w1 = x_G - x_AC_w1 ; 
X_AC_w2 = x_G - x_AC_w2 ;  
Z_AC_w1 = z_G - z_AC_w1 ;
Z_AC_w2 = z_G - z_AC_w2 ;


% CALCOLO DELLE DERIVATE DI MOMENTO

C_M_G = C_M_AC_0_w1 + eta.*csi.*(C_M_AC_0_w2 + C_M_AC_delta_e_w2.*delta_e)...
        +(a_w1.*X_AC_w1.*(h_0 + h_alpha.*alpha))./c_w1...
        -(Z_AC_w1.*(C_D_0_w1 + (a_w1.^2.*K_w1.*(h_0+h_alpha.*alpha).^2)./(AR_w1.*e_w1.*pi)))./c_w1...
        +(X_AC_w2.*(a_w2.*(k_0+k_alpha.*alpha)+ b_w2.*delta_e).*eta)./c_w1...
        -(Z_AC_w2.*eta.*(C_D_0_w2 + (K_w2.*(a_w2.*(k_0+k_alpha.*alpha)+ b_w2.*delta_e).^2)./(AR_w2.*e_w2.*pi)))./(c_w1)


C_M_G_alpha = (a_w1.*h_alpha.*X_AC_w1)./c_w1...
              -(2.*a_w1.^2.*h_alpha.*K_w1.*Z_AC_w1.*(h_0+h_alpha.*alpha))./(AR_w1.*c_w1.*e_w1.*pi)...
              +(a_w2.*k_alpha.*X_AC_w2.*eta)./c_w1...
              -(2.*a_w2.*K_w2.*k_alpha.*Z_AC_w2.*(a_w2.*(k_0+k_alpha.*alpha)+b_w2.*delta_e).*eta)./(AR_w2.*c_w1.*e_w2.*pi);
          
C_M_G_delta_e = (b_w2.*X_AC_w2.*eta)./c_w1...
                -(2.*b_w2.*K_w2.*Z_AC_w2.*(a_w2.*(k_0+k_alpha.*alpha) +b_w2.*delta_e).*eta)./(AR_w2.*c_w1.*e_w2.*pi)...
                + C_M_AC_delta_e_w2.*eta.*csi;



% ------------------------------------------------------------------------------------------%
% ------------------------------------------------------------------------------------------%

% OUTPUT: 
COEFF = struct();
COEFF.C_M_G = C_M_G; % [-] coefficiente di momento del velivolo completo (dipendente da alpha e delta_e) rispetto al baricentro
COEFF.C_L = C_L; % [-] coefficiente di portanza del velivolo completo (dipendente da alpha e delta_e)
COEFF.C_D = C_D; % [-] coefficiente di resistenza del velivolo completo (dipendente da alpha e delta_e)

DERIV = struct();
DERIV.C_L_alpha = C_L_alpha; % [1/rad] derivata del coeff. di portanza del velivolo completo rispetto ad alpha
DERIV.C_L_delta_e = C_L_delta_e; % [1/rad] derivata del coeff. di portanza del velivolo completo rispetto a delta_e
DERIV.C_D_alpha = C_D_alpha; % [1/rad] derivata del coeff. di resistenza del velivolo completo rispetto ad alpha
DERIV.C_D_delta_e = C_D_delta_e; % [1/rad] derivata del coeff. di resistenza del velivolo completo rispetto a delta_e
DERIV.C_M_G_alpha = C_M_G_alpha; % [1/rad] derivata del coeff. di momento del velivolo completo rispetto a delta_e rispetto a G
DERIV.C_M_G_delta_e = C_M_G_delta_e; % [1/rad] derivata del coeff. di momento del velivolo completo rispetto a delta_e rispetto a G

end
