//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : Inspector Spy ME
//           (employed in PIPE Monitor Checker | UNCHANGED)
//-----------------------------------------------------------------------------
// Description: UART Serializer
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

module uart_tx #(
  parameter C_CLKS_PER_BIT     = 16'd976 // 128000 baud rate with 125 MHz clock
  )
  (
  input wire clk,
  input wire rst_n,
  input wire enable,
  input wire i_tx_dv,
  input wire [8-1:0] i_tx_byte,
  output reg o_tx_active,
  output reg o_tx_done,
  output reg o_tx_serial
  );
  
  localparam[4-1:0] S_IDLE          = 4'h0,
                    S_TX_START_BIT  = 4'h1,
                    S_TX_DATA_BITS  = 4'h2,
                    S_TX_PARITY_BIT = 4'h3,
                    S_TX_STOP_BIT   = 4'h4,
                    S_CLEANUP       = 4'h5;
                  
  reg [8-1:0]   r_tx_data;
  reg           r_i_tx_dv;
  reg [16-1:0]  r_clock_count;
  reg [3-1:0]   r_bit_index;
  reg [4-1:0]   r_sm_main;
  reg           odd_parity_bit;
  
  
  always @(negedge rst_n or posedge clk)
  begin
    if (~rst_n) begin
      // Outputs
      o_tx_active   <= 1'b0;
      o_tx_done     <= 1'b0;
      o_tx_serial   <= 1'b1;
      // Internal signals
      r_tx_data     <= 8'b0;
      r_clock_count <= 16'b0;
      r_bit_index   <= 3'b0;
      odd_parity_bit<= 1'b0;
      r_i_tx_dv     <= 1'b0;
      r_sm_main     <= S_IDLE; 
    end else begin
      if (enable == 1'b0) begin
        // Outputs
        o_tx_active   <= 1'b0;
        o_tx_done     <= 1'b0;
        o_tx_serial   <= 1'b1;
        // Internal signals
        r_tx_data     <= 8'b0;
        r_clock_count <= 16'b0;
        r_bit_index   <= 3'b0;
        odd_parity_bit<= 1'b0;
        r_i_tx_dv     <= 1'b0;
        r_sm_main     <= S_IDLE;  
      end else begin
        r_i_tx_dv <= i_tx_dv;
        // Control RX state machine
        case (r_sm_main)
          S_IDLE :
          begin
            if (i_tx_dv == 1'b1 && r_i_tx_dv == 1'b0) begin  //Start transmission detected
              o_tx_active  <= 1'b1;
              r_tx_data    <= i_tx_byte;
              r_sm_main    <= S_TX_START_BIT;
            end
          end
          
          // Send out Start bit. Start bit = 0
          S_TX_START_BIT :
          begin
            o_tx_serial     <= 1'b0;
            odd_parity_bit  <= ((r_tx_data[0]^r_tx_data[1])^(r_tx_data[2]^r_tx_data[3])^(r_tx_data[4]^r_tx_data[5])^(r_tx_data[6]^r_tx_data[7]))^1'b1;  // Assignement not done in "S_TX_PARITY_BIT" state to avoid "o_tx_serial" event delay
            // Wait C_CLKS_PER_BIT -1 clock cycles for start bit to finish
            if (r_clock_count < C_CLKS_PER_BIT-1'b1) begin
              r_clock_count <= r_clock_count + 1'b1;
            end else if (r_clock_count == C_CLKS_PER_BIT-1'b1) begin
              r_clock_count <= 16'b0;  // reset counter
              r_sm_main <= S_TX_DATA_BITS;  
            end
          end
            
          // Wait C_CLKS_PER_BIT -1 clock cycles for data bit to finish
          S_TX_DATA_BITS :
          begin
            o_tx_serial <= r_tx_data[r_bit_index];
            if (r_clock_count < C_CLKS_PER_BIT-1'b1) begin
              r_clock_count <= r_clock_count + 1'b1;
            end else if (r_clock_count == C_CLKS_PER_BIT-1'b1) begin
              r_clock_count  <= 16'b0;
              if (r_bit_index < 3'd7) begin 
                r_bit_index <= r_bit_index + 1'b1;
              end else if (r_bit_index == 3'd7)begin
                r_bit_index <= 3'b0;
                r_sm_main   <= S_TX_PARITY_BIT;
              end
            end
          end
          
          // Odd parity bit generator 
          S_TX_PARITY_BIT :
          begin
            o_tx_serial <= odd_parity_bit;
            if (r_clock_count < C_CLKS_PER_BIT-1'b1) begin
              r_clock_count <= r_clock_count + 1'b1;
            end else if (r_clock_count == C_CLKS_PER_BIT-1'b1) begin
              r_clock_count  <= 16'b0;
              r_sm_main      <= S_TX_STOP_BIT;
            end
          end
          
          // Send out Stop bit. Stop bit = 1
          S_TX_STOP_BIT :
          begin
            o_tx_serial <= 1'b1;
            // Wait C_CLKS_PER_BIT -1 clock cycles for Stop bit to finish
            if (r_clock_count < C_CLKS_PER_BIT-1'b1) begin
              r_clock_count <= r_clock_count + 1'b1;
            end else if (r_clock_count == C_CLKS_PER_BIT-1'b1) begin
              o_tx_done     <= 1'b1;
              o_tx_active   <= 1'b0;
              r_clock_count <= 16'b0;
              r_sm_main     <= S_CLEANUP;
            end
          end
          
          S_CLEANUP :
          begin
            o_tx_done     <= 1'b0;
            o_tx_serial   <= 1'b1;  // Drive line high for Idle
            r_clock_count <= 16'b0;
            r_bit_index   <= 3'b0;
            r_sm_main     <= S_IDLE;
          end
            
          default : 
          begin
            o_tx_done     <= 1'b0;
            o_tx_serial   <= 1'b1;  // Drive line high for Idle
            r_clock_count <= 16'b0;
            r_bit_index   <= 3'b0;
            r_sm_main     <= S_IDLE;
          end
        endcase
      end
    end
  end
  
endmodule //uart_tx