//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : Inspector Spy ME
//           (employed in PIPE Monitor Checker | UNCHANGED)
//-----------------------------------------------------------------------------
// Description: UART Deserializer
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

module uart_rx #(
  parameter C_CLKS_PER_BIT     = 16'd976 // 128000 baud rate with 125 MHz clock
  )
  (
  input wire clk,
  input wire rst_n,
  input wire enable,
  input wire i_rx_serial,
  output reg o_rx_dv,
  output reg [8-1:0] o_rx_byte,
  output reg parity_error,
  output reg start_bit_error
  );
  
  localparam[4-1:0] S_IDLE            = 4'h0,
                    S_RX_START_BIT    = 4'h1,
                    S_RX_DATA_BITS    = 4'h2,
                    S_RX_PARITY_BIT   = 4'h3,
                    S_RX_STOP_BIT     = 4'h4,
                    S_CLEANUP         = 4'h5;
                  
  
  reg           i_rx_serial_r;
  reg [16-1:0]  r_clock_count;
  reg [3-1:0]   r_bit_index;
  reg [4-1:0]   r_sm_main;
  reg           odd_parity_bit;
  
  always @(negedge rst_n or posedge clk)
  begin
    if (~rst_n) begin
      // Outputs
      o_rx_byte       <= 8'b0;
      o_rx_dv         <= 1'b0;
      parity_error    <= 1'b0;
      start_bit_error <= 1'b0;
      // Internal signals
      i_rx_serial_r   <= 1'b1;
      r_clock_count   <= 16'b0;
      r_bit_index     <= 3'b0;
      odd_parity_bit  <= 1'b0;
      r_sm_main       <= S_IDLE;  
    end else begin
      if (enable == 1'b0) begin
        // Outputs
        o_rx_dv         <= 1'b0;
        o_rx_byte       <= 8'b0;
        parity_error    <= 1'b0;
        start_bit_error <= 1'b0;
        // Internal signals
        i_rx_serial_r   <= 1'b1;
        r_clock_count   <= 16'b0;
        r_bit_index     <= 3'b0;
        odd_parity_bit  <= 1'b0;
        r_sm_main       <= S_IDLE;   
      end else begin
        i_rx_serial_r   <= i_rx_serial;
        // Control RX state machine
        case (r_sm_main)
          S_IDLE :
          begin
            if (i_rx_serial == 1'b0 && i_rx_serial_r == 1'b1) begin  //Start bit detected
              r_sm_main <= S_RX_START_BIT;
            end
          end
          
          // Check middle of start bit to make sure it is still low
          S_RX_START_BIT :
          begin
            if (r_clock_count < ((C_CLKS_PER_BIT-1'b1)>>1)) begin
              r_clock_count <= r_clock_count + 1'b1;
            end else if (r_clock_count == ((C_CLKS_PER_BIT-1'b1)>>1)) begin
              if (i_rx_serial == 1'b1) begin
                start_bit_error <= 1'b1;
              end else begin
                start_bit_error <= 1'b0;
              end
              r_clock_count <= 16'b0;  // reset counter, found the middle
              o_rx_byte     <= 8'b0;
              r_sm_main     <= S_RX_DATA_BITS;
            end
          end
            
          // Wait C_CLKS_PER_BIT -1 clock cycles to sample serial data   
          S_RX_DATA_BITS :
          begin
            if (r_clock_count < C_CLKS_PER_BIT-1'b1) begin
              r_clock_count <= r_clock_count + 1'b1;
            end else if (r_clock_count == C_CLKS_PER_BIT-1'b1) begin
              r_clock_count           <= 16'b0;
              o_rx_byte[r_bit_index]  <= i_rx_serial;
              if (r_bit_index < 3'd7) begin 
                r_bit_index <= r_bit_index + 1'b1;
              end else if (r_bit_index == 3'd7)begin
                r_bit_index <= 3'b0;
                r_sm_main   <= S_RX_PARITY_BIT;
              end
            end
          end
          
          // Odd parity bit checker
          S_RX_PARITY_BIT : 
          begin
            odd_parity_bit <= ((o_rx_byte[0]^o_rx_byte[1])^(o_rx_byte[2]^o_rx_byte[3])^(o_rx_byte[4]^o_rx_byte[5])^(o_rx_byte[6]^o_rx_byte[7]))^1'b1;
            if (r_clock_count < C_CLKS_PER_BIT-1'b1) begin
              r_clock_count <= r_clock_count + 1'b1;
            end else if (r_clock_count == C_CLKS_PER_BIT-1'b1) begin
              r_clock_count  <= 16'b0;
              r_sm_main      <= S_RX_STOP_BIT;
              if (i_rx_serial != odd_parity_bit) begin
                parity_error <= 1'b1;
              end
            end
          end
          
          // Receive Stop bit.
          S_RX_STOP_BIT :
          begin
            // Wait C_CLKS_PER_BIT -1 clock cycles for Stop bit to finish
            if (r_clock_count < C_CLKS_PER_BIT-1'b1) begin
              r_clock_count <= r_clock_count + 1'b1;
            end else if (r_clock_count == C_CLKS_PER_BIT-1'b1) begin
              o_rx_dv       <= 1'b1;
              r_clock_count <= 16'b0;
              r_sm_main     <= S_CLEANUP;
            end
          end
          
          S_CLEANUP :
          begin
            o_rx_dv         <= 1'b0;
            start_bit_error <= 1'b0;
            parity_error    <= 1'b0;
            r_clock_count   <= 16'b0;
            r_bit_index     <= 3'b0;
            r_sm_main       <= S_IDLE;
          end
            
          default : 
          begin
            o_rx_dv         <= 1'b0;
            start_bit_error <= 1'b0;
            parity_error    <= 1'b0;
            r_clock_count   <= 16'b0;
            r_bit_index     <= 3'b0;
            r_sm_main       <= S_IDLE;
          end
        endcase
      end
    end
  end
  
endmodule //uart_rx