//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Uart to AXI Top module. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) UART Serializer
//                (3) UART Deserializer
//                (4) UART Controller
//
// Dependency : uart_rx; uart_rx, uart_axi_ps
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

module uart_axi_top #(
    //-------------------------------------------------------------------------------
    parameter C_CLKS_PER_BIT  = 16'd976 , // 128000 baud rate with 125 MHz clock
    parameter G_ADDR_WIDTH    = 64      ,
    parameter G_DATA_WIDTH    = 512
    //-------------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------------
    // Resets, Enable and Clock
    input wire                                                  en                  ,
    input wire                                                  arstn               ,
    input wire                                                  clk                 ,

    //-------------------------------------------------------------------------------
    // UART Serial Port
    input  wire                                                 uart_rx             ,
    output wire                                                 uart_tx             ,
    //-------------------------------------------------------------------------------
    // AXI4 Slave Interface
    //  . Read Address Channel                                 
    output wire [4                                -1:0] arid                  ,
    output wire [64                               -1:0] araddr                ,
    output wire [4                                -1:0] arregion              , // NOT USED
    output wire [8                                -1:0] arlen                 , // 8'h0
    output wire [3                                -1:0] arsize                , // NOT USED | Byte for transfer
    output wire [2                                -1:0] arburst               , // x00
    output wire                                         arlock                , // Locked Transaction
    output wire [4                                -1:0] arcache               , // NOT USED
    output wire [3                                -1:0] arprot                , // NOT USED
    output wire [4                                -1:0] arqos                 , // NOT USED
    output wire                                         arvalid               ,
    input  wire                                         arready               ,
    //  . Read Data Channel
    input  wire [4                                -1:0] rid                   ,
    input  wire [G_DATA_WIDTH                     -1:0] rdata                 ,
    input  wire [2                                -1:0] rresp                 ,
    input  wire                                         rlast                 ,
    input  wire                                         rvalid                ,
    output wire                                         rready
    //-------------------------------------------------------------------------------
    );

    //---------------------------------------------------------------------------
    // Local Declarations
    //---------------------------------------------------------------------------
    wire                        rx_datas_rdy        ;
    wire                        tx_datas_start      ;
    
    reg                         uart_enable_r       ;
    reg                         uart_enable_rr      ;
    reg                         uart_rx_r           ;
    reg                         uart_rx_rr          ;

    wire [8               -1:0] rx_datas            ;
    wire                        rx_datas_rdy_int    ;
    wire                        rx_parity_error     ;
    wire                        rx_start_bit_error  ;

    wire                        tx_datas_start_int  ;
    wire [8               -1:0] tx_datas            ;
    wire                        tx_busy             ;


    //---------------------------------------------------------------------------
    // Assignations
    //---------------------------------------------------------------------------
    assign  rx_datas_rdy = rx_datas_rdy_int;
    assign  tx_datas_start = tx_datas_start_int;


    always @(negedge arstn or posedge clk)
    begin
      if (arstn==1'b0) begin
        uart_rx_r       <= 1'b1;
        uart_rx_rr      <= 1'b1;
        uart_enable_r   <= 1'b1;
        uart_enable_rr  <= 1'b1;
      end else begin
        uart_rx_r       <= uart_rx;
        uart_rx_rr      <= uart_rx_r;
        uart_enable_r   <= en;
        uart_enable_rr  <= uart_enable_r;
      end
    end


    //---------------------------------------------------------------------------
    // UART Serializer and Deserializer
    //---------------------------------------------------------------------------
    uart_rx #(
      .C_CLKS_PER_BIT(C_CLKS_PER_BIT)
    ) uart_rx_inst  (
      .clk              (clk                ),
      .rst_n            (arstn              ),
      .enable           (uart_enable_rr     ),
      .i_rx_serial      (uart_rx_rr         ),
      .o_rx_dv          (rx_datas_rdy_int   ),
      .o_rx_byte        (rx_datas           ),
      .parity_error     (rx_parity_error    ),
      .start_bit_error  (rx_start_bit_error )
    );

    uart_tx #(
      .C_CLKS_PER_BIT(C_CLKS_PER_BIT)
    ) uart_tx_inst  (
      .clk              (clk           ),
      .rst_n            (arstn          ),
      .enable           (uart_enable_rr     ),
      .i_tx_dv          (tx_datas_start_int ),
      .i_tx_byte        (tx_datas           ),
      .o_tx_active      (tx_busy            ),
      .o_tx_serial      (uart_tx            ),
      .o_tx_done        ( )
    );
    

    //---------------------------------------------------------------------------
    // UART Controller with AXI Interface
    //---------------------------------------------------------------------------
    uart_axi_ps #(
      .G_ADDR_WIDTH (G_ADDR_WIDTH ),
      .G_DATA_WIDTH (G_DATA_WIDTH )
    ) uart_ps_inst  (
      //--------------------------------------
      //  Clocks, EN and Resets
      .en               (en                 ),
      .arstn            (arstn              ),
      .clk              (clk                ),      
      //--------------------------------------
      // UART Interface
      //  . UART RX
      .rx_datas_rdy     (rx_datas_rdy_int   ),
      .rx_datas         (rx_datas           ),
      .parity_error     (rx_parity_error    ),
      .start_bit_error  (rx_start_bit_error ),
      //  . UART TX
      .tx_busy          (tx_busy            ),
      .tx_datas_start   (tx_datas_start_int ),
      .tx_datas         (tx_datas           ),
      //--------------------------------------
      // AXI4 Slave Interface
      //  . Read Address Channel  
      .arid             (arid               ),
      .araddr           (araddr             ),
      .arregion         (arregion           ),
      .arlen            (arlen              ),
      .arsize           (arsize             ),
      .arburst          (arburst            ),
      .arlock           (arlock             ),
      .arcache          (arcache            ),
      .arprot           (arprot             ),
      .arqos            (arqos              ),
      .arvalid          (arvalid            ),
      .arready          (arready            ),
      //  . Read Data Channel
      .rid              (rid                ),
      .rdata            (rdata              ),
      .rresp            (rresp              ),
      .rlast            (rlast              ),
      .rvalid           (rvalid             ),
      .rready           (rready             )
      //--------------------------------------
    );

endmodule