//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : Inspector Spy ME
//          (employed in PIPE Monitor Checker | MODIFIED by nrigotti)
//-----------------------------------------------------------------------------
// Description: UART Controller with AXI Interface
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module uart_axi_ps
#(
  //-------------------------------------------------------------------------
  // Constants
  parameter   G_ADDR_WIDTH  = 64,
  parameter   G_DATA_WIDTH  = 512
  //-------------------------------------------------------------------------
) (
  //-------------------------------------------------------------------------
  // Enable, Clock and Reset
  input wire                                          en                    ,
  input wire                                          arstn                 ,
  input wire                                          clk                   ,
  
  //-------------------------------------------------------------------------
  // UART Interface
  //  . UART RX
  (* mark_debug = "true" *) input wire                                          rx_datas_rdy          ,
  (* mark_debug = "true" *) input wire [8                                 -1:0] rx_datas              ,
  (* mark_debug = "true" *) input wire                                          parity_error          ,
  (* mark_debug = "true" *) input wire                                          start_bit_error       ,
  
  //  . UART TX
  (* mark_debug = "true" *) input wire                                          tx_busy,
  (* mark_debug = "true" *) output reg                                          tx_datas_start        ,
  (* mark_debug = "true" *) output reg [8                                 -1:0] tx_datas              ,

  //-------------------------------------------------------------------------
  // AXI4 Slave Interface
  //  . Read Address Channel                                         
  output wire [4                                -1:0] arid                  ,
  output wire [64                               -1:0] araddr                ,
  output wire [4                                -1:0] arregion              , // NOT USED
  output wire [8                                -1:0] arlen                 , // 8'h0
  output wire [3                                -1:0] arsize                , // NOT USED | Byte for transfer
  output wire [2                                -1:0] arburst               , // x00
  output wire                                         arlock                , // Locked Transaction
  output wire [4                                -1:0] arcache               , // NOT USED
  output wire [3                                -1:0] arprot                , // NOT USED
  output wire [4                                -1:0] arqos                 , // NOT USED
  output wire                                         arvalid               ,
  input  wire                                         arready               ,
  //  . Read Data Channel
  input  wire [4                                -1:0] rid                   ,
  input  wire [G_DATA_WIDTH                     -1:0] rdata                 ,
  input  wire [2                                -1:0] rresp                 ,
  input  wire                                         rlast                 ,
  input  wire                                         rvalid                ,
  output wire                                         rready
  //-------------------------------------------------------------------------
  );

  //-------------------------------------------------------------------------
  // Local declarations 
  //-------------------------------------------------------------------------
  // UART Parameters
  localparam UART_DATA_WIDTH = 16'd8;

  // UART FSM signals
  localparam[4     -1:0] FSM_UART_IDLE    =                     4'h0,
                         FSM_UART_ADDR    = FSM_UART_IDLE     + 4'h1,
                         FSM_UART_RD_AXI  = FSM_UART_ADDR     + 4'h1,
                         FSM_UART_WR_UART = FSM_UART_RD_AXI   + 4'h1,
                         FSM_UART_WR_SPY  = FSM_UART_WR_UART  + 4'h1;

  // N CYCLES to receive or transmmit using UART
  localparam NC_RCVD_ADDR = 16'd1;
  localparam NC_RCVD_DATA = 16'd1;
  localparam NC_TRSM_DATA = G_DATA_WIDTH / UART_DATA_WIDTH;
  
  // AXI4 Slave Interface
  //  . Read Address Channel                                         
  reg [4                                -1:0] reg_arid              ;
  reg [64                               -1:0] reg_araddr            ;
  reg [8                                -1:0] reg_arlen             ; // 8'h0
  reg [2                                -1:0] reg_arburst           ; // x00
  reg                                         reg_arlock            ; // Locked Transaction
  reg                                         reg_arvalid           ;
  //  . Read Data Channel
  reg                                         reg_rready            ;

  // UART Interface
  reg [G_DATA_WIDTH                     -1:0] rdata_r               ;
  reg                                         rx_datas_rdy_r        ;

  // SYNC FSM signals
  localparam[4     -1:0]  FSM_SYNC_P0       = 4'h0,
                          FSM_SYNC_P1       = 4'h1,
                          FSM_SYNC_P2       = 4'h2,
                          FSM_SYNC_P3       = 4'h3;
  
  reg [4                                -1:0] fsm_sync              ;
  (* mark_debug = "true" *) reg                                         sync_signal_detected  ;
  reg                                         sync_signal_detected_r;
  
  // UART FSM signals
  (* mark_debug = "true" *) reg [4                                -1:0] fsm_uart              ;
  reg                                         rd_wrn                ;
  (* mark_debug = "true" *) reg [16                                -1:0] cnt                   ;
  

  //-------------------------------------------------------------------------
  // Output Assignation
  //------------------------------------------------------------------------- 
  assign arid     = reg_arid    ;
  assign araddr   = reg_araddr  ;
  assign arregion = 'h0  ;
  assign arlen    = reg_arlen   ;
  assign arsize   = 'h0  ;
  assign arburst  = reg_arburst ;
  assign arlock   = reg_arlock  ;
  assign arcache  = 'h0 ;
  assign arprot   = 'h0 ;
  assign arqos    = 'h0 ;
  assign arvalid  = reg_arvalid ;

  assign rready   = reg_rready  ;

  //-------------------------------------------------------------------------
  // UART Controller FSM
  //-------------------------------------------------------------------------
  always @(negedge arstn or posedge clk)
  begin
    if (~arstn) begin
      // Outputs UART
      tx_datas_start  <= 1'b0;
      tx_datas        <= 8'b0;
      //Output AXI
      reg_arid        <= 4'h0;
      reg_arburst     <=  `AXI_AxBURST_INCR;
      reg_arlock      <=  1'b0;
      reg_araddr      <= 64'h0;
      reg_arlen       <=  8'h0;
      reg_arvalid     <=  1'b0;

      reg_rready      <=  1'b0;
      // Internal signals
      fsm_uart        <= FSM_UART_IDLE;
      rd_wrn          <= 1'b1;
      cnt             <= 16'h0;
      rdata_r         <= {(G_DATA_WIDTH){1'b0}};
      rx_datas_rdy_r  <= 1'b0;
    end else begin
      if (en == 1'b0) begin
        // Output UART
        tx_datas_start  <= 1'b0;
        tx_datas        <= 8'b0;
        //Output AXI
        reg_arid        <= 4'h0;
        reg_arburst     <=  `AXI_AxBURST_INCR;
        reg_arlock      <=  1'b0;
        reg_araddr      <= 64'h0;
        reg_arlen       <=  8'h0;
        reg_arvalid     <=  1'b0;

        reg_rready      <=  1'b0;
        // Internal signals
        fsm_uart        <= FSM_UART_IDLE;
        rd_wrn          <= 1'b1;
        cnt             <= 4'h0;
        rdata_r         <= {(G_DATA_WIDTH){1'b0}};
        rx_datas_rdy_r  <= 1'b0;
      end else begin  
        rx_datas_rdy_r <= rx_datas_rdy;
        if (sync_signal_detected == 1'b1 && sync_signal_detected_r == 1'b0) begin 
          fsm_uart  <= FSM_UART_IDLE;
        end else begin 
          case (fsm_uart)
          //-------------------------------------------------------------------
          FSM_UART_IDLE : // Wait for command word from SW. Bit 0 indicates read/write access to registers 0x00
          //-------------------------------------------------------------------
          begin
            // FSM
            cnt             <= 16'h0;
            // AXI
            reg_araddr      <= 64'h0;
            reg_rready      <=  1'b0;
            // UART TX
            tx_datas_start <= 1'b0;
            tx_datas       <= 8'b0;

            if (rx_datas_rdy == 1'b1 && rx_datas_rdy_r == 1'b0) begin
              if (parity_error || start_bit_error || (rx_datas != 8'h00 && rx_datas != 8'h01)) begin
                fsm_uart <= FSM_UART_IDLE;
              end else begin
                fsm_uart    <= FSM_UART_ADDR;
                rd_wrn      <= rx_datas[0];
              end  
              tx_datas        <= {6'b0,start_bit_error,parity_error};
              tx_datas_start  <= 1'b1;
            end
          end
          //-------------------------------------------------------------------
          FSM_UART_ADDR : // Send Received Address from UART to AXI 0x01
          //-------------------------------------------------------------------
          begin
            tx_datas_start <= 1'b0;


            if (rx_datas_rdy == 1'b1 && rx_datas_rdy_r == 1'b0) begin // rx_datas_rdy rising edge
              if (parity_error || start_bit_error) begin
                fsm_uart <= FSM_UART_IDLE;
              end else begin

                if (rd_wrn) begin
                  reg_araddr [cnt*UART_DATA_WIDTH +:UART_DATA_WIDTH] <= rx_datas; // Load Read Address Byte 
                end else begin
                  // Load Write Address Byte not implemented yet
                end
                
                // If the address is bigger then one byte (TODO)
                if (cnt < NC_RCVD_ADDR - 1) begin
                  cnt <= cnt + 16'b1;
                  fsm_uart    <= FSM_UART_ADDR;
                end
                else begin // Full Address Received
                  cnt <= 16'h0;
                  if (rd_wrn) begin
                    fsm_uart <= FSM_UART_RD_AXI;
                    // AXI Read Request
                    reg_arburst <= `AXI_AxBURST_INCR;
                    reg_arlen   <= 8'h0             ;
                    reg_arlock  <= 1'b1             ;
                    reg_arvalid <= 1'b1             ;
                  end else begin
                    fsm_uart <= FSM_UART_IDLE; // Write on RAM not implemented yet
                  end
                end
              end
              tx_datas_start  <= 1'b1;
              tx_datas        <= {6'b0,start_bit_error,parity_error};
            end
          end
          //-------------------------------------------------------------------
          FSM_UART_RD_AXI : // Read Received Data from AXI and send the first byte to UART 0x02
          //-------------------------------------------------------------------
          begin

            if (arready) begin // AXI Read Request Accepted
              reg_arburst <= `AXI_AxBURST_FIXED ;
              reg_arlen   <= 8'h0               ;
              reg_arlock  <= 1'b0               ;
              reg_arvalid <= 1'b0               ;
            end


            // AXI Interface Ready to Read
            reg_rready  <=  1'b1;

            if (tx_busy == 1'b0 && rvalid && rresp == `AXI_xRESP_OKAY)  begin // wait until transmitter is not busy
              reg_rready  <=  1'b0;
              // Send first byte to UART
              tx_datas_start <= 1'b1 ;
              tx_datas       <= rdata [0 +:UART_DATA_WIDTH];

              if (NC_TRSM_DATA <= 'd1) begin
                cnt <= 16'b0;
                if(rlast) begin
                  fsm_uart <= FSM_UART_IDLE;
                end else begin // Read Burst (TODO)
                  fsm_uart <= FSM_UART_RD_AXI;
                end
              end
              else begin // Keep transmitting
                // Save received data
                rdata_r <= rdata;
                cnt <= cnt + 16'b1;
                fsm_uart <= FSM_UART_WR_UART;
              end
            end
            else begin
              tx_datas_start  <= 1'b0;
              fsm_uart <= FSM_UART_RD_AXI;
            end
          end
          //-------------------------------------------------------------------
          FSM_UART_WR_UART : // Send the rest of received bytes from AXI to UART 0x03
          //-------------------------------------------------------------------
          begin
            // AXI Interface not Ready to Read
            reg_rready  <=  1'b0;

            if (tx_busy == 1'b0 )  begin // wait until UART transmitter is not busy
              
              // Transmit next byte
              tx_datas_start <= 1'b1 ;
              tx_datas       <= rdata_r [cnt*UART_DATA_WIDTH +:UART_DATA_WIDTH]; // Load Data Byte

              if (cnt < (NC_TRSM_DATA - 1)) begin
                cnt <= cnt + 16'b1;
                fsm_uart <= FSM_UART_WR_UART; // Keep transmitting
              end
              else begin // Full Data Transmitted
                cnt <= 16'h0;
                if(rlast) begin
                  fsm_uart <= FSM_UART_IDLE;
                end else begin // Read Burst (TODO)
                  fsm_uart <= FSM_UART_WR_UART;
                end
              end
            end
            else begin
              tx_datas_start  <= 1'b0;
              fsm_uart <= FSM_UART_WR_UART;
            end
          end
          /* Write Data on RAM not implemented yet
          //-------------------------------------------------------------------
          FSM_UART_WR_SPY : // Send Received Data from UART to AXI Not implemented for the moment
          //-------------------------------------------------------------------
          begin
            tx_datas_start <= 1'b0;
            if (rx_datas_rdy == 1'b1 && rx_datas_rdy_r == 1'b0) begin
              if (parity_error == 1'b0 && start_bit_error == 1'b0) begin
                data_out  <= rx_datas;
                data_rcvd <= 1'b1;
              end
            fsm_uart        <= FSM_UART_IDLE;
            tx_datas_start  <= 1'b1;
            tx_datas        <= {6'b0,start_bit_error,parity_error};
            end  
          end */
          //-------------------------------------------------------------------
          default : fsm_uart  <= FSM_UART_IDLE;
          //-------------------------------------------------------------------
          endcase
        end
      end
    end
  end
  
  // Detect synchronization signal
  always @(negedge arstn or posedge clk)
  begin
    if (~arstn) begin
      fsm_sync                <= FSM_SYNC_P0;
      sync_signal_detected    <= 1'b0;
      sync_signal_detected_r  <= 1'b0;
    end else begin
      if (en == 1'b0) begin
        fsm_sync                <= FSM_SYNC_P0;
        sync_signal_detected    <= 1'b0;
        sync_signal_detected_r  <= 1'b0;
      end else begin
        sync_signal_detected_r  <= sync_signal_detected;
        if (rx_datas_rdy == 1'b1 && rx_datas_rdy_r == 1'b0) begin // rx_datas_rdy posedge detection
          case (fsm_sync)
            //-------------------------------------------------------------------
            FSM_SYNC_P0 :
            //-------------------------------------------------------------------
            begin
              sync_signal_detected <= 1'b0;
              if (rx_datas == 8'hff) begin
                fsm_sync  <= FSM_SYNC_P1;
              end
            end
            //-------------------------------------------------------------------
            FSM_SYNC_P1 :
            //-------------------------------------------------------------------
            begin
              if (rx_datas == 8'hff) begin
                fsm_sync  <= FSM_SYNC_P1;
              end else if (rx_datas == 8'hfe) begin
                fsm_sync  <= FSM_SYNC_P2;
              end else begin
                fsm_sync  <= FSM_SYNC_P0;
              end
            end
            //-------------------------------------------------------------------
            FSM_SYNC_P2 :
            //-------------------------------------------------------------------
            begin
              if (rx_datas == 8'hff) begin
                fsm_sync  <= FSM_SYNC_P1;
              end else if (rx_datas == 8'hfd) begin
                fsm_sync  <= FSM_SYNC_P3;
              end else begin
                fsm_sync  <= FSM_SYNC_P0;
              end
            end
            //-------------------------------------------------------------------
            FSM_SYNC_P3 :
            //-------------------------------------------------------------------
            begin
              if (rx_datas == 8'hff) begin
                fsm_sync  <= FSM_SYNC_P1;
              end else begin
                fsm_sync  <= FSM_SYNC_P0;
                if (rx_datas == 8'hfc) begin
                  sync_signal_detected <= 1'b1;
                end
              end
            end
            //-------------------------------------------------------------------
            default : fsm_sync  <= FSM_SYNC_P0;
            //-------------------------------------------------------------------
          endcase
        end
      end
    end
  end

endmodule