//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: PIPE Monitor Checker [pmc]. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) us Counter
//                (3) PIPE State Logic
//                (4) Reported Error Logic
//                (5) Message Bus Logic
//                (6) Data Logic
//                (7) AXI RAM Backdoor Access
//                (6) RAMs
//                (8) UART to AXI interface
//
// Dependency : pipe_state_logic, reported_error_logic, msg_bus_logic, data_logic
//              xr5axi_dcram, uart_axi_top, axi_ram_backdoor
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_config_h.v"

module pipemc_top #(
    //-------------------------------------------------------------------------------
    parameter G_PIPE_INTF         = 'd1       ,
    parameter G_PCIE_NUM_LANES    = 'd16      ,
    parameter G_PIPE_WIDTH        = 'h88422       // Gen 4
    //-------------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------------
    // Indipendent Clock, Reset and Enable
    //-------------------------------------------------------------------------------
    input  wire                                                 pmc_refclk          ,   // Reference Clock - 125 MHz??
    input  wire                                                 pmc_en              ,   // Enable
    input  wire                                                 pmc_arstn           ,   // Asynchronous Reset#
    input  wire                                                 pmc_srstn           ,   // Synchronous Reset#

    //-------------------------------------------------------------------------------
    // TL Timer
    //-------------------------------------------------------------------------------
    input  wire [4                                        -1:0] tl_report_timer     ,   // | Bit 3-2: Reserved | Bit 1: 1ms pulse | Bit 0: 1 us pulse |
    
    //-------------------------------------------------------------------------------
    // PIPE Interface Signals
    //-------------------------------------------------------------------------------
    //  . Clock and Resets
    input  wire                                                 pl_pclk             , // USED
    input  wire                                                 pl_rstn             ,
    input  wire                                                 pl_rstnp            ,
    input  wire                                                 pl_srst             ,
    input  wire                                                 pl_npor             ,
    input  wire                                                 pl_spor             ,
    input  wire                                                 pl_rstn_srst_out    ,
    input  wire [4                                       -1:0]  pl_exit             ,

    //  . PIPE signals (common to all lanes)
    input  wire [4                                       -1:0]  pl_powerdown        , // USED
    input  wire [3                                       -1:0]  pl_rate             ,
    input  wire [5                                       -1:0]  pl_ltssm            , // USED    
    input  wire [2                                       -1:0]  pl_equ_phase        , // USED

    //  . PIPE signals (interface width/clock rate change)
    input  wire [3                                       -1:0]  pl_width            , // USED
    input  wire [5                                       -1:0]  pl_pclk_rate        ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_pclk_change_ok   ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_pclk_change_ack  ,
    input  wire [5                                       -1:0]  pl_pll_rate         ,
    input  wire                                                 pl_pll_ack          ,
    input  wire                                                 pl_esm_cal_request  , // NOT USED
    input  wire                                                 pl_esm_cal_complete , // NOT USED
    input  wire [                                        31:0]  pl_esm_rate         , // NOT USED
    
    //  . PIPE signals (per lane)
    input  wire [64*G_PCIE_NUM_LANES                     -1:0]  pl_sideband_out     ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_txdetectrx       , // USED
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8   -1:0]  pl_txdata           , // USED
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)     -1:0]  pl_txdatak          , // USED
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_txdatavalid      , // USED
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_txstartblock     ,
    input  wire [G_PCIE_NUM_LANES*2                      -1:0]  pl_txsyncheader     ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_txelecidle       ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_txcompliance     ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_rxstandby        ,
    input  wire [64*G_PCIE_NUM_LANES                     -1:0]  pl_sideband_in      ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_phystatus        , // USED
    input  wire [G_PCIE_NUM_LANES*3                      -1:0]  pl_rxstatus         , // USED
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8   -1:0]  pl_rxdata           , // USED
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)     -1:0]  pl_rxdatak          , // USED
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_rxdatavalid      , // USED
    input  wire [  (1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0]  pl_rxstartblock     ,
    input  wire [  (1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0]  pl_rxstartbhigh     ,
    input  wire [2*(1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0]  pl_rxsyncheader     ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_rxvalid          ,
    input  wire [G_PCIE_NUM_LANES                        -1:0]  pl_rxelecidle       ,
    
    //  . PHY-MAC interface (per lane)
    input  wire [G_PCIE_NUM_LANES*8                     -1:0]   pl_p2m_msgbus       , // USED
    input  wire [G_PCIE_NUM_LANES*8                     -1:0]   pl_m2p_msgbus       , // USED
    
    //  . WAKE# pin
    input  wire                                                 pl_wake_oen         , // NOT USED
    input  wire                                                 pl_wake_in          , // NOT USED

    //  . CLKREQ# management signals
    input  wire                                                 pl_clkreq_oen       , // NOT USED
    input  wire                                                 pl_clkrem_allow     , // NOT USED
    //-------------------------------------------------------------------------------
    // UART
    //-------------------------------------------------------------------------------
    (* mark_debug = "true" *) input  wire                                                 uart_rx             , // USED
    (* mark_debug = "true" *) output wire                                                 uart_tx               // USED
    //-------------------------------------------------------------------------------
    );

    //---------------------------------------------------------------------------
    // Local declarations
    //---------------------------------------------------------------------------
    genvar i;

    //-------------------------------------------------------------------------------
    //  Time
    reg [`G_COUNTER_US_WIDTH                          -1:0] global_time_us          ;

    //-------------------------------------------------------------------------------
    //  PIPE State Signals
    wire                                                    data_encoding           ;
    wire                                                    detect_txn_ltssm        ;
    wire [5                                           -1:0] prev_ltssm              ;

    //-------------------------------------------------------------------------------
    //  AXI Backdoor Access
    localparam  N_TOT_RAM = (`NRAM_W0 + 2 * `NRAM_W1 + 4 * `NRAM_W1 + 8 * `NRAM_W3) + G_PCIE_NUM_LANES; // It adds the msgbus RAMs as well

    //  . AXI4 Master Interface Read Address Channel
    wire [4                                           -1:0] axi4_mst1_arid          ;
    wire [64                                          -1:0] axi4_mst1_araddr        ;
    wire [4                                           -1:0] axi4_mst1_arregion      ;
    wire [8                                           -1:0] axi4_mst1_arlen         ;
    wire [3                                           -1:0] axi4_mst1_arsize        ;
    wire [2                                           -1:0] axi4_mst1_arburst       ;
    wire                                                    axi4_mst1_arlock        ;
    wire [4                                           -1:0] axi4_mst1_arcache       ;
    wire [3                                           -1:0] axi4_mst1_arprot        ;
    wire [4                                           -1:0] axi4_mst1_arqos         ;
    (* mark_debug = "true" *)wire                                                    axi4_mst1_arvalid       ;
    (* mark_debug = "true" *)wire                                                    axi4_mst1_arready       ;
    //  . AXI4 Master Interface Read Data Channel
    wire [4                                           -1:0] axi4_mst1_rid           ;
    wire [`MEM_DATAPATH_WIDTH                         -1:0] axi4_mst1_rdata         ;
    wire [2                                           -1:0] axi4_mst1_rresp         ;
    wire                                                    axi4_mst1_rlast         ;
    (* mark_debug = "true" *)wire                                                    axi4_mst1_rvalid        ;
    (* mark_debug = "true" *)wire                                                    axi4_mst1_rready        ;

    //-------------------------------------------------------------------------------
    //AXI Backdoor RAM Ports
    (* mark_debug = "true" *) wire [N_TOT_RAM                                   -1:0] shared_ram_rden         ;
    (* mark_debug = "true" *) wire [`MEM_ADDR_WIDTH_MAX                         -1:0] shared_ram_rdaddr       ;
    wire [N_TOT_RAM * `MEM_DATAPATH_WIDTH             -1:0] shared_ram_rddata       ;
    
    //-------------------------------------------------------------------------------
    //  RAM Signals
    //  . LTSSM RAM
    wire                                                    dcram_ltssm_wren      ;
    //wire [`MEM_DATAPATH_WIDTH/8               -1:0] dcram_ltssm_wrbe      ;
    wire [`MEM_ADDR_WIDTH_LTSSM_PHASE                 -1:0] dcram_ltssm_wraddr    ;
    wire [`MEM_DATAPATH_WIDTH                         -1:0] dcram_ltssm_wrdata    ;

    //  . Reported Errors RAM
    wire                                                    dcram_rerr_wren       ;
    // wire [`MEM_DATAPATH_WIDTH/8                      -1:0] dcram_rerr_wrbe       ;
    wire [`MEM_ADDR_WIDTH_RERR                        -1:0] dcram_rerr_wraddr     ;
    wire [`MEM_DATAPATH_WIDTH                         -1:0] dcram_rerr_wrdata     ;

    //  . Message Bus RAM
    wire [G_PCIE_NUM_LANES                            -1:0] dcram_msgbus_wren     ;
    // wire [G_PCIE_NUM_LANES * `MEM_DATAPATH_WIDTH/8 -1:0] dcram_msgbus_wrbe     ;
    wire [G_PCIE_NUM_LANES * `MEM_ADDR_WIDTH_MSGBUS   -1:0] dcram_msgbus_wraddr   ;
    wire [G_PCIE_NUM_LANES * `MEM_DATAPATH_WIDTH      -1:0] dcram_msgbus_wrdata   ;

    //  . Data Logic RAM | Event
    wire                                                    dcram_evntx_wren     ;
    // wire [`MEM_DATAPATH_WIDTH/8                 -1:0] dcram_evntx_wrbe     ;
    wire [`MEM_ADDR_WIDTH_DATA_EVNT                   -1:0] dcram_evntx_wraddr   ;
    wire [`MEM_DATAPATH_WIDTH                         -1:0] dcram_evntx_wrdata   ;

    

    //---------------------------------------------------------------------------
    // us Counter
    //---------------------------------------------------------------------------
    always @(posedge pmc_refclk or negedge pmc_arstn) begin
      if(!pmc_arstn) begin
        global_time_us <= {(`G_COUNTER_US_WIDTH){1'b0}};
      end
      else begin
        if (!pmc_srstn) begin
          global_time_us <= {(`G_COUNTER_US_WIDTH){1'b0}};
        end
        else if (tl_report_timer[0]) begin
          global_time_us <= global_time_us + 'h1;
        end
        else begin
          global_time_us <= global_time_us ;
        end
      end
    end


    //---------------------------------------------------------------------------
    // PIPE State Logic
    //---------------------------------------------------------------------------
    pipe_state_logic #(
      .PKT_WIDTH              (`PKT_WIDTH_LTSSM_PHASE       ),
      .MEM_ADDR_WIDTH         (`MEM_ADDR_WIDTH_LTSSM_PHASE  ),
      .MEM_DATA_WIDTH         (`MEM_DATAPATH_WIDTH          )
      //-------------------------------------------------------------------------
    ) pipe_state_logic_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (pmc_en                                       ),
      .arstn                  (pmc_arstn                                    ),
      .srstn                  (pmc_srstn                                    ),
      .pclk                   (pl_pclk                                      ),
      //  . Timer
      .global_time            (global_time_us[`TIME_WIDTH_LTSSM_PHASE -1:0] ),
      //  . PIPE State
      .pl_ltssm               (pl_ltssm                                     ),
      .pl_equ_phase           (pl_equ_phase                                 ),
      //  . PIPE Configuration
      .pl_rate                (pl_rate                                      ),
      .pl_width               (pl_width                                     ),
      .pl_pclk_rate           (pl_pclk_rate                                 ),
      //  . Output Signals
      .data_encoding          (data_encoding                                ),
      .detect_txn_ltssm       (detect_txn_ltssm                             ),
      .prev_ltssm             (prev_ltssm                                   ),
      //  . RAM Port
      .ram_wren               (dcram_ltssm_wren                             ),
      .ram_wrbe               (  ),
      .ram_wraddr             (dcram_ltssm_wraddr                           ),
      .ram_wrdata             (dcram_ltssm_wrdata                           )
      //-------------------------------------------------------------------------
    );



    //---------------------------------------------------------------------------
    // Reported Error Logic
    //---------------------------------------------------------------------------
    reported_error_logic #(
      //-------------------------------------------------------------------------
      //  . PIPE Parameters
      .G_PCIE_NUM_LANES (G_PCIE_NUM_LANES     ),
      .G_PIPE_WIDTH     (G_PIPE_WIDTH         ),
      //  . Module and Memory Parameters
      .PKT_WIDTH        (`PKT_WIDTH_RERR      ),
      .MEM_ADDR_WIDTH   (`MEM_ADDR_WIDTH_RERR ),
      .MEM_DATA_WIDTH   (`MEM_DATAPATH_WIDTH  )
      //-------------------------------------------------------------------------
    )reported_error_logic_inst (
      //-------------------------------------------------------------------------
      // Inputs
      //  . Clocks, EN and Resets
      .en                     (pmc_en                   ),
      .arstn                  (pmc_arstn                ),
      .srstn                  (pmc_srstn                ),
      .pclk                   (pl_pclk                  ),

      //  . Timer
      .global_time            (global_time_us[`TIME_WIDTH_RERR - G_PCIE_NUM_LANES -1:0] ),

      //  . PIPE RX Signals
      .pl_rxdata              (pl_rxdata                ),
      .pl_rxstatus            (pl_rxstatus              ),
      .pl_rxvalid             (pl_rxvalid               ),

      //-------------------------------------------------------------------------
      // Output
      //  . RAM Port
      .ram_wren               (dcram_rerr_wren          ),
      .ram_wrbe               (        ),
      .ram_wraddr             (dcram_rerr_wraddr        ),
      .ram_wrdata             (dcram_rerr_wrdata        )
      //-------------------------------------------------------------------------
    );



    //---------------------------------------------------------------------------
    // Message Bus Logic
    //---------------------------------------------------------------------------
    msg_bus_logic #(
      //-------------------------------------------------------------------------
      //  . PIPE Parameters
      .G_PCIE_NUM_LANES (G_PCIE_NUM_LANES       ),
      .G_PIPE_WIDTH     (G_PIPE_WIDTH           ),
      //  . Module and Memory Parameters
      .PKT_WIDTH        (`PKT_WIDTH_MSGBUS      ),
      .MEM_ADDR_WIDTH   (`MEM_ADDR_WIDTH_MSGBUS ),
      .MEM_DATA_WIDTH   (`MEM_DATAPATH_WIDTH    )
      //-------------------------------------------------------------------------
    ) msg_bus_logic_inst (
      //-------------------------------------------------------------------------
      // Inputs
      //  . Clocks, EN and Resets
      .en                       (pmc_en                                   ),
      .arstn                    (pmc_arstn                                ),
      .srstn                    (pmc_srstn                                ),
      .pclk                     (pl_pclk                                  ),

      //  . Timer
      .global_time              (global_time_us[`TIME_WIDTH_MSGBUS -1:0]  ),

      //  . PIPE RX Signals
      .pl_p2m_msgbus            (pl_p2m_msgbus                            ),
      .pl_m2p_msgbus            (pl_m2p_msgbus                            ),

      //-------------------------------------------------------------------------
      // Output
      //  . RAM Port
      .ram_wren                 (dcram_msgbus_wren                        ),
      .ram_wrbe                 (                      ),
      .ram_wraddr               (dcram_msgbus_wraddr                      ),
      .ram_wrdata               (dcram_msgbus_wrdata                      )
      //-------------------------------------------------------------------------
    );




    //---------------------------------------------------------------------------
    // Data Logic
    //---------------------------------------------------------------------------
    data_logic #(
      //-------------------------------------------------------------------------
      //  . PIPE Parameters
      .G_PCIE_NUM_LANES (G_PCIE_NUM_LANES           ),
      .G_PIPE_WIDTH     (G_PIPE_WIDTH               ),
      //  . Module and Memory Parameters
      .PKT_WIDTH        (`PKT_WIDTH_DATA_EVNT       ),
      .MEM_ADDR_WIDTH   (`MEM_ADDR_WIDTH_DATA_EVNT  ),
      .MEM_DATA_WIDTH   (`MEM_DATAPATH_WIDTH        )
      //-------------------------------------------------------------------------
    ) data_logic_inst (
      //-------------------------------------------------------------------------
      // Inputs
      //  . Clocks, EN and Resets
      .en                     (pmc_en                   ),
      .arstn                  (pmc_arstn                ),
      .srstn                  (pmc_srstn                ),
      .pclk                   (pl_pclk                  ),

      //  . Timer
      .global_time            (global_time_us[`TIME_WIDTH_DATA_EVNT -1:0] ),

      //-------------------------------------------------------------------------
      // PIPE Signals
      //  . Signals all lanes
      .pl_powerdown           (pl_powerdown             ),
      .pl_rate                (pl_rate                  ),
      .pl_ltssm               (pl_ltssm                 ),
      .pl_equ_phase           (pl_equ_phase             ),

      .pl_width               (pl_width                 ),

      //  . Signals per lane - TX
      .pl_txdetectrx          (pl_txdetectrx            ),
      .pl_txdata              (pl_txdata                ),
      .pl_txdatak             (pl_txdatak               ),
      .pl_txdatavalid         (pl_txdatavalid           ),

      //  . Signals per lane - RX
      .pl_phystatus           (pl_phystatus             ),
      .pl_rxstatus            (pl_rxstatus              ),
      .pl_rxdata              (pl_rxdata                ),
      .pl_rxdatak             (pl_rxdatak               ),
      .pl_rxdatavalid         (pl_rxdatavalid           ),
      .pl_rxvalid             (pl_rxvalid               ),

      //-------------------------------------------------------------------------
      // PIPE State Logic Signals
      .data_encoding          (data_encoding            ),
      .detect_txn_ltssm       (detect_txn_ltssm         ),
      .prev_ltssm             (prev_ltssm               ),

      //-------------------------------------------------------------------------
      // Output
      //  . RAM Port
      .ram_wren               (dcram_evntx_wren         ),
      .ram_wrbe               (         ),
      .ram_wraddr             (dcram_evntx_wraddr       ),
      .ram_wrdata             (dcram_evntx_wrdata       )
      //-------------------------------------------------------------------------
    );

    //---------------------------------------------------------------------------
    // PIPE Monitor Checker Registers
    //---------------------------------------------------------------------------
    pipemc_reg #(
      //-------------------------------------------------------------------------
      // XR5AXI Layer Constants
      .ADDR_WIDTH              (`MEM_ADDR_WIDTH_REG                 ),
      .DATA_WIDTH              (`MEM_DATAPATH_WIDTH ),
      //-------------------------------------------------------------------------
      // PIPE Configuration
      .G_PIPE_INTF             (G_PIPE_INTF         ),
      .G_PCIE_NUM_LANES        (G_PCIE_NUM_LANES    ),
      .G_PIPE_WIDTH            (G_PIPE_WIDTH        )
      //-------------------------------------------------------------------------
    ) pipemc_reg_inst (
      //-------------------------------------------------------------------------
      // Registers Interface
      //  . Read  Port
      .reg_rdclk              (pmc_refclk                                       ),
      .reg_rden               (shared_ram_rden[0]                               ),
      .reg_rdaddr             (shared_ram_rdaddr [0     +: 2                  ] ),
      .reg_rddata             (shared_ram_rddata [0     +: `MEM_DATAPATH_WIDTH] ),
      //-------------------------------------------------------------------------
      // RAM Interface
      .reg_wrclk              (pl_pclk                                          ),
      //  . LTSSM RAM
      .dcram_ltssm_wren       (dcram_ltssm_wren                                 ),
      .dcram_ltssm_wraddr     (dcram_ltssm_wraddr                               ),
      //  . Reported Errors RAM
      .dcram_rerr_wren        (dcram_rerr_wren                                  ),
      .dcram_rerr_wraddr      (dcram_rerr_wraddr                                ),
      //  . Message Bus RAM
      .dcram_msgbus_wren      (dcram_msgbus_wren                                ),
      .dcram_msgbus_wraddr    (dcram_msgbus_wraddr                              ),
      //  . Data Logic RAM RAM
      .dcram_evntx_wren       (dcram_evntx_wren                                 ),
      .dcram_evntx_wraddr     (dcram_evntx_wraddr                               )
      //-------------------------------------------------------------------------
    );


    //---------------------------------------------------------------------------
    // AXI Backdoor Interface
    //---------------------------------------------------------------------------
    axi_ram_backdoor #(
      //-------------------------------------------------------------------------
      .AXI_ADDR_WIDTH                   (`AXI_ADDR_WIDTH        ),
      .AXI_DATA_WIDTH                   (`MEM_DATAPATH_WIDTH    ),
      .RAM_1_N                          (`NRAM_W0 + G_PCIE_NUM_LANES ), // It considers the msgbus RAMs as well
      .RAM_2_N                          (`NRAM_W1               ),
      .RAM_4_N                          (`NRAM_W2               ),
      .RAM_8_N                          (`NRAM_W3               ),
      .MEM_ADDR_WIDTH_MAX               (`MEM_ADDR_WIDTH_MAX    )
      //-------------------------------------------------------------------------
    ) axi_ram_backdoor_inst (
      //-------------------------------------------------------------------------
      // AXI4 Slave Interface
      //  . Clock and Resets
      .arstn                            (pmc_arstn              ),
      .srstn                            (pmc_srstn              ),
      .clk                              (pmc_refclk             ),
      //  . Read Address Channel     
      .arid                             (axi4_mst1_arid         ),
      .araddr                           (axi4_mst1_araddr       ),
      .arregion                         (axi4_mst1_arregion     ),
      .arlen                            (axi4_mst1_arlen        ),
      .arsize                           (axi4_mst1_arsize       ),
      .arburst                          (axi4_mst1_arburst      ),
      .arlock                           (axi4_mst1_arlock       ),
      .arcache                          (axi4_mst1_arcache      ),
      .arprot                           (axi4_mst1_arprot       ),
      .arqos                            (axi4_mst1_arqos        ),
      .arvalid                          (axi4_mst1_arvalid      ),
      .arready                          (axi4_mst1_arready      ),
      //  . Read Data Channel
      .rid                              (axi4_mst1_rid          ),
      .rdata                            (axi4_mst1_rdata        ),
      .rresp                            (axi4_mst1_rresp        ),
      .rlast                            (axi4_mst1_rlast        ),
      .rvalid                           (axi4_mst1_rvalid       ),
      .rready                           (axi4_mst1_rready       ),
      //-------------------------------------------------------------------------
      //RAM Ports
      .shared_ram_rden                  (shared_ram_rden        ),
      .shared_ram_rdaddr                (shared_ram_rdaddr      ),
      .shared_ram_rddata                (shared_ram_rddata      )
      //-------------------------------------------------------------------------
    );


    //---------------------------------------------------------------------------
    // RAMs
    //---------------------------------------------------------------------------
    //  . LTSSM RAM - Connected to PIPE State Logic
    xr5axi_dcram #(
      //-------------------------------------------------------------------------
      // XR5AXI Layer Constants
      .DATA_WIDTH              (`MEM_DATAPATH_WIDTH ),
      .ADDR_WIDTH              (`MEM_ADDR_WIDTH_LTSSM_PHASE )
      //-------------------------------------------------------------------------
      ) xr5axi_dcram_ltssm_inst (
      //-------------------------------------------------------------------------
      // Buffer Interface
      //  . Write Port
      .dcram_wrclk             (pl_pclk                 ),
      .dcram_wren              (dcram_ltssm_wren        ),
      .dcram_wraddr            (dcram_ltssm_wraddr      ),
      .dcram_wrdata            (dcram_ltssm_wrdata      ),
      //  . Read  Port
      .dcram_rdclk             (pmc_refclk              ),
      .dcram_rden              (shared_ram_rden[1]      ),
      .dcram_rdaddr            (shared_ram_rdaddr [0 +: `MEM_ADDR_WIDTH_LTSSM_PHASE]),
      .dcram_rddata            (shared_ram_rddata [1*`MEM_DATAPATH_WIDTH +: `MEM_DATAPATH_WIDTH])
      //-------------------------------------------------------------------------
    );


    //  . Reported Error RAM
    xr5axi_dcram #(
      //-------------------------------------------------------------------------
      // XR5AXI Layer Constants
      .DATA_WIDTH              (`MEM_DATAPATH_WIDTH  ),
      .ADDR_WIDTH              (`MEM_ADDR_WIDTH_RERR  )
      //-------------------------------------------------------------------------
    ) xr5axi_dcram_rerr_inst (
      //-------------------------------------------------------------------------
      // Buffer Interface
      //  . Write Port
      .dcram_wrclk    (pl_pclk                 ),
      .dcram_wren     (dcram_rerr_wren         ),
      .dcram_wraddr   (dcram_rerr_wraddr       ),
      .dcram_wrdata   (dcram_rerr_wrdata       ),
      //  . Read  Port
      .dcram_rdclk    (pmc_refclk              ), 
      .dcram_rden     (shared_ram_rden[2]      ),
      .dcram_rdaddr   (shared_ram_rdaddr [0 +: `MEM_ADDR_WIDTH_RERR]),
      .dcram_rddata   (shared_ram_rddata [2*`MEM_DATAPATH_WIDTH   +: `MEM_DATAPATH_WIDTH])
      //-------------------------------------------------------------------------
    );

    //  . Data Logic | Logged Event
    xr5axi_dcram #(
      //-------------------------------------------------------------------------
      // XR5AXI Layer Constants
      .DATA_WIDTH              (`MEM_DATAPATH_WIDTH  ),
      .ADDR_WIDTH              (`MEM_ADDR_WIDTH_DATA_EVNT  )
      //-------------------------------------------------------------------------
    ) xr5axi_dcram_evnt_inst (
      //-------------------------------------------------------------------------
      // Buffer Interface
      //  . Write Port
      .dcram_wrclk    (pl_pclk                 ),
      .dcram_wren     (dcram_evntx_wren        ),
      .dcram_wraddr   (dcram_evntx_wraddr      ),
      .dcram_wrdata   (dcram_evntx_wrdata      ),
      //  . Read  Port
      .dcram_rdclk    (pmc_refclk              ),
      .dcram_rden     (shared_ram_rden[3]      ),
      .dcram_rdaddr   (shared_ram_rdaddr [0 +: `MEM_ADDR_WIDTH_DATA_EVNT]),
      .dcram_rddata   (shared_ram_rddata [3*`MEM_DATAPATH_WIDTH   +: `MEM_DATAPATH_WIDTH])
      //-------------------------------------------------------------------------
    );


    //  . Message Bus RAM
    generate
      for (i = 0; i < G_PCIE_NUM_LANES; i = i + 1) begin
        xr5axi_dcram #(
          //-------------------------------------------------------------------------
          // XR5AXI Layer Constants
          .DATA_WIDTH              (`MEM_DATAPATH_WIDTH  ),
          .ADDR_WIDTH              (`MEM_ADDR_WIDTH_MSGBUS  )
          //-------------------------------------------------------------------------
        ) xr5axi_dcram_msgbus_inst (
          //-------------------------------------------------------------------------
          // Buffer Interface
          //  . Write Port
          .dcram_wrclk          (pl_pclk                ),
          .dcram_wren           (dcram_msgbus_wren   [i]),
          .dcram_wraddr         (dcram_msgbus_wraddr [i*`MEM_ADDR_WIDTH_MSGBUS       +:`MEM_ADDR_WIDTH_MSGBUS  ]),
          .dcram_wrdata         (dcram_msgbus_wrdata [i*`MEM_DATAPATH_WIDTH       +:`MEM_DATAPATH_WIDTH  ]),
          //  . Read  Port
          .dcram_rdclk          (pmc_refclk             ),
          .dcram_rden           (shared_ram_rden   [4+i]),
          .dcram_rdaddr         (shared_ram_rdaddr [0 +: `MEM_ADDR_WIDTH_MSGBUS]),
          .dcram_rddata         (shared_ram_rddata [(4+i)*`MEM_DATAPATH_WIDTH   +: `MEM_DATAPATH_WIDTH])
          //-------------------------------------------------------------------------
        );
      end
    endgenerate

    



    //---------------------------------------------------------------------------
    // UART Interface
    //---------------------------------------------------------------------------
    uart_axi_top  #(
      //-------------------------------------------------------------------------
      .C_CLKS_PER_BIT           (`C_CLKS_PER_BIT        ),
      .G_ADDR_WIDTH             (`AXI_ADDR_WIDTH        ),
      .G_DATA_WIDTH             (`MEM_DATAPATH_WIDTH    )
      //-------------------------------------------------------------------------
    ) uart_axi_top_inst (
      //-------------------------------------------------------------------------
      // Resets, Enable and Clock
      .en                       (pmc_en                 ),
      .arstn                    (pmc_arstn              ),
      .clk                      (pmc_refclk             ),   
      //-------------------------------------------------------------------------------
      // UART Serial Port
      .uart_rx                  (uart_rx                ),
      .uart_tx                  (uart_tx                ),
      //-------------------------------------------------------------------------------
      // AXI4 Slave Interface
      //  . Read Address Channel   
      .arid                     (axi4_mst1_arid         ),
      .araddr                   (axi4_mst1_araddr       ),
      .arregion                 (axi4_mst1_arregion     ),
      .arlen                    (axi4_mst1_arlen        ),
      .arsize                   (axi4_mst1_arsize       ),
      .arburst                  (axi4_mst1_arburst      ),
      .arlock                   (axi4_mst1_arlock       ),
      .arcache                  (axi4_mst1_arcache      ),
      .arprot                   (axi4_mst1_arprot       ),
      .arqos                    (axi4_mst1_arqos        ),
      .arvalid                  (axi4_mst1_arvalid      ),
      .arready                  (axi4_mst1_arready      ),
      //  . Read Data Channel
      .rid                      (axi4_mst1_rid          ),
      .rdata                    (axi4_mst1_rdata        ),
      .rresp                    (axi4_mst1_rresp        ),
      .rlast                    (axi4_mst1_rlast        ),
      .rvalid                   (axi4_mst1_rvalid       ),
      .rready                   (axi4_mst1_rready       )
      //-------------------------------------------------------------------------
    );



endmodule