//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : XpressRich5-AXI/AXI_BFM
//           (employed in PIPE Monitor Checker | UNCHANGED)
// Author  : caires (based on the module AXI RAM written by rtuszewski)
//-----------------------------------------------------------------------------
// Description: Write Memory Controller. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Output assignations
//
// Dependency : none
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

module write_memory_controller #(
    parameter PKT_WIDTH         = 'd16    ,     // Packet Width [bits]
    parameter MEM_ADDR_WIDTH    = 'd8     ,
    parameter MEM_DATA_WIDTH    = 'd32
  )(
    //-------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                         en                    ,
    input  wire                                         arstn                 , //  Asynchronous Reset
    input  wire                                         srstn                 , //  Synchronous Reset
    input  wire                                         pclk                  ,

    //-------------------------------------------------------------------------
    //  Write enable and data packet to send
    input  wire                                         trig_write            ,
    input  wire [PKT_WIDTH                        -1:0] packet_in             ,

    //-------------------------------------------------------------------------
    //  Write Port
    output wire                                         scram_wren            ,
    output wire [MEM_DATA_WIDTH/8                 -1:0] scram_wrbe            ,
    output wire [MEM_ADDR_WIDTH                   -1:0] scram_wraddr          ,
    output wire [MEM_DATA_WIDTH                   -1:0] scram_wrdata                          

    );

    
    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    integer i ;

    //  . Write RAM Signals
    reg                                                 reg_wren            ;
    reg [MEM_DATA_WIDTH/8                         -1:0] reg_wrbe            ;
    reg [MEM_ADDR_WIDTH                           -1:0] reg_wraddr          ;
    reg [MEM_DATA_WIDTH                           -1:0] reg_wrdata          ;
    reg [MEM_ADDR_WIDTH * MEM_DATA_WIDTH          -1:0] reg_wraddr_bytes    ;


    //-------------------------------------------------------------------------
    // Output Assignations 
    //-------------------------------------------------------------------------
    assign scram_wren           = reg_wren          ;
    assign scram_wrbe           = reg_wrbe          ;
    assign scram_wraddr         = reg_wraddr        ;
    assign scram_wrdata         = reg_wrdata        ;

    
    //-------------------------------------------------------------------------
    // Controller Logic 
    //-------------------------------------------------------------------------
    always @(negedge arstn or posedge pclk)
    begin
      if (arstn == 1'b0) begin
        reg_wren   <= 1'b0 ;
        reg_wrbe   <= {(MEM_DATA_WIDTH/8       ) {1'b0}}; 
        reg_wraddr <= {(MEM_ADDR_WIDTH         ) {1'b0}};
        reg_wrdata <= {(MEM_DATA_WIDTH ) {1'b0}}; 
        reg_wraddr_bytes <= {(MEM_ADDR_WIDTH * MEM_DATA_WIDTH) {1'b0}};
      end 
      else begin
        if (srstn == 1'b0) begin
          reg_wren   <= 1'b0 ;
          reg_wrbe   <= {(MEM_DATA_WIDTH/8       ) {1'b0}}; 
          reg_wraddr <= {(MEM_ADDR_WIDTH         ) {1'b0}}; 
          reg_wrdata <= {(MEM_DATA_WIDTH ) {1'b0}}; 
          reg_wraddr_bytes <= {(MEM_ADDR_WIDTH * MEM_DATA_WIDTH) {1'b0}};
        end 
        else begin
          
          if( trig_write && en ) begin
            reg_wraddr_bytes <= reg_wraddr_bytes + PKT_WIDTH ; // Update Byte Address
          end

          if( trig_write && en)
          begin
            reg_wren   <= 1'b1;

            for ( i = 0; i < (MEM_DATA_WIDTH/8) ; i = i+1) begin
              reg_wrbe[i]   <=  (i >= ((reg_wraddr_bytes % MEM_DATA_WIDTH)/8)) ? 1'b1 : 1'b0 ; // Select Correct Bytes
              // $display ("[display] time=%0t  index=0x%0h  i=0x%0h", $time, ((reg_wraddr_bytes % MEM_DATA_WIDTH)/8), i);
            end
            
            reg_wraddr <= (reg_wraddr_bytes / MEM_DATA_WIDTH); // Obtain Memory Address from Byte Address

            if(reg_wraddr_bytes % MEM_DATA_WIDTH == 'd0) begin
              reg_wrdata <=  {{(MEM_DATA_WIDTH - PKT_WIDTH){1'b0}}, packet_in};
            end
            else begin
              reg_wrdata[(reg_wraddr_bytes % MEM_DATA_WIDTH) +: PKT_WIDTH] <= packet_in ; // Add Packet
            end
  
          end  
          else reg_wren <= 1'b0;

        end
      end
    end

endmodule