//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : XpressRich5-AXI/AXI_BFM
//           (employed in PIPE Monitor Checker | MODIFIED by nrigotti)
// Author  : caires (based on the module AXI RAM written by rtuszewski)
//-----------------------------------------------------------------------------
// Description: The AXI Backdoor Access implements an AXI interface that gives access
// to N AXI RAMs translanting AXI <-> RAM ports. Good luck :)
//
//              This module is composed of:
//                (1) Local Declarations
//                (2) write_mgt   : AXI4 Slave Write Request Management
//                (3) read_mgt    : AXI4 Slave Read  Request Management
//
// Dependency : -
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

module axi_slv_backdoor
 #(
    //-------------------------------------------------------------------------
    // Constants
    parameter   G_ADDR_WIDTH            =    64,
    parameter   G_DATA_WIDTH            =   512,
    parameter   G_AXI_BFM_RAM_PORTS_N   =    28,
    parameter   G_AXI_BFM_N             =     1
    //-------------------------------------------------------------------------
  ) (
    //-------------------------------------------------------------------------
    // AXI4 Slave Interface
    //  . Clock and Resets
    input  wire                                         aclk                  ,
    input  wire                                         aresetn               ,
    input  wire                                         asrst                 ,
    //  . Read Address Channel                                         
    input  wire [4                                -1:0] arid                  ,
    input  wire [64                               -1:0] araddr                ,
    input  wire [4                                -1:0] arregion              , // NOT USED
    input  wire [8                                -1:0] arlen                 , // 8'h0
    input  wire [3                                -1:0] arsize                , // NOT USED | Byte for transfer
    input  wire [2                                -1:0] arburst               , // x01
    input  wire                                         arlock                , // Locked Transaction
    input  wire [4                                -1:0] arcache               , // NOT USED
    input  wire [3                                -1:0] arprot                , // NOT USED
    input  wire [4                                -1:0] arqos                 , // NOT USED
    input  wire                                         arvalid               ,
    output wire                                         arready               ,
    //  . Read Data Channel
    output wire [4                                -1:0] rid                   ,
    output wire [G_DATA_WIDTH                     -1:0] rdata                 ,
    output wire [2                                -1:0] rresp                 ,
    output wire                                         rlast                 ,
    output wire                                         rvalid                ,
    input  wire                                         rready                ,
    //-------------------------------------------------------------------------
    // RAM Ports
    output reg  [G_AXI_BFM_RAM_PORTS_N*1             *G_AXI_BFM_N    -1:0] shared_ram_wren       ,
    output wire [                      G_DATA_WIDTH/8                -1:0] shared_ram_wrbe       ,
    output wire [                      G_ADDR_WIDTH                  -1:0] shared_ram_wraddr     ,
    output wire [                      G_DATA_WIDTH                  -1:0] shared_ram_wrdata     ,
    output reg  [G_AXI_BFM_RAM_PORTS_N*1             *G_AXI_BFM_N    -1:0] shared_ram_rden       ,
    output wire [                      G_ADDR_WIDTH                  -1:0] shared_ram_rdaddr     ,
    input  wire [G_AXI_BFM_RAM_PORTS_N*G_DATA_WIDTH  *G_AXI_BFM_N    -1:0] shared_ram_rddata     ,
     
    //-------------------------------------------------------------------------
    // Access priotity Ports
    input  wire [G_AXI_BFM_RAM_PORTS_N*1             *G_AXI_BFM_N    -1:0] shared_r_access_resp  , // Useless
    output reg  [G_AXI_BFM_RAM_PORTS_N*1             *G_AXI_BFM_N    -1:0] shared_r_access_req     // Useless

  );

  //----------------------------------------------------------------------------
  // Local Declaraitons 
  //---------------------------------------------------------------------------- 

  integer i; 
  
  // Backdoor access addresses 
  localparam G_DATA_WIDTH_BYTES = G_DATA_WIDTH/8 ;
  localparam G_AXI_REG_ADDR     = 'h000000000/G_DATA_WIDTH_BYTES ;
  localparam G_AXI_REG_MAX_ADDR = 'h000000100/G_DATA_WIDTH_BYTES ;  

  localparam G_AXI_MMON_ADDR    = 'h000001000/G_DATA_WIDTH_BYTES ;
  localparam G_AXI_SMON_ADDR    = 'h00000d000/G_DATA_WIDTH_BYTES ;
  localparam G_AXI_PMON_ADDR    = 'h000019000/G_DATA_WIDTH_BYTES ;
  localparam G_AXI_MAX_ADDR     = 'h000025000/G_DATA_WIDTH_BYTES ;  

  localparam G_RR_MEM_ADDR  = 'h1000/G_DATA_WIDTH_BYTES ;
  localparam G_WD_MEM_ADDR  = 'h2000/G_DATA_WIDTH_BYTES ;
  localparam G_RD_MEM_ADDR  = 'h6000/G_DATA_WIDTH_BYTES ;
  localparam G_WDI_MEM_ADDR = 'ha000/G_DATA_WIDTH_BYTES ;
  localparam G_RDI_MEM_ADDR = 'ha800/G_DATA_WIDTH_BYTES ;
  localparam G_WC_MEM_ADDR  = 'hb000/G_DATA_WIDTH_BYTES ;
  localparam G_ER_MEM_ADDR  = 'hb400/G_DATA_WIDTH_BYTES ;
  localparam G_MAX_MEM_ADDR = 'hb800/G_DATA_WIDTH_BYTES ;  
  
  localparam G_UNB_ADDR         = 'h000030000/G_DATA_WIDTH_BYTES ;
  localparam G_SLV_MEM_ADDR     = 'h2000/G_DATA_WIDTH_BYTES ;
  localparam G_UNB_REG_ADDR     = 'h6000/G_DATA_WIDTH_BYTES ;
  localparam G_UNB_REG_MAX_ADDR = 'h60c0/G_DATA_WIDTH_BYTES ;

  localparam G_UNB_REG_PARAM_ADDR = 'h6000/G_DATA_WIDTH_BYTES ;
  localparam G_UNB_REG_WREQ_ADDR  = 'h6040/G_DATA_WIDTH_BYTES ;
  localparam G_UNB_REG_RREQ_ADDR  = 'h6060/G_DATA_WIDTH_BYTES ;
  localparam G_UNB_REG_CONFIG_ADDR= 'h6080/G_DATA_WIDTH_BYTES ;

  localparam G_AXI_BFM_TOTAL_SIZE = 'h40000/G_DATA_WIDTH_BYTES ;   

  // Constants Declaration
  localparam ADDR_LSB_BIT                  = (G_DATA_WIDTH == 512) ? 6 : 
                                             (G_DATA_WIDTH == 256) ? 5 :
                                             (G_DATA_WIDTH == 128) ? 4 :
                                             (G_DATA_WIDTH ==  64) ? 3 :
                                             (G_DATA_WIDTH ==  32) ? 2 : 0 ;
  //FSM constants  
  localparam RD_ADDR_STATE                 = 3'b000 ;
  localparam RD_XBFM_STATE                 = 3'b001 ;
  localparam RD_PREP_STATE                 = 3'b010 ;
  localparam RD_ACC_STATE                  = 3'b011 ;
  localparam RD_PIPE_DATA_STATE            = 3'b100 ;
  localparam RD_TREAT_DATA_STATE           = 3'b101 ;
  localparam RD_RESP_STATE                 = 3'b110 ;

  // AXI4 Slave Read  Request Management
  reg  [3                                -1:0] rd_sm                ;
  reg                                          arready_i            ;
  reg  [G_ADDR_WIDTH                     -1:0] araddr_i             ;
  reg  [8                                -1:0] arlen_i              ;
  reg                                          arlock_i             ;
  reg                                          rvalid_i             ;
  reg  [4                                -1:0] rid_i                ;
  reg                                          rlast_i              ;

  // These codes defines which RAM space of which ABFM will be accessed.
  //   Its size depends on the number of ABFM being controlled. Considering 
  //   28 ports (16<N_PORTS<32), 5 bits are needed to represent 0-31 spaces. 
  //   Each bit added gives the possibility to control 32 more spaces (or another ABFM). 
  reg  [4+G_AXI_BFM_N                   -1:0] r_dest_code           ;   

  reg [G_AXI_BFM_RAM_PORTS_N*G_DATA_WIDTH*G_AXI_BFM_N    -1:0] shared_rddata_reg ;
  reg [G_DATA_WIDTH                                      -1:0] rddata_reg        ;
  reg [G_DATA_WIDTH                                      -1:0] axi_rddata        ;
  reg                                                          wait_rready       ;


  //---------------------------------------------------------------------------
  // AXI4 Slave Read Request Management
  //---------------------------------------------------------------------------
  always @(negedge aresetn or posedge aclk)
  begin: read_mgt
    if (aresetn == 1'b0) begin
        rd_sm     <= RD_ADDR_STATE ;
        arready_i <= 1'b0 ;
        araddr_i  <= {(G_ADDR_WIDTH){1'b0}};
        arlen_i   <= 8'b0 ;
        arlock_i  <= 1'b0 ;
        rvalid_i  <= 1'b0 ;
        rid_i     <= 4'b0 ;
        rlast_i   <= 1'b0 ;

        r_dest_code <=  {(4+G_AXI_BFM_N){1'b1}} ; // Because of 28 RAM ports (~5 bits)

        shared_r_access_req <= {(G_AXI_BFM_RAM_PORTS_N*1*G_AXI_BFM_N ){1'b0}};
        shared_ram_rden     <= {(G_AXI_BFM_RAM_PORTS_N*1*G_AXI_BFM_N ){1'b0}};
        shared_rddata_reg   <= {(G_AXI_BFM_RAM_PORTS_N*G_DATA_WIDTH*G_AXI_BFM_N ){1'b0}};

        rddata_reg <= {(G_DATA_WIDTH) {1'b0}} ;
        axi_rddata     <= {(G_DATA_WIDTH) {1'b0}} ;
        wait_rready    <= 1'b0 ;
    end else begin
      if (asrst == 1'b1) begin
        rd_sm     <= RD_ADDR_STATE ;
        arready_i <= 1'b0 ;
        araddr_i  <= {(G_ADDR_WIDTH){1'b0}};
        arlen_i   <= 8'b0 ;
        arlock_i  <= 1'b0 ;
        rvalid_i  <= 1'b0 ;
        rid_i     <= 4'b0 ;
        rlast_i   <= 1'b0 ;

        r_dest_code <=  {(4+G_AXI_BFM_N){1'b1}} ;

        shared_r_access_req <= {(G_AXI_BFM_RAM_PORTS_N*1*G_AXI_BFM_N ){1'b0}};
        shared_ram_rden <= {(G_AXI_BFM_RAM_PORTS_N*1*G_AXI_BFM_N ){1'b0}};
        rddata_reg <= {(G_DATA_WIDTH) {1'b0}} ;
        axi_rddata     <= {(G_DATA_WIDTH) {1'b0}} ;
        wait_rready    <= 1'b0 ;
      end else begin
        case (rd_sm)
          //-------------------------------------------------------------------
          RD_ADDR_STATE :
            begin
              axi_rddata <=  {(G_DATA_WIDTH) {1'b0}} ;
              if (arvalid && arready_i) begin
                // Read  Address is received: saving AXI request parameters and 
                // then sending the read data phase(s)
                arready_i <= 1'b0 ;
                araddr_i  <= araddr[ADDR_LSB_BIT +: (G_ADDR_WIDTH-ADDR_LSB_BIT)]; // Alignement datapath 
                arlen_i   <= arlen ;
                arlock_i  <= arlock ;
            
                rid_i     <= arid ;
                if (arlen == 0)
                  rlast_i   <= 1'b1 ;
                else
                  rlast_i   <= 1'b0 ;
  
                rd_sm <= RD_XBFM_STATE ;
              end else begin
                // Wait for valid read address to be received
                arready_i <= 1'b1 ;
                rvalid_i  <= 1'b0 ;
              end
            end
          //-------------------------------------------------------------------
          // Discovering the BFM ID target of the RD request
          RD_XBFM_STATE  :
          begin
            r_dest_code <= (araddr_i/G_AXI_BFM_TOTAL_SIZE)*G_AXI_BFM_RAM_PORTS_N; // calculating base dest_code for BFM target
            araddr_i <= araddr_i - ((araddr_i/G_AXI_BFM_TOTAL_SIZE)*G_AXI_BFM_TOTAL_SIZE); // removing BFM offset from addr

            rd_sm <= RD_PREP_STATE ;
          end
          //-------------------------------------------------------------------
          RD_PREP_STATE :
          // Discovering the RAM space target of the RD request in the pre 
          // determined BFM and asking access for it
          begin

            rd_sm        <= RD_ACC_STATE ;

          end
          //-------------------------------------------------------------------
          RD_ACC_STATE :
          // Waiting for access to be given to the pre determined RAM space 
          // target of the RD request
          begin
            if (shared_r_access_resp[r_dest_code +:1] == 1'b1) begin 
              rd_sm     <= RD_PIPE_DATA_STATE ;
              
              araddr_i  <= araddr_i + 2'b01 ;
            end
          end
          //-------------------------------------------------------------------
          RD_PIPE_DATA_STATE :
          // Pipelining received data from shared RD data port
          begin
            rd_sm     <= RD_TREAT_DATA_STATE ;
              
            shared_rddata_reg <= shared_ram_rddata; 
            araddr_i  <= araddr_i + 2'b01 ;
          end
          //-------------------------------------------------------------------
          RD_TREAT_DATA_STATE :
          // Determining specific Data bits in the shared data register pipelined 
          begin
            rvalid_i  <= 1'b1 ;
            rd_sm     <= RD_RESP_STATE ;

            shared_rddata_reg <= shared_ram_rddata; 
            axi_rddata <= shared_rddata_reg[r_dest_code*G_DATA_WIDTH +:G_DATA_WIDTH]; 
            araddr_i  <= araddr_i + 2'b01 ;
            
          end
          //-------------------------------------------------------------------
          default : // RD_RESP_STATE
          // Responding AXI RD REQ
          begin
            // Last Read Response is acknowledged: AXI Transaction is now complete
            if (rready && rlast_i) begin
              rd_sm     <= RD_ADDR_STATE ;
              r_dest_code <=  {(4+G_AXI_BFM_N){1'b1}};
              arready_i <= 1'b1 ;
              rvalid_i  <= 1'b0 ;

              shared_r_access_req [r_dest_code +:1] <= 1'b0; //end of WR operation, access is no long necessary
              shared_ram_rden     [r_dest_code +:1] <= 1'b0;
            end
            // Read parameters are updated after each valid read data phase
            if (rready) begin
              araddr_i  <= araddr_i + 2'b01 ;

              arlen_i   <= arlen_i  - 1'b1 ;
              if (arlen_i == 1) rlast_i   <= 1'b1 ;
              else rlast_i   <= 1'b0 ;
              
              rddata_reg <= {(G_DATA_WIDTH) {1'b0}} ;

              wait_rready    <= 1'b0 ;
            end
            else begin
              rddata_reg <= rdata ;
              wait_rready    <= 1'b1 ;
            end

            if (wait_rready == 1'b0 ) begin 
              axi_rddata <= shared_rddata_reg[r_dest_code*G_DATA_WIDTH +:G_DATA_WIDTH];  
              shared_rddata_reg <= shared_ram_rddata;
            end
          end
          //-------------------------------------------------------------------
        endcase
      end
    end
  end

  assign arready = arready_i ;
  assign rvalid  = rvalid_i  ;
  assign rid     = rid_i     ;
  assign rdata   = (wait_rready) ? rddata_reg : axi_rddata;
  assign rresp   = {1'b0 , arlock_i}; // EXOKAY or OKAY write response
  assign rlast   = rlast_i   ;

  // Read  Port
  assign shared_ram_rdaddr = araddr_i ;


endmodule

`resetall