//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Reported Error Logic
//              Module to detect PIPE error reported by the PHY or PCIe Controller.
//              Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) EDB Symbol detection
//                (3) RxStatus Reported Error detection
//                (4) Package creation
//                (5) Writing Memory Controller
//
// Dependency : write_memory_controller
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module reported_error_logic #(
    //-------------------------------------------------------------------------
    //  PIPE Parameters
    parameter G_PCIE_NUM_LANES  = 'd2       ,
    parameter G_PIPE_WIDTH      = 'h044422  ,   // Gen ??
    parameter SH_BITS      = 'd16      , 
    //-------------------------------------------------------------------------
    //  Module and Memory Parameters
    parameter PKT_WIDTH         = 'd16      ,     // Packet Width   [bits]
    parameter MEM_ADDR_WIDTH    = 'd8       ,
    parameter MEM_DATA_WIDTH    = 'd32
    //-------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                               en              ,
    input  wire                                               arstn           , //  Asynchronous Reset
    input  wire                                               srstn           , //  Synchronous Reset
    input  wire                                               pclk            ,

    //-------------------------------------------------------------------------
    //  Timer
    input  wire [(PKT_WIDTH - 3 - G_PCIE_NUM_LANES)         -1:0] global_time ,

    //-------------------------------------------------------------------------
    //  PIPE RX Signals
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>SH_BITS)*8 -1:0] pl_rxdata   ,
    input  wire [G_PCIE_NUM_LANES*3                         -1:0] pl_rxstatus ,
    input  wire [G_PCIE_NUM_LANES                           -1:0] pl_rxvalid  ,

    //-------------------------------------------------------------------------
    //  RAM Port
    output wire                                                   ram_wren    ,
    output wire [MEM_DATA_WIDTH/8                           -1:0] ram_wrbe    ,
    output wire [MEM_ADDR_WIDTH                             -1:0] ram_wraddr  ,
    output wire [MEM_DATA_WIDTH                             -1:0] ram_wrdata        
    //-------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    localparam ACT_PIPE_WIDTH = G_PIPE_WIDTH >> SH_BITS;

    //  . Reported Error Monitoring
    reg                                                 trig_write            ;
    reg [PKT_WIDTH                                -1:0] pkt_rep_error         ; //  LTSSM Packet

    reg [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)      -1:0] det_edb               ;
    reg [G_PCIE_NUM_LANES                         -1:0] err_decode_err        ;
    reg [G_PCIE_NUM_LANES                         -1:0] err_elstc_buff_ovfl   ;
    reg [G_PCIE_NUM_LANES                         -1:0] err_elstc_buff_unfl   ;
    reg [G_PCIE_NUM_LANES                         -1:0] err_rx_disparity      ;


    //-------------------------------------------------------------------------
    // Monitoring errors reported by the PHY
    //-------------------------------------------------------------------------
    integer  i;

    // Combinational logic to detect EDB symbols
    always @(*) begin
      for (i = 0; i < (G_PCIE_NUM_LANES*(ACT_PIPE_WIDTH)); i=i+1) begin
        det_edb[i] = (pl_rxdata[8*i +:8] == `B_EDB) ? 1'b1: 1'b0;
      end
    end

    // Combinational logic to detect reported errors (Req 1.3A, 1.3B, 1.3C and 1.3E)
    // NOTE Actual width not considered
    always @(*) begin
      for (i = 0; i < G_PCIE_NUM_LANES; i=i+1) begin
        // 8.13.1 - 8B/10B Decode Errors | PIPE v6.0
        err_decode_err[i]       = (en && pl_rxvalid[i] && (pl_rxstatus[3*i +:3] == `RXST_DEC_ERR) && (det_edb[i*(ACT_PIPE_WIDTH) +:(ACT_PIPE_WIDTH)] > 0)) ? 1'b1 : 1'b0;

        // 8.13.2 - Disparity Errors | PIPE v6.0
        err_rx_disparity[i]     = (en && pl_rxvalid[i] && (pl_rxstatus[3*i +:3] == `RXST_DISP_ERR)) ? 1'b1 : 1'b0;

        // 8.13.3 - Elastic Buffer Errors | PIPE v6.0
        err_elstc_buff_ovfl[i]  = (en && pl_rxvalid[i] && (pl_rxstatus[3*i +:3] == `RXST_ELBUF_OVFL) && (det_edb[i*(ACT_PIPE_WIDTH) +:(ACT_PIPE_WIDTH)] > 0)) ? 1'b1 : 1'b0;
        err_elstc_buff_unfl[i]  = (en && pl_rxvalid[i] && (pl_rxstatus[3*i +:3] == `RXST_ELBUF_UNFL) && (det_edb[i*(ACT_PIPE_WIDTH) +:(ACT_PIPE_WIDTH)] > 0)) ? 1'b1 : 1'b0;
      end
    end

    // Sequential logic to create the packet to save into memory
    always @(posedge pclk or negedge arstn) begin
      if (!arstn) begin
        pkt_rep_error   <=  {(PKT_WIDTH) {1'b0}}  ;
        trig_write      <=  1'b0  ;
      end else begin
        if (!srstn) begin
          pkt_rep_error   <=  {(PKT_WIDTH) {1'b0}}  ;
          trig_write      <=  1'b0  ;
        end
        else if(err_decode_err > 0) begin
          pkt_rep_error   <=  {global_time, `RXST_DEC_ERR, err_decode_err}  ;
          trig_write      <=  1'b1 ;
        end
        else if(err_rx_disparity > 0) begin
          pkt_rep_error   <=  {global_time, `RXST_DISP_ERR, err_rx_disparity}  ;
          trig_write      <=  1'b1 ;
        end
        else if(err_elstc_buff_ovfl > 0) begin
          pkt_rep_error   <=  {global_time, `RXST_ELBUF_OVFL, err_elstc_buff_ovfl}  ;
          trig_write      <=  1'b1 ;
        end
        else if(err_elstc_buff_unfl > 0) begin
          pkt_rep_error   <=  {global_time, `RXST_ELBUF_UNFL, err_elstc_buff_unfl}  ;
          trig_write      <=  1'b1 ;
        end
        else begin
          trig_write    <=  1'b0 ;
        end
      end        
    end


    //-------------------------------------------------------------------------
    // Write Memory Controller
    //-------------------------------------------------------------------------

    write_memory_controller #(
      .PKT_WIDTH              (PKT_WIDTH                ),
      .MEM_ADDR_WIDTH         (MEM_ADDR_WIDTH           ),
      .MEM_DATA_WIDTH         (MEM_DATA_WIDTH           )
      //-------------------------------------------------------------------------
    ) write_memory_controller_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Write enable and data packet to send
      .trig_write             (trig_write               ),
      .packet_in              (pkt_rep_error            ),
      //  . Write Port
      .scram_wren             (ram_wren                 ),
      .scram_wrbe             (ram_wrbe                 ),
      .scram_wraddr           (ram_wraddr               ),
      .scram_wrdata           (ram_wrdata               )
      //-------------------------------------------------------------------------
    );

endmodule