//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: PIPE State Logic module. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Encoding Monitoring
//                (3) LTSSM and Phase Monitoring
//                (4) Write Memory Controller
//
// Dependency : write_memory_controller
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

module pipe_state_logic #(
    parameter PKT_WIDTH         = 'd16    ,     // Packet Width [bits]
    parameter MEM_ADDR_WIDTH    = 'd8     ,
    parameter MEM_DATA_WIDTH    = 'd32
  )(
    //-------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                         en                    ,
    input  wire                                         arstn                 , //  Asynchronous Reset
    input  wire                                         srstn                 , //  Synchronous Reset
    input  wire                                         pclk                  ,

    //-------------------------------------------------------------------------
    //  Timer
    input  wire [(PKT_WIDTH - 5)                -1:0]   global_time           ,

    //-------------------------------------------------------------------------
    //  PIPE Signals | PIPE State
    input  wire [5                                -1:0] pl_ltssm              ,
    input  wire [2                                -1:0] pl_equ_phase          ,

    //-------------------------------------------------------------------------
    //  PIPE Signals | PIPE Configuration
    input  wire [3                                -1:0] pl_rate               ,
    input  wire [3                                -1:0] pl_width              ,
    input  wire [5                                -1:0] pl_pclk_rate          ,

    //-------------------------------------------------------------------------
    //  Output
    output wire                                         data_encoding         ,
    output wire                                         detect_txn_ltssm      ,
    output wire [5                                -1:0] prev_ltssm            ,

    //-------------------------------------------------------------------------
    //  RAM Port
    output wire                                         ram_wren              ,
    output wire [MEM_DATA_WIDTH/8                 -1:0] ram_wrbe              ,
    output wire [MEM_ADDR_WIDTH                   -1:0] ram_wraddr            ,
    output wire [MEM_DATA_WIDTH                   -1:0] ram_wrdata        
    //-------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    //  . LTSSM Monitoring
    reg                                                 trig_write            ;
    reg [5                                        -1:0] reg_ltssm             ;
    reg [2                                        -1:0] reg_phase             ;
    reg [PKT_WIDTH                                -1:0] pkt_ltssm_phase       ; //  LTSSM Packet

    reg [5                                        -1:0] reg_prev_ltssm        ;
    
   
   
    //-------------------------------------------------------------------------
    // Encoding Monitoring
    //-------------------------------------------------------------------------
    assign data_encoding = !(pl_rate == 3'h0 || pl_rate == 3'h1) ; //  Check if Link data rate is 8.0 GT/s or higher


    //-------------------------------------------------------------------------
    // LTSSM and PHASE Monitoring - Req 1.1A and 1.5F
    //-------------------------------------------------------------------------

    always @(posedge pclk or negedge arstn) begin
      if (!arstn) begin
        reg_ltssm       <=  5'h00 ;
        reg_phase       <=  2'h0  ;
        pkt_ltssm_phase <=   'h0  ;
        trig_write      <=  1'b0  ;

        reg_prev_ltssm  <=  5'h00 ;
      end else begin
        if (!srstn) begin
          reg_ltssm       <=  5'h00 ;
          reg_phase       <=  2'h0  ;
          pkt_ltssm_phase <=   'h0  ;
          trig_write      <=  1'b0  ;

          reg_prev_ltssm  <=  5'h00 ;
        end
        else if (reg_ltssm != pl_ltssm && en) begin
          reg_prev_ltssm  <=  reg_ltssm ;
          reg_ltssm       <=  pl_ltssm  ;
          pkt_ltssm_phase <=  {global_time, pl_ltssm} ;             // Create packet -> | TIME | LTSSM CODE |
          trig_write      <=  1'b1  ;
        end
        else if (reg_phase != pl_equ_phase && pl_ltssm == 5'h0c && en) begin
          reg_phase       <=  pl_equ_phase  ;
          pkt_ltssm_phase <=  {global_time, 3'b111 ,pl_equ_phase} ; // Create packet -> | TIME | 111 | PHASE CODE |
          trig_write      <=  1'b1  ;
        end
        else begin
          trig_write    <=  1'b0 ;
        end
      end        
    end

    assign detect_txn_ltssm = trig_write;
    assign prev_ltssm = reg_prev_ltssm;


    //-------------------------------------------------------------------------
    // Write Memory Controller
    //-------------------------------------------------------------------------

    write_memory_controller #(
      .PKT_WIDTH              (PKT_WIDTH                ),
      .MEM_ADDR_WIDTH         (MEM_ADDR_WIDTH           ),
      .MEM_DATA_WIDTH         (MEM_DATA_WIDTH           )
      //-------------------------------------------------------------------------
    ) write_memory_controller_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Write enable and data packet to send
      .trig_write             (trig_write               ),
      .packet_in              (pkt_ltssm_phase          ),
      //  . Write Port
      .scram_wren             (ram_wren                 ),
      .scram_wrbe             (ram_wrbe                 ),
      .scram_wraddr           (ram_wraddr               ),
      .scram_wrdata           (ram_wrdata               )
      //-------------------------------------------------------------------------
    );

endmodule