//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Message Bus Logic module. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Message Bus Monitor
//
// Dependency : write_memory_controller, msg_bus_monitor_x1
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

module msg_bus_logic #(
    //-------------------------------------------------------------------------
    //  PIPE Parameters
    parameter G_PCIE_NUM_LANES  = 'd2       ,
    parameter G_PIPE_WIDTH      = 'h88422   ,    // Gen 4
    //-------------------------------------------------------------------------
    //  Module and Memory Parameters
    parameter PKT_WIDTH         = 'd32      ,     // Packet Width   [bits]
    parameter MEM_ADDR_WIDTH    = 'd8       ,
    parameter MEM_DATA_WIDTH    = 'd64
    //-------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                               en              ,
    input  wire                                               arstn           , //  Asynchronous Reset
    input  wire                                               srstn           , //  Synchronous Reset
    input  wire                                               pclk            ,

    //-------------------------------------------------------------------------
    //  Timer
    input  wire [(PKT_WIDTH - 1 - 8)                    -1:0] global_time     ,

    //-------------------------------------------------------------------------
    //  PIPE RX Signals
    input  wire [G_PCIE_NUM_LANES*8                     -1:0] pl_p2m_msgbus   ,
    input  wire [G_PCIE_NUM_LANES*8                     -1:0] pl_m2p_msgbus   ,

    //-------------------------------------------------------------------------
    //  RAM Port
    output wire [G_PCIE_NUM_LANES                       -1:0] ram_wren    ,
    output wire [G_PCIE_NUM_LANES * MEM_DATA_WIDTH/8    -1:0] ram_wrbe    ,
    output wire [G_PCIE_NUM_LANES * MEM_ADDR_WIDTH      -1:0] ram_wraddr  ,
    output wire [G_PCIE_NUM_LANES * MEM_DATA_WIDTH      -1:0] ram_wrdata        
    //-------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    //  . Message Bus Commands
    localparam NOP              = 4'b0000  ;
    localparam WRITE_UNCOMMITED = 4'b0001  ;
    localparam WRITE_COMMITED   = 4'b0010  ;
    localparam READ             = 4'b0011  ;
    localparam READ_C0MPL       = 4'b0100  ;
    localparam WRITE_ACK        = 4'b0101  ;


    //-------------------------------------------------------------------------
    // Message Bus Monitor
    //-------------------------------------------------------------------------
    genvar i;

    generate
      for (i = 0; i < G_PCIE_NUM_LANES; i = i + 1) begin
        msg_bus_monitor_x1 #(
          .PKT_WIDTH              (PKT_WIDTH                ),
          .MEM_ADDR_WIDTH         (MEM_ADDR_WIDTH           ),
          .MEM_DATA_WIDTH         (MEM_DATA_WIDTH           )
          //-------------------------------------------------------------------------
        ) msg_bus_monitor_x1_inst (
          //-------------------------------------------------------------------------
          //  . Clocks, EN and Resets
          .en                     (en                                        ),
          .arstn                  (arstn                                     ),
          .srstn                  (srstn                                     ),
          .pclk                   (pclk                                      ),
          //  . Timer
          .global_time            (global_time                               ),
          //  . PIPE RX Signals
          .pl_p2m_msgbus_x1       (pl_p2m_msgbus  [i*8                  +:8 ]),
          .pl_m2p_msgbus_x1       (pl_m2p_msgbus  [i*8                  +:8 ]),
          //  . Write Port
          .ram_wren               (ram_wren   [i                                        ]),
          .ram_wrbe               (ram_wrbe   [i*MEM_DATA_WIDTH/8     +:MEM_DATA_WIDTH/8]),
          .ram_wraddr             (ram_wraddr [i*MEM_ADDR_WIDTH       +:MEM_ADDR_WIDTH  ]),
          .ram_wrdata             (ram_wrdata [i*MEM_DATA_WIDTH       +:MEM_DATA_WIDTH  ])
          //-------------------------------------------------------------------------
        );
      end
    endgenerate

    



endmodule