//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor/Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Clock Frequency Monitor. Good luck :)
//              Module to estimate the clock signal frequency.
//
//              This module is composed of:
//                (1) Counter used as Clock Divider
//                (2) Clock Pulse Counter
//                (3) Output Register 
//                (4) 4-state FSM
//
// Dependency : none
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

module freq_monitor #(
    //-------------------------------------------------------------------------
    parameter  G_CNTR_WIDTH         =   'd8,    // Pulse Counter Width              ->  Affect the Measurable Max Frequence
    parameter  G_FDIV_CNTR_WIDTH    =   'd3     // Frequency Divider Counter Width  ->  Affect the Measurable Min Frequence
    //-------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                         fm_en                   ,
    input  wire                                         fm_arstn                ,
    input  wire                                         fm_refclk               ,
    input  wire                                         fm_measclk              ,

    //-------------------------------------------------------------------------
    //  Counter Output
    output wire [G_CNTR_WIDTH    -1:0]                  fm_cout                 
    //-------------------------------------------------------------------------
    );


    //-------------------------------------------------------------------------
    // Clock Divider
    //-------------------------------------------------------------------------
    reg [G_FDIV_CNTR_WIDTH      -1:0]   cdiv_cout   ;
    
    always @(posedge fm_refclk or negedge fm_arstn) begin
        if (!fm_arstn)
            cdiv_cout <= 0;
        else if (fm_en) begin
            cdiv_cout <= cdiv_cout + 'd1;
        end
    end

    
    //-------------------------------------------------------------------------
    // Clock Pulse Counter
    //-------------------------------------------------------------------------
    reg                                 pcnt_srstn  ;
    reg [G_CNTR_WIDTH           -1:0]   pcnt_cout   ;
    reg                                 pcnt_en     ;
    reg                                 pcnt_en_dly ;
   

    always @(posedge fm_refclk) begin
        pcnt_en <= cdiv_cout[G_FDIV_CNTR_WIDTH - 1];
    end

    always @(posedge fm_refclk) begin
        pcnt_en_dly <= pcnt_en;
        pcnt_srstn  <= ~pcnt_en && pcnt_en_dly;
    end
  
    always @(posedge fm_measclk or negedge fm_arstn) begin
        if (!fm_arstn)
            pcnt_cout <= 0;
        else begin
            if (pcnt_srstn)
                pcnt_cout <= 0;
            else if (pcnt_en && fm_en)
                pcnt_cout <= pcnt_cout + 1;
            else
                pcnt_cout <= pcnt_cout;
        end       
    end


    //-------------------------------------------------------------------------
    // Output Register
    //-------------------------------------------------------------------------
    reg [G_CNTR_WIDTH           -1:0]   reg_cout    ;
    wire                                reg_ld      ;

    assign reg_ld = ~pcnt_en && pcnt_en_dly;
    
    always @(posedge fm_refclk or negedge fm_arstn) begin
        if (!fm_arstn)
            reg_cout    <=  0   ;
        else begin
            if (reg_ld && fm_en)
                reg_cout    <=  pcnt_cout   ;
            else
                reg_cout    <=  reg_cout    ;
        end
    end

    assign fm_cout = reg_cout;

endmodule