//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Data Logic
//              Module to monitor and check tx_data and rx_data signals. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Data Decoding
//                (3) TS Counters
//                (4) Event Detector/Packet Encoder
//                (5) Writing Memory Controller
//
// Dependency : pipemc_shift_register, data_decoder_x1, write_memory_controller
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module data_logic #(
    //-------------------------------------------------------------------------------
    //  PIPE Parameters
    parameter G_PCIE_NUM_LANES  = 'd2       ,
    parameter G_PIPE_WIDTH      = 'h88422    ,   // Gen 4
    //-------------------------------------------------------------------------------
    //  Module and Memory Parameters
    parameter PKT_WIDTH         = 'd16      ,     // Packet Width   [bits]
    parameter MEM_ADDR_WIDTH    = 'd8       ,
    parameter MEM_DATA_WIDTH    = 'd32
    //-------------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                                en                   ,
    input  wire                                                arstn                , //  Asynchronous Reset#
    input  wire                                                srstn                , //  Synchronous  Reset#
    input  wire                                                pclk                 ,

    //-------------------------------------------------------------------------------
    //  Timer
    input  wire [(PKT_WIDTH - 5)                          -1:0] global_time         ,

    //-------------------------------------------------------------------------------
    //  PIPE Signals
    //  . Signals all lanes
    input  wire [4                                        -1:0] pl_powerdown        ,
    input  wire [3                                        -1:0] pl_rate             ,
    input  wire [5                                        -1:0] pl_ltssm            ,
    input  wire [2                                        -1:0] pl_equ_phase        ,

    input  wire [3                                        -1:0] pl_width            ,


    //  . Signals per lane - TX
    input  wire [G_PCIE_NUM_LANES                         -1:0] pl_txdetectrx       ,
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8    -1:0] pl_txdata           ,
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)      -1:0] pl_txdatak          ,
    input  wire [G_PCIE_NUM_LANES                         -1:0] pl_txdatavalid      ,

    //  . Signals per lane - RX
    input  wire [G_PCIE_NUM_LANES                         -1:0] pl_phystatus        ,
    input  wire [G_PCIE_NUM_LANES*3                       -1:0] pl_rxstatus         ,
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8    -1:0] pl_rxdata           ,
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)      -1:0] pl_rxdatak          ,
    input  wire [G_PCIE_NUM_LANES                         -1:0] pl_rxdatavalid      ,
    input  wire [G_PCIE_NUM_LANES                         -1:0] pl_rxvalid          ,

    //-------------------------------------------------------------------------------
    //  PIPE State Logic signals
    input  wire                                                 data_encoding       ,
    input  wire                                                 detect_txn_ltssm    ,
    input  wire [5                                        -1:0] prev_ltssm          ,

    //-------------------------------------------------------------------------------
    //  RAM Port
    output wire                                                 ram_wren            ,
    output wire [MEM_DATA_WIDTH/8                         -1:0] ram_wrbe            ,
    output wire [MEM_ADDR_WIDTH                           -1:0] ram_wraddr          ,
    output wire [MEM_DATA_WIDTH                           -1:0] ram_wrdata 
    //-------------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    localparam PKT_DEPTH      = 4;
    localparam CNT_PKTS_WIDTH = 2;

    // Detection Flags
    wire [G_PCIE_NUM_LANES                        -1:0] detect_rx             ;
    wire [G_PCIE_NUM_LANES*2                      -1:0] detect_ts1            ;
    wire [G_PCIE_NUM_LANES*2                      -1:0] detect_ts2            ;
    wire [G_PCIE_NUM_LANES*2                      -1:0] detect_eios           ;
    wire [G_PCIE_NUM_LANES*2                      -1:0] detect_eieos          ;
    wire [G_PCIE_NUM_LANES*2                      -1:0] detect_skpos          ;
    wire [G_PCIE_NUM_LANES*2                      -1:0] detect_kskpos         ;

    wire [G_PCIE_NUM_LANES*4                      -1:0] detect_chg_nlink      ;
    wire [G_PCIE_NUM_LANES*4                      -1:0] detect_chg_nlane      ;
    wire [G_PCIE_NUM_LANES*4                      -1:0] detect_chg_idr        ;
    wire [G_PCIE_NUM_LANES*4                      -1:0] detect_chg_psets      ;

    // TS Fields
    wire [G_PCIE_NUM_LANES*8*4                    -1:0] data_rate_ids         ;

    // TS Counter
    reg  [4                                       -1:0] cnt_16                ;
    reg  [16                                      -1:0] cnt_TS1_tx            ;
    reg  [16                                      -1:0] cnt_TS2_tx            ;
    reg  [16                                      -1:0] cnt_TS1_rx_cont       ;
    reg  [16                                      -1:0] cnt_TS2_rx_cont       ;
    reg  [16                                      -1:0] cnt_TS1_rx_cont_max   ;
    reg  [16                                      -1:0] cnt_TS2_rx_cont_max   ;

    //  Write Controller
    reg [CNT_PKTS_WIDTH                           -1:0] cnt_pkts              ;
    reg [PKT_DEPTH                                -1:0] trig_write            ;
    reg [PKT_WIDTH * PKT_DEPTH                    -1:0] pkt_evnt              ;



    //-------------------------------------------------------------------------
    // DATA Decoder
    //-------------------------------------------------------------------------
    genvar i;

    generate
      for (i = 0; i < G_PCIE_NUM_LANES; i = i + 1) begin
        data_decoder_x1 #(
          //-------------------------------------------------------------------------
          .G_PIPE_WIDTH            (G_PIPE_WIDTH            )
          //-------------------------------------------------------------------------
        ) data_decoder_x1_inst (
          //-------------------------------------------------------------------------
          //  Clocks, EN and Resets
          .en                     (en                                     ),
          .arstn                  (arstn                                  ),
          .srstn                  (srstn                                  ),
          .pclk                   (pclk                                   ),
          //-------------------------------------------------------------------------
          //  PIPE Signals
          //  . Signals all lanes
          .pl_powerdown           (pl_powerdown                           ),
          .pl_rate                (pl_rate                                ),
          .pl_width               (pl_width                               ),

          //  . Signals per lane
          .pl_txdetectrx          (pl_txdetectrx[i]                       ),
          .pl_txdata              (pl_txdata [(G_PIPE_WIDTH>>16)*8*i  +:(G_PIPE_WIDTH>>16)*8] ),
          .pl_txdatak             (pl_txdatak[(G_PIPE_WIDTH>>16)*i    +:(G_PIPE_WIDTH>>16)  ] ),
          .pl_txdatavalid         (pl_txdatavalid[i]                      ),

          .pl_phystatus           (pl_phystatus[i]                        ),
          .pl_rxstatus            (pl_rxstatus[i*3 +:3]                   ),
          .pl_rxdata              (pl_rxdata[(G_PIPE_WIDTH>>16)*8*i   +:(G_PIPE_WIDTH>>16)*8] ),
          .pl_rxdatak             (pl_rxdatak[(G_PIPE_WIDTH>>16)*i    +:(G_PIPE_WIDTH>>16)  ] ),
          .pl_rxdatavalid         (pl_rxdatavalid[i]                      ),

          //-------------------------------------------------------------------------
          //  PIPE State Logic signals
          .data_encoding          (data_encoding                          ), // 0 -> 8b/10b | 1 -> 128b/130b
          .detect_txn_ltssm       (detect_txn_ltssm                       ),

          //-------------------------------------------------------------------------
          //  Detection Flags
          .detect_rx              (detect_rx[i]                           ),
          .detect_ts1             (detect_ts1       [2*i  +:2]            ),  // | 0 TX | 1 RX |
          .detect_ts2             (detect_ts2       [2*i  +:2]            ),  // | 0 TX | 1 RX |
          .detect_eios            (detect_eios      [2*i  +:2]            ),  // | 0 TX | 1 RX |
          .detect_eieos           (detect_eieos     [2*i  +:2]            ),  // | 0 TX | 1 RX |
          .detect_skpos           (detect_skpos     [2*i  +:2]            ),  // | 0 TX | 1 RX |
          .detect_kskpos          (detect_kskpos    [2*i  +:2]            ),  // | 0 TX | 1 RX |

          .detect_chg_nlink       (detect_chg_nlink [4*i  +:4]            ),  // | 0 TS1 TX | 1 TS2 TX | 2 TS1 RX | 3 TS RX |
          .detect_chg_nlane       (detect_chg_nlane [4*i  +:4]            ),  // | 0 TS1 TX | 1 TS2 TX | 2 TS1 RX | 3 TS RX |
          .detect_chg_idr         (detect_chg_idr   [4*i  +:4]            ),  // | 0 TS1 TX | 1 TS2 TX | 2 TS1 RX | 3 TS RX |
          .detect_chg_psets       (detect_chg_psets [4*i  +:4]            ),  // | 0 TS1 TX | 1 TS2 TX | 2 TS1 RX | 3 TS RX |

          //-------------------------------------------------------------------------
          //  TS Fields
          .data_rate_ids          (data_rate_ids[i*8*4 +:8*4]             )   // | 0... DRI TX TS1 ...7 | 8... DRI TX TS2 ...15 | 16... DRI RX TS1 ...23 |24... DRI RX TS2 ...31 |
          //-------------------------------------------------------------------------
        ); 
      end
    endgenerate
    

    //-------------------------------------------------------------------------
    // TS1 and TS2 Counters
    //-------------------------------------------------------------------------
    // Count the sended TS1 and TS2
    always @(posedge pclk or negedge arstn) begin
      if(!arstn)begin
        cnt_TS1_tx     <= 16'h0000;
        cnt_TS2_tx     <= 16'h0000;
      end
      else begin
        if (!srstn || detect_txn_ltssm) begin
          cnt_TS1_tx     <= 16'h0000;
          cnt_TS2_tx     <= 16'h0000;
        end
        if (en && detect_ts1[0])
          cnt_TS1_tx <= cnt_TS1_tx + 16'h0001;
        if (en && detect_ts2[0])
          cnt_TS2_tx <= cnt_TS2_tx + 16'h0001;
      end      
    end

    
    // Count consecutive received TS1 and TS2
    always @(posedge pclk or negedge arstn) begin
      if(!arstn) begin
        cnt_TS1_rx_cont_max <= {(PKT_WIDTH){1'b0}};
        cnt_TS2_rx_cont_max <= {(PKT_WIDTH){1'b0}};
        cnt_TS1_rx_cont     <= {(PKT_WIDTH){1'b0}};
        cnt_TS2_rx_cont     <= {(PKT_WIDTH){1'b0}};

        cnt_16 <= 4'h0;
      end
      else begin
        if (!srstn || detect_txn_ltssm) begin
          cnt_TS1_rx_cont_max <= {(PKT_WIDTH){1'b0}};
          cnt_TS2_rx_cont_max <= {(PKT_WIDTH){1'b0}};
          cnt_TS1_rx_cont     <= {(PKT_WIDTH){1'b0}};
          cnt_TS2_rx_cont     <= {(PKT_WIDTH){1'b0}};

          cnt_16 <= 4'h0;
        end
        else if (en) begin

          if(cnt_TS1_rx_cont > cnt_TS1_rx_cont_max)
              cnt_TS1_rx_cont_max <= cnt_TS1_rx_cont;
          if(cnt_TS2_rx_cont > cnt_TS2_rx_cont_max)
              cnt_TS2_rx_cont_max <= cnt_TS2_rx_cont;

          
          case (pl_ltssm)
            `LTSSM_RCV_RLOCK: begin
              // if(detect_ts1[1]) $display ("[display] TS1 Detected \t time=%0t \t Data Rate Id: %b \t speed_change bit %0b", $time, data_rate_ids[2*8 +:8], data_rate_ids[7 + 2*8]);
                              
              if(detect_ts1[1] && data_rate_ids[7 + 2*8]) begin // Check speed_change bit
                cnt_16  <= 4'h0;
                cnt_TS1_rx_cont <= cnt_TS1_rx_cont + 16'h0001;
                cnt_TS2_rx_cont <= 16'h0000;
              end
              else if(detect_ts2[1] && data_rate_ids[7 + 3*8]) begin // Check speed_change bit
                cnt_16  <= 4'h0;
                cnt_TS1_rx_cont <= 16'h0000;
                cnt_TS2_rx_cont <= cnt_TS2_rx_cont + 16'h0001;
              end
              else if(cnt_16 == 4'hF) begin // Not consecuitive TS1/TS2
                cnt_16  <= 4'h0;
                cnt_TS1_rx_cont <= 16'h0000;
                cnt_TS2_rx_cont <= 16'h0000;
              end
              else
                cnt_16  <= cnt_16 + 4'h1;              
            end // LTSSM_RCV_RLOCK

            `LTSSM_RCV_RCFG: begin
              // if(detect_ts1[1]) $display ("[display] TS1 Detected \t time=%0t \t Data Rate Id: %b \t speed_change bit %0b", $time, data_rate_ids[2*8 +:8], data_rate_ids[7 + 2*8]);
              if((detect_chg_idr > 0 || detect_chg_idr > 0 ) && !detect_ts1[1]) begin
                cnt_TS2_rx_cont <= 16'h0000;
              end
              if(detect_ts1[1] && data_rate_ids[7 + 2*8]) begin // Check speed_change bit
                cnt_16  <= 4'h0;
                cnt_TS1_rx_cont <= cnt_TS1_rx_cont + 16'h0001;
                cnt_TS2_rx_cont <= 16'h0000;
              end
              else if(detect_ts2[1] && data_rate_ids[7 + 3*8]) begin // (Spec p360) Identical IDR and Symb 6, speed_change set to 1b | Next state Recovery.Speed
                cnt_16  <= 4'h0;
                cnt_TS1_rx_cont <= 16'h0000;
                cnt_TS2_rx_cont <= cnt_TS2_rx_cont + 16'h0001;
              end
              else if(cnt_16 == 4'hF) begin // Not consecuitive TS1/TS2
                cnt_16  <= 4'h0;
                cnt_TS1_rx_cont <= 16'h0000;
                cnt_TS2_rx_cont <= 16'h0000;
              end
              else
                cnt_16  <= cnt_16 + 4'h1;        
            end // LTSSM_RCV_RCFG

            default: begin
              if(detect_ts1[1]) begin
                cnt_16  <= 4'h0;
                cnt_TS1_rx_cont <= cnt_TS1_rx_cont + 16'h0001;
                cnt_TS2_rx_cont <= 16'h0000;
              end
              else if(detect_ts2[1]) begin
                cnt_16  <= 4'h0;
                cnt_TS2_rx_cont <= cnt_TS2_rx_cont + 16'h0001;
                cnt_TS1_rx_cont <= 16'h0000;
              end
              else if(cnt_16 == 4'hF) begin // Not consecuitive TS1/TS2
                cnt_16  <= 4'h0;
                cnt_TS1_rx_cont <= 16'h0000;
                cnt_TS2_rx_cont <= 16'h0000;
              end
              else
                cnt_16  <= cnt_16 + 4'h1;
            end // default
          endcase
        end // else if(en)
      end // else
    end // always


    //-------------------------------------------------------------------------
    // Event Detector and Packet Encoder
    //  see PCI Express® Base Specification Revision 5.0 Version 1.0 | 4.2.6 Link Training and Status Rules (p 315)
    //-------------------------------------------------------------------------
    localparam EVENT_ID_BIT     =                      'd5  ;

    localparam EVNT_RX_DET      =                     5'h0  ;
    localparam EVNT_SLOCK_START = (EVNT_RX_DET      + 5'h1) ;
    localparam EVNT_SLOCK_CMPL  = (EVNT_SLOCK_START + 5'h1) ;
    localparam EVNT_SLOCK_FAIL  = (EVNT_SLOCK_CMPL  + 5'h1) ;
    localparam EVNT_NLINK_SET   = (EVNT_SLOCK_FAIL  + 5'h1) ;
    localparam EVNT_NLANE_SET   = (EVNT_NLINK_SET   + 5'h1) ;
    localparam EVNT_SPEED_CH_P  = (EVNT_NLANE_SET   + 5'h1) ;
    localparam EVNT_SPEED_CH_R  = (EVNT_SPEED_CH_P  + 5'h1) ;
    localparam EVNT_NLINK_CHG   = (EVNT_SPEED_CH_R  + 5'h1) ;
    localparam EVNT_NLANE_CHG   = (EVNT_NLINK_CHG   + 5'h1) ;
    localparam EVNT_EIEOS       = (EVNT_NLANE_CHG   + 5'h1) ; // 10

    always @(posedge pclk or negedge arstn) begin
      if (!arstn) begin
        trig_write  <= {(PKT_DEPTH){1'b0}};
        pkt_evnt    <= {(PKT_WIDTH*PKT_DEPTH){1'b0}};
        cnt_pkts    <= {(CNT_PKTS_WIDTH){1'b0}};
      end else begin
        if (!srstn) begin
          trig_write  <= {(PKT_DEPTH){1'b0}};
          pkt_evnt    <= {(PKT_WIDTH*PKT_DEPTH){1'b0}};
          cnt_pkts    <= {(CNT_PKTS_WIDTH){1'b0}};
        end
        else if (en) begin
          if(cnt_pkts > 0) begin
            trig_write  <= trig_write >> 1;
            pkt_evnt    <= pkt_evnt >> PKT_WIDTH;
            cnt_pkts    <= cnt_pkts - 'd1;
          end

          case (pl_ltssm)
            `LTSSM_DET_QUIET: begin
              if (detect_txn_ltssm && prev_ltssm == `LTSSM_POL_ACT) begin // Req 1.4 | Bit/Symbol Lock Failed
                $display ("[pmc] time=%0t \t Symbol Lock Failed \t TS1 Transmitted: %0d \t MAX TS1 Received Consecutively %0d", $time, cnt_TS1_tx, cnt_TS1_rx_cont_max);

                pkt_evnt [cnt_pkts      * PKT_WIDTH  +:PKT_WIDTH] <=  {global_time, EVNT_SLOCK_FAIL} ;  // Create packet (0) -> | TIME | EVENT CODE |
                pkt_evnt [(cnt_pkts + 1)* PKT_WIDTH  +:PKT_WIDTH] <=  {cnt_TS1_tx} ;                    // Create packet (1) -> | NUM of TS1 Transm |
                pkt_evnt [(cnt_pkts + 2)* PKT_WIDTH  +:PKT_WIDTH] <=  {cnt_TS1_rx_cont_max} ;           // Create packet (2) -> | NUM of TS1 Receiv |

                trig_write[cnt_pkts +:3]  <=  3'b111        ; // It sends three packets
                cnt_pkts                  <= cnt_pkts + 'd3 ; // It sends three packets
              end
            end
            `LTSSM_DET_ACT: begin
              if (detect_rx > 0) begin // Req 1.4B | Receiver Detection
                $display ("[pmc] time=%0t \t Recevier Detection \t Lane Detected: %0b", $time, detect_rx);

                pkt_evnt [cnt_pkts*PKT_WIDTH  +:PKT_WIDTH]  <=  {global_time, EVNT_RX_DET} ; // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]    <=  1'b1          ;
                cnt_pkts                <= cnt_pkts + 'd1 ;
              end
            end
            `LTSSM_POL_ACT: begin 
              if ((detect_ts1[0] > 0) && cnt_TS1_tx == 16'h0000 ) begin // Req 1.4C | Start Bit/Symbol Lock Sequence
                $display ("[pmc] time=%0t \t Bit/Symbol Lock Sequence Started ", $time);
                
                pkt_evnt [cnt_pkts * PKT_WIDTH  +:PKT_WIDTH]  <=  {global_time, EVNT_SLOCK_START} ;  // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]    <=  1'b1          ;
                cnt_pkts                <= cnt_pkts + 'd1 ;
              end
            end
            `LTSSM_POL_CFG: begin
              if (detect_txn_ltssm && prev_ltssm == `LTSSM_POL_ACT) begin // Req 1.4D | Bit/Symbol Lock Complete Successfully
                $display ("[display] time=%0t \t Bit/Symbol Lock Complete \t TS1 Transmitted: %0d \t MAX TS1 Received Consecutively %0d", $time, cnt_TS1_tx, cnt_TS1_rx_cont_max);
                
                pkt_evnt [cnt_pkts      * PKT_WIDTH  +:PKT_WIDTH] <=  {global_time, EVNT_SLOCK_CMPL} ;  // Create packet (0) -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]  <=  1'b1              ;
                cnt_pkts <= cnt_pkts + 'd1                  ;
              end
            end
            `LTSSM_CFG_LWSTR: begin
              if (detect_chg_nlink > 0) begin // Req 1.4E | Link Number assignation
                $display ("[pmc] time=%0t \t Link number assigned", $time);

                pkt_evnt  <=  {global_time, EVNT_NLINK_SET} ; // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]  <=  1'b1              ;
                cnt_pkts              <= cnt_pkts + 'd1     ;
              end
            end
            `LTSSM_CFG_LWAIT: begin
              if (detect_chg_nlane > 0) begin // Req 1.4F | Lane Number Assignation
                $display ("[pmc] time=%0t \t Lane number assigned ", $time);

                pkt_evnt  <=  {global_time, EVNT_NLANE_SET} ; // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]  <=  1'b1              ;
                cnt_pkts              <= cnt_pkts + 'd1     ;
              end
            end
            `LTSSM_L0: begin
              if (detect_txn_ltssm && (
                  (pl_rate == `LINK_RATE_25GT && data_rate_ids[2] && data_rate_ids[2+2*8]) || // 5  GT/s Supported
                  (pl_rate == `LINK_RATE_5GT  && data_rate_ids[3] && data_rate_ids[3+2*8]) || // 8  GT/s Supported
                  (pl_rate == `LINK_RATE_8GT  && data_rate_ids[4] && data_rate_ids[4+2*8]) || // 16 GT/s Supported
                  (pl_rate == `LINK_RATE_16GT && data_rate_ids[5] && data_rate_ids[5+2*8])    // 32 GT/s Supported
                  )) begin // Req 1.5A | Detect a Possible Speed Change
                $display ("[pmc] time=%0t \t Possible Speed Change \t Current Data Rate: b%b \t Data Rate Identifier TX: b%b \t RX: b%b", $time, pl_rate, data_rate_ids[7:0], data_rate_ids[23:16]);
                
                pkt_evnt  <=  {global_time, EVNT_SPEED_CH_P} ;  // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]  <=  1'b1              ;
                cnt_pkts              <= cnt_pkts + 'd1     ;
              end
            end
            `LTSSM_RCV_RCFG: begin
              if (detect_txn_ltssm && (cnt_TS1_rx_cont_max > 7 || cnt_TS2_rx_cont_max > 7) && !data_encoding) begin // Req 1.5B | Speed Change Request (MindShare p575)
                $display ("[pmc] time=%0t \t Sequence to Change Speed Started (Data Rate 2.5GT/s)", $time);

                pkt_evnt  <=  {global_time, EVNT_SPEED_CH_R} ;  // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]  <=  1'b1              ;
                cnt_pkts              <= cnt_pkts + 'd1     ;
              end

              if (detect_chg_nlink > 0) begin // Unexpected change of Link number
                $display ("[pmc] time=%0t \t Unexpected change of Link number", $time);
                
                pkt_evnt  <=  {global_time, EVNT_NLINK_CHG} ;  // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]  <=  1'b1              ;
                cnt_pkts              <= cnt_pkts + 'd1     ;
              end

              if (detect_chg_nlink > 0) begin // Unexpected change of Lane number
                $display ("[pmc] time=%0t \t Unexpected change of Lane number", $time);
                
                pkt_evnt  <=  {global_time, EVNT_NLANE_CHG} ;  // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]  <=  1'b1              ;
                cnt_pkts              <= cnt_pkts + 'd1     ;
              end

              if (detect_eieos > 0) begin
                $display ("[pmc] time=%0t \t EIEOS Detected", $time);
                
                pkt_evnt  <=  {global_time, EVNT_EIEOS} ;  // Create packet -> | TIME | EVENT CODE |
                trig_write[cnt_pkts]  <=  1'b1              ;
                cnt_pkts              <= cnt_pkts + 'd1     ;
              end
            end
            default: begin

            end
          endcase
        end
      end        
    end
  


    //-------------------------------------------------------------------------
    // Write Memory Controller
    //-------------------------------------------------------------------------
    always @(posedge pclk or negedge arstn) begin
      if (!arstn) begin
        trig_write  <= {(PKT_DEPTH){1'b0}};
        pkt_evnt    <= {(PKT_WIDTH*PKT_DEPTH){1'b0}};
      end
      else begin
        if(!srstn) begin
          trig_write  <= {(PKT_DEPTH){1'b0}};
          pkt_evnt    <= {(PKT_WIDTH*PKT_DEPTH){1'b0}};
        end
        else if (en) begin
          trig_write  <= trig_write >> 1;
          pkt_evnt    <= pkt_evnt >> PKT_WIDTH;
        end
      end
    end

    write_memory_controller #(
      //-------------------------------------------------------------------------
      .PKT_WIDTH              (PKT_WIDTH                 ),
      .MEM_ADDR_WIDTH         (MEM_ADDR_WIDTH            ),
      .MEM_DATA_WIDTH         (MEM_DATA_WIDTH            )
      //-------------------------------------------------------------------------
    ) write_memory_controller_event_tx_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (en                         ),
      .arstn                  (arstn                      ),
      .srstn                  (srstn                      ),
      .pclk                   (pclk                       ),
      //  . Write enable and data packet to send
      .trig_write             (trig_write[0]              ),
      .packet_in              (pkt_evnt[PKT_WIDTH   -1:0] ),
      //  . Write Port
      .scram_wren             (ram_wren                   ),
      .scram_wrbe             (ram_wrbe                   ),
      .scram_wraddr           (ram_wraddr                 ),
      .scram_wrdata           (ram_wrdata                 )
      //-------------------------------------------------------------------------
    );

endmodule