//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Training Sequence Ordered Set Detector. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Output assignations
//                (3) TS Detector
//                (4) TS Field sampling
//
// Dependency : 
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module tsos_detector #(
    //-------------------------------------------------------------------------------
    //  Shift Registers Parameters
    parameter SHIFT_DATA_WIDTH  = 'd8 ,
    parameter SHIFT_DATAK_WIDTH = 'd1 ,
    parameter SHIFT_DATA_DEPTH  = 'd16,
    //-------------------------------------------------------------------------------
    //  TS Symbols
    parameter B_TS              = `B_TS1      ,     // Packet Width   [bits]
    parameter D_TS              = `D_TS1       
    //-------------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------------
    //  Enable, Resets and Clock
    input  wire                                                en                   ,
    input  wire                                                arstn                , //  Asynchronous Reset
    input  wire                                                srstn                , //  Synchronous Reset
    input  wire                                                pclk                 ,
    //-------------------------------------------------------------------------------
    //  Data Encoding
    input  wire                                                data_encoding        ,
    //-------------------------------------------------------------------------------
    //  Parallel Data and DataK
    input  wire [SHIFT_DATA_DEPTH * SHIFT_DATA_WIDTH     -1:0] paral_data           ,
    input  wire [SHIFT_DATA_DEPTH * SHIFT_DATAK_WIDTH    -1:0] paral_datak          ,

    //-------------------------------------------------------------------------------
    //  Detection Signal
    output wire                                                detect_ts            ,
    output wire                                                detect_chg_nlink     , // Symbol 1
    output wire                                                detect_chg_nlane     , // Symbol 2
    output wire                                                detect_chg_idr       , // Symbol 4
    output wire                                                detect_chg_psets     , // Symbol 6
    //-------------------------------------------------------------------------------
    //  TS Fields
    output wire [8                                       -1:0] nlink                , // Symbol 1
    output wire [8                                       -1:0] nlane                , // Symbol 2
    output wire [8                                       -1:0] data_rate_id         , // Symbol 4
    output wire [8                                       -1:0] presets                // Symbol 6
    //-------------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    // TS Detection Flags
    reg                                                 reg_detect_ts         ;
    reg                                                 reg_detect_chg_nlink  ;
    reg                                                 reg_detect_chg_nlane  ;
    reg                                                 reg_detect_chg_idr    ;
    reg                                                 reg_detect_chg_psets  ;

    // TS Fields
    reg [8                                        -1:0] reg_nlink             ;
    reg [8                                        -1:0] reg_nlane             ;
    reg [8                                        -1:0] reg_data_rate_id      ;
    reg [8                                        -1:0] reg_presets           ;

    //-------------------------------------------------------------------------
    // Output assignations
    //-------------------------------------------------------------------------
    // TS Detection Flags
    assign detect_ts        = reg_detect_ts         ;
    assign detect_chg_nlink = reg_detect_chg_nlink  ;
    assign detect_chg_nlane = reg_detect_chg_nlane  ;
    assign detect_chg_idr   = reg_detect_chg_idr    ;
    assign detect_chg_psets = reg_detect_chg_psets  ;

    // TS Fields
    assign nlink            = reg_nlink             ;
    assign nlane            = reg_nlane             ;
    assign data_rate_id     = reg_data_rate_id      ;
    assign presets          = reg_presets           ;

  
    //-------------------------------------------------------------------------
    // TS1/TS2 Detector (Combinational Logic)
    //  see PCI Express® Base Specification Revision 5.0 Version 1.0 | 4.2.4.1 Training Sequences (p 228)
    //-------------------------------------------------------------------------
    always @(*) begin
      if (en) begin
        if (data_encoding == 1'b1 &&  // 128b/130b encoding
            paral_data[7:0]      == B_TS // && // Symbol 0
//            paral_data[87:80]    == D_TS && // Symbol 10 Scrambled
//            paral_data[95:88]    == D_TS && // Symbol 11 Scrambled
//            paral_data[103:96]   == D_TS && // Symbol 12 Scrambled
//            paral_data[111:104]  == D_TS    // Symbol 13 Scrambled
            ) begin 
          reg_detect_ts = 1'b1;
        end
        else if (data_encoding == 1'b0 && // 8b/10b encoding
            paral_data[7:0]      == `B_COM && paral_datak[0]  == 1'b1 && // Symbol 0
            paral_data[63:56]    == D_TS && paral_datak[7]    == 1'b0 && // Symbol 7
            paral_data[71:64]    == D_TS && paral_datak[8]    == 1'b0 && // Symbol 8
            paral_data[79:72]    == D_TS && paral_datak[9]    == 1'b0 && // Symbol 9
            paral_data[87:80]    == D_TS && paral_datak[10]   == 1'b0 && // Symbol 10
            paral_data[95:88]    == D_TS && paral_datak[11]   == 1'b0 && // Symbol 11
            paral_data[103:96]   == D_TS && paral_datak[12]   == 1'b0 && // Symbol 12
            paral_data[111:104]  == D_TS && paral_datak[13]   == 1'b0 && // Symbol 13
            paral_data[119:112]  == D_TS && paral_datak[14]   == 1'b0 && // Symbol 14
            paral_data[127:120]  == D_TS && paral_datak[15]   == 1'b0    // Symbol 15
            ) begin 
          reg_detect_ts = 1'b1;
        end
        else begin
          reg_detect_ts = 1'b0;
        end
/*
        if(paral_data[7:0] == `B_COM)begin
          $display ("[display] time=%0t \t %0h \t %0h \t %0h \t %0h \t %0h \t %0h \t %0h \t %0h \t %0h \t %0h", $time, paral_data[7:0]      == `B_COM, paral_data[63:56]    == D_TS,
            paral_data[71:64]    == D_TS, paral_data[79:72]    == D_TS, paral_data[87:80]    == D_TS, paral_data[95:88]    == D_TS,
            paral_data[103:96]   == D_TS, paral_data[111:104]  == D_TS, paral_data[119:112]  == D_TS, paral_data[127:120]  == D_TS  );
        end*/
      end
    end

    //-------------------------------------------------------------------------
    // TS1/TS2 Field Sampling
    //-------------------------------------------------------------------------   
    always @(posedge pclk or negedge arstn) begin
      if (!arstn) begin
        reg_nlink             <= `B_PAD;
        reg_nlane             <= `B_PAD;
        reg_data_rate_id      <= {2'b00, `DRI_2_5GT_SUPP, 1'b0};

        reg_detect_chg_nlink  <= 1'b0;
        reg_detect_chg_nlane  <= 1'b0;
        reg_detect_chg_idr    <= 1'b0;
        reg_detect_chg_psets  <= 1'b0;
      end
      else begin
        if(!srstn) begin
          reg_nlink             <= `B_PAD;
          reg_nlane             <= `B_PAD;
          reg_data_rate_id      <= {2'b00, `DRI_2_5GT_SUPP, 1'b0};

          reg_detect_chg_nlink  <= 1'b0;
          reg_detect_chg_nlane  <= 1'b0;
          reg_detect_chg_idr    <= 1'b0;
          reg_detect_chg_psets  <= 1'b0;
        end
        else if (en && reg_detect_ts && data_encoding == 1'b0) begin 
          if (reg_nlink == `B_PAD && paral_data[15:8] != reg_nlink) begin // Warning
             reg_detect_chg_nlink  <= 1'b1;
             reg_nlink <= paral_data[15:8];
          end
          else if (reg_nlane == `B_PAD && paral_data[23:16] != reg_nlane) begin
            reg_detect_chg_nlane  <= 1'b1;
            reg_nlane <= paral_data[23:16];
          end
          else if (paral_data[39:32] != reg_data_rate_id) begin
            reg_detect_chg_idr    <= 1'b1;
            reg_data_rate_id <= paral_data[39:32];
          end
          else if (paral_data[55:48] != reg_presets) begin
            reg_detect_chg_psets    <= 1'b1;
            reg_presets <= paral_data[55:48];
          end
        end
        else begin
          reg_detect_chg_nlink  <= 1'b0;
          reg_detect_chg_nlane  <= 1'b0;
          reg_detect_chg_idr    <= 1'b0;
          reg_detect_chg_psets  <= 1'b0;
        end
      end
    end

endmodule