//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Training Sequence Ordered Set Detector. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Output assignations
//                (3) TS Detector
//                (4) TS Field sampling
//
// Dependency : 
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module skpos_detector #(
      //-------------------------------------------------------------------------------
    //  SKPOS D Parameters
    parameter TX_RXn_PATTERN    = 1'b1 , // If 1, it looks for transmitted SKPOSs, otherwise it detects received SKPOSs
    //-------------------------------------------------------------------------------
    //  Shift Registers Parameters
    parameter SHIFT_DATA_WIDTH  = 'd8 ,
    parameter SHIFT_DATAK_WIDTH = 'd1 ,
    parameter SHIFT_DATA_DEPTH  = 'd16
    //-------------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------------
    //  Enable, Resets and Clock
    input  wire                                                en                   ,
    input  wire                                                arstn                , //  Asynchronous Reset
    input  wire                                                srstn                , //  Synchronous Reset
    input  wire                                                pclk                 ,
    //-------------------------------------------------------------------------------
    //  PIPE Signals
    input  wire [3                                       -1:0] pl_rate              ,
    //-------------------------------------------------------------------------------
    //  Data Encoding
    input  wire                                                data_encoding        ,
    //-------------------------------------------------------------------------------
    //  Parallel Data and DataK
    input  wire [SHIFT_DATA_DEPTH * SHIFT_DATA_WIDTH     -1:0] paral_data           ,
    input  wire [SHIFT_DATA_DEPTH * SHIFT_DATAK_WIDTH    -1:0] paral_datak          ,

    //-------------------------------------------------------------------------------
    //  Detection Signal
    output wire                                                detect_skpos         ,
    output wire                                                detect_kskpos            
    //-------------------------------------------------------------------------------
  );

    //-----------------------------------------------------------------------------
    // Local declarations 
    //-----------------------------------------------------------------------------
    // Detection Flags
    reg                                                 reg_detect_skpos          ;
    reg                                                 reg_detect_kskpos         ;

    //-----------------------------------------------------------------------------
    // Output assignations
    //-----------------------------------------------------------------------------
    assign detect_skpos   = reg_detect_skpos  ;
    assign detect_kskpos  = reg_detect_kskpos ;

    //-----------------------------------------------------------------------------------------------------------------------
    // SKPOS Detectors
    //  see PCI Express® Base Specification Revision 5.0 Version 1.0 | 4.2.7.1 SKP Ordered Set for 8b/10b Encoding    (p 382)
    //                                                               | 4.2.7.2 SKP Ordered Set for 128b/130b Encoding (p 382)
    //-----------------------------------------------------------------------------------------------------------------------

    generate

      if (TX_RXn_PATTERN) begin // Detect Transmitted SKP Ordered Sets
        always @(*) begin
          if (!data_encoding && // 8b/10b Encoding
              {paral_datak[0], paral_data[7:0]} == `K_COM && // Symbol 0
              {paral_datak[1], paral_data[15:8], paral_datak[2], paral_data[23:16], paral_datak[3], paral_data[32:24]} == {(3){`K_SKP}} // Symbols 1-3
          ) begin
            $display ("[pmc] time=%0t \t SKP Ordered Set Detected (2.5 or 5 GT/s | trasmitted)", $time);
            reg_detect_skpos  = 1'b1;
            reg_detect_kskpos = 1'b0;
          end
          if (data_encoding && // 128b/130b Encoding
              paral_data[95:0] == {(12){8'hAA}} // SKPOS Header | Symbols 0-11
          ) begin
            if (paral_data[103:96] == `T_SKP_END) begin // Standard SKP Ordered Set END | Symbol 12
              $display ("[pmc] time=%0t \t Standard SKP Ordered Set Detected (8 GT/s or greater | trasmitted)", $time);
              reg_detect_skpos  = 1'b1;
              reg_detect_kskpos = 1'b0;
            end
            else if (paral_data[103:96] == `T_SKP_END_CTL) begin  // Control SKP Ordered Set END | Symbol 12
              $display ("[pmc] time=%0t \t Control SKP Ordered Set Detected (8 GT/s or greater | trasmitted)", $time);
              reg_detect_skpos  = 1'b1;
              reg_detect_kskpos = 1'b1;
            end
            else begin
              reg_detect_skpos  = 1'b0;
              reg_detect_kskpos = 1'b0;
            end
          end
          else
            reg_detect_skpos  = 1'b0;
            reg_detect_kskpos = 1'b0;
        end
      end

      else begin  // Detect Received SKP Ordered Sets
        always @(*) begin
          if (!data_encoding && // 8b/10b Encoding
              {paral_datak[0], paral_data[7:0]} == `K_COM &&      // Symbol 0
              {paral_datak[1], paral_data[15:8]} == {(3){`K_SKP}} // Symbol 1
            ) begin
            // $display ("[pmc] time=%0t \t SKP Ordered Set Detected (2.5 or 5 GT/s | Received)", $time);
            reg_detect_skpos  = 1'b1;
            reg_detect_kskpos = 1'b0;
          end
          if (data_encoding && // 128b/130b Encoding
              paral_data[95:64] == {(4){8'hAA}} // SKPOS Header | Symbols 8-11
          ) begin
            if (paral_data[103:96] == `T_SKP_END) begin // Standard SKP Ordered Set END | Symbol 12
              // $display ("[pmc] time=%0t \t Standard SKP Ordered Set Detected (8 GT/s or greater | received)", $time);
              reg_detect_skpos  = 1'b1;
              reg_detect_kskpos = 1'b0;
            end
            else if (paral_data[103:96] == `T_SKP_END_CTL) begin  // Control SKP Ordered Set END | Symbol 12
              // $display ("[pmc] time=%0t \t Control SKP Ordered Set Detected (8 GT/s or greater | received)", $time);
              reg_detect_skpos  = 1'b1;
              reg_detect_kskpos = 1'b1;
            end
            else begin
              reg_detect_skpos  = 1'b0;
              reg_detect_kskpos = 1'b0;
            end
          end
          else
            reg_detect_skpos  = 1'b0;
            reg_detect_kskpos = 1'b0;
        end
      end

    endgenerate

    
   


endmodule