//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Reported Error Logic
//              Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) 
//
// Dependency :
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module pipemc_shift_register #(
    //-------------------------------------------------------------------------------
    //  PIPE Parameters
    parameter G_PIPE_WIDTH      = 'h88422  ,   // Gen 4 
    //-------------------------------------------------------------------------------
    //  Shift Registers Parameters
    parameter SHIFT_WIDTH       = 'd8       ,
    parameter SHIFT_DEPTH       = 'd16 
    //-------------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                                 en                  ,
    input  wire                                                 arstn               , //  Asynchronous Reset
    input  wire                                                 srstn               , //  Synchronous Reset
    input  wire                                                 pclk                ,
    //-------------------------------------------------------------------------------
    //  PIPE Signals
    input  wire [3                                        -1:0] pl_width            ,
    //-------------------------------------------------------------------------------
    //  Input Data
    input  wire [(G_PIPE_WIDTH>>16)*SHIFT_WIDTH           -1:0] shift_data_in       ,
    input  wire                                                 shift_datavalid     ,
    //-------------------------------------------------------------------------------
    //  Parallel Data Output
    output wire [SHIFT_DEPTH * SHIFT_WIDTH                -1:0] shift_data_out    
    //-------------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    reg [SHIFT_DEPTH * SHIFT_WIDTH                -1:0] reg_shift_data_out    ;

    //-------------------------------------------------------------------------
    // Shift Register
    //-------------------------------------------------------------------------

    assign shift_data_out = reg_shift_data_out;

    always @(posedge pclk or negedge arstn) begin
      if (!arstn)
        reg_shift_data_out <= {(SHIFT_DEPTH * SHIFT_WIDTH){1'b0}};
      else begin
        if (!srstn)
          reg_shift_data_out <= {(SHIFT_DEPTH * SHIFT_WIDTH){1'b0}};
        else if (en && shift_datavalid) begin
          if (pl_width == `PIPE_WIDTH_8) begin // 8 bit Datapath width
            reg_shift_data_out <= (reg_shift_data_out >> SHIFT_WIDTH) ;
            reg_shift_data_out[SHIFT_DEPTH * SHIFT_WIDTH - 1: (SHIFT_DEPTH - 1) * SHIFT_WIDTH] <= shift_data_in;
          end
          else if (pl_width == `PIPE_WIDTH_16) begin // 16 bit Datapath width
            reg_shift_data_out <= (reg_shift_data_out >> 2 *SHIFT_WIDTH) ;
            reg_shift_data_out[SHIFT_DEPTH * SHIFT_WIDTH - 1: (SHIFT_DEPTH - 2) * SHIFT_WIDTH] <= shift_data_in;
          end
          else if (pl_width == `PIPE_WIDTH_32) begin // 32 bit Datapath width
            reg_shift_data_out <= (reg_shift_data_out >> 4 * SHIFT_WIDTH) ;
            reg_shift_data_out[SHIFT_DEPTH * SHIFT_WIDTH - 1: (SHIFT_DEPTH - 4) * SHIFT_WIDTH] <= shift_data_in;
          end
          else if (pl_width == `PIPE_WIDTH_64) begin // 64 bit Datapath width
            reg_shift_data_out <= (reg_shift_data_out >> 8 * SHIFT_WIDTH)  ;
            reg_shift_data_out[SHIFT_DEPTH * SHIFT_WIDTH - 1: (SHIFT_DEPTH - 8) * SHIFT_WIDTH] <= shift_data_in;
          end
          else if (pl_width == `PIPE_WIDTH_128) begin // 128 bit Datapath width
            reg_shift_data_out <= (reg_shift_data_out >> 16 * SHIFT_WIDTH)  ;
            reg_shift_data_out[SHIFT_DEPTH * SHIFT_WIDTH - 1: (SHIFT_DEPTH - 16) * SHIFT_WIDTH] <= shift_data_in;
          end
        end
      end
    end



endmodule