//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Message Bus monitor for single Lane configuration. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Pipeline Input Registers
//                (3) Message Bus Monitor
//                (4) Write Memory Controller
//
// Dependency : write_memory_controller
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

module msg_bus_monitor_x1 #( 
    //-------------------------------------------------------------------------
    //  Module and Memory Parameters
    parameter PKT_WIDTH         = 'd16      ,     // Packet Width   [bits]
    parameter MEM_ADDR_WIDTH    = 'd8       ,
    parameter MEM_DATA_WIDTH    = 'd64
    //-------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                               en              ,
    input  wire                                               arstn           , //  Asynchronous Reset
    input  wire                                               srstn           , //  Synchronous Reset
    input  wire                                               pclk            ,

    //-------------------------------------------------------------------------
    //  Timer
    input  wire [(PKT_WIDTH - 1 - 8)                    -1:0] global_time     ,

    //-------------------------------------------------------------------------
    //  PIPE RX Signals
    input  wire [8                                      -1:0] pl_p2m_msgbus_x1,
    input  wire [8                                      -1:0] pl_m2p_msgbus_x1,

    //-------------------------------------------------------------------------
    //  RAM Port
    output wire                                               ram_wren        ,
    output wire [MEM_DATA_WIDTH/8                       -1:0] ram_wrbe        ,
    output wire [MEM_ADDR_WIDTH                         -1:0] ram_wraddr      ,
    output wire [MEM_DATA_WIDTH                         -1:0] ram_wrdata        
    //-------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------


    //  . Message Bus Commands
    localparam NOP              = 4'b0000  ;
    localparam WRITE_UNCOMMITED = 4'b0001  ;
    localparam WRITE_COMMITED   = 4'b0010  ;
    localparam READ             = 4'b0011  ;
    localparam READ_C0MPL       = 4'b0100  ;
    localparam WRITE_ACK        = 4'b0101  ;

    //  . Input registers
    reg [8                                        -1:0] reg_in1_p2m_msgbus    ;
    reg [8                                        -1:0] reg_in1_m2p_msgbus    ;
    reg [8                                        -1:0] reg_in2_p2m_msgbus    ;
    reg [8                                        -1:0] reg_in2_m2p_msgbus    ;

    //  . Message Bus Monitoring
    reg                                                 trig_write            ;
    reg [PKT_WIDTH                                -1:0] pkt_msgbus            ; //  LTSSM Packet

    //-------------------------------------------------------------------------
    // Pipeline Input Registers
    //-------------------------------------------------------------------------
    always @(posedge pclk or negedge arstn) begin
      if (!arstn) begin
        reg_in1_p2m_msgbus <= 8'h00  ;
        reg_in1_m2p_msgbus <= 8'h00  ;
        reg_in2_p2m_msgbus <= 8'h00  ;
        reg_in2_m2p_msgbus <= 8'h00  ;
      end else begin
        if (!srstn) begin
          reg_in1_p2m_msgbus <= 8'h00  ;
          reg_in1_m2p_msgbus <= 8'h00  ;
          reg_in2_p2m_msgbus <= 8'h00  ;
          reg_in2_m2p_msgbus <= 8'h00  ;
        end
        else if (en) begin
          reg_in1_p2m_msgbus <= pl_p2m_msgbus_x1              ;
          reg_in1_m2p_msgbus <= pl_m2p_msgbus_x1              ;
          reg_in2_p2m_msgbus <= reg_in1_p2m_msgbus            ;
          reg_in2_m2p_msgbus <= reg_in1_m2p_msgbus            ;
        end
        else begin
          reg_in1_p2m_msgbus <= 8'h00  ;
          reg_in1_m2p_msgbus <= 8'h00  ;
          reg_in2_p2m_msgbus <= 8'h00  ;
          reg_in2_m2p_msgbus <= 8'h00  ;
        end
      end        
    end


    //-------------------------------------------------------------------------
    // Message Bus Monitor (Req 1.2A, 1.2B, 1.2C, 1.2D and 1.2E)
    //    see PIPE v6.0 | 6.1.4 - Message Bus Interface (p 72)
    //-------------------------------------------------------------------------
    localparam IDLE        = 3'b000  ;
    localparam P2M_CYCLE_1 = 3'b001  ;
    localparam P2M_CYCLE_2 = 3'b010  ;
    localparam P2M_CYCLE_3 = 3'b011  ;
    localparam M2P_CYCLE_1 = 3'b100  ;
    localparam M2P_CYCLE_2 = 3'b101  ;
    localparam M2P_CYCLE_3 = 3'b110  ;

    reg   [2:0]   nextstate;
    reg   [2:0]   currentstate;

    // Current State Update
    always @(posedge pclk or negedge arstn) begin
      if(!arstn)  currentstate <= IDLE;
      else        currentstate <= nextstate;
    end

    // Conditional State-Transition
    always @(*) begin
      case (currentstate)
        IDLE: begin
          if(pl_p2m_msgbus_x1[7:4] == WRITE_ACK) // P2M Single Cycle
            nextstate = P2M_CYCLE_1;
          else if(pl_p2m_msgbus_x1[7:4] == READ || pl_p2m_msgbus_x1[7:4] == READ_C0MPL) // P2M Two Cycles
            nextstate = P2M_CYCLE_2;
          else if(pl_p2m_msgbus_x1[7:4] == WRITE_COMMITED || pl_p2m_msgbus_x1[7:4] == WRITE_UNCOMMITED) // P2M Three Cycles
            nextstate = P2M_CYCLE_3;
          else if(pl_m2p_msgbus_x1[7:4] == WRITE_ACK) // M2P Single Cycle
            nextstate = M2P_CYCLE_1;
          else if(pl_m2p_msgbus_x1[7:4] == READ || pl_m2p_msgbus_x1[7:4] == READ_C0MPL) // M2P Two Cycles
            nextstate = M2P_CYCLE_2;
          else if(pl_m2p_msgbus_x1[7:4] == WRITE_COMMITED || pl_m2p_msgbus_x1[7:4] == WRITE_UNCOMMITED) // M2P Three Cycles
            nextstate = M2P_CYCLE_3;
          else
            nextstate = IDLE;
        end

        P2M_CYCLE_1: begin
          if(pl_p2m_msgbus_x1[7:4] == WRITE_ACK) // P2M Single Cycle
            nextstate = P2M_CYCLE_1;
          else if(pl_p2m_msgbus_x1[7:4] == READ || pl_p2m_msgbus_x1[7:4] == READ_C0MPL) // P2M Two Cycles
            nextstate = P2M_CYCLE_2;
          else if(pl_p2m_msgbus_x1[7:4] == WRITE_COMMITED || pl_p2m_msgbus_x1[7:4] == WRITE_UNCOMMITED) // P2M Three Cycles
            nextstate = P2M_CYCLE_3;
          else if(pl_m2p_msgbus_x1[7:4] == WRITE_ACK) // M2P Single Cycle
            nextstate = M2P_CYCLE_1;
          else if(pl_m2p_msgbus_x1[7:4] == READ || pl_m2p_msgbus_x1[7:4] == READ_C0MPL) // M2P Two Cycles
            nextstate = M2P_CYCLE_2;
          else if(pl_m2p_msgbus_x1[7:4] == WRITE_COMMITED || pl_m2p_msgbus_x1[7:4] == WRITE_UNCOMMITED) // M2P Three Cycles
            nextstate = M2P_CYCLE_3;
          else
            nextstate = IDLE;
        end
        P2M_CYCLE_2: begin
          if(!srstn)
            nextstate = IDLE;
          else
            nextstate = P2M_CYCLE_1;
        end
        P2M_CYCLE_3: begin
          if(!srstn)
            nextstate = IDLE;
          else
            nextstate = P2M_CYCLE_2;
        end
        
        M2P_CYCLE_1: begin
          if(pl_p2m_msgbus_x1[7:4] == WRITE_ACK) // P2M Single Cycle
            nextstate = P2M_CYCLE_1;
          else if(pl_p2m_msgbus_x1[7:4] == READ || pl_p2m_msgbus_x1[7:4] == READ_C0MPL) // P2M Two Cycles
            nextstate = P2M_CYCLE_2;
          else if(pl_p2m_msgbus_x1[7:4] == WRITE_COMMITED || pl_p2m_msgbus_x1[7:4] == WRITE_UNCOMMITED) // P2M Three Cycles
            nextstate = P2M_CYCLE_3;
          else if(pl_m2p_msgbus_x1[7:4] == WRITE_ACK) // M2P Single Cycle
            nextstate = M2P_CYCLE_1;
          else if(pl_m2p_msgbus_x1[7:4] == READ || pl_m2p_msgbus_x1[7:4] == READ_C0MPL) // M2P Two Cycles
            nextstate = M2P_CYCLE_2;
          else if(pl_m2p_msgbus_x1[7:4] == WRITE_COMMITED || pl_m2p_msgbus_x1[7:4] == WRITE_UNCOMMITED) // M2P Three Cycles
            nextstate = M2P_CYCLE_3;
          else
            nextstate = IDLE;
        end
        M2P_CYCLE_2: begin
          if(!srstn)
            nextstate = IDLE;
          else
            nextstate = M2P_CYCLE_1;
        end
        M2P_CYCLE_3: begin
          if(!srstn)
            nextstate = IDLE;
          else
            nextstate = M2P_CYCLE_2;
        end

        default:
          nextstate = IDLE;
      endcase
      
    end

    // Output Logic
    //  . Packet Register
    always @(*)
    begin
        if (!arstn) begin
          pkt_msgbus  =  {(PKT_WIDTH) {1'b0}} ;
          trig_write  =  1'b0  ;
        end
        else begin
          if (currentstate == IDLE) begin
            pkt_msgbus  =  {(PKT_WIDTH) {1'b0}} ;
            trig_write  =  1'b0  ;
          end
          else if (currentstate == P2M_CYCLE_1 || currentstate == P2M_CYCLE_2 || currentstate == P2M_CYCLE_3) begin
            pkt_msgbus  =  {global_time, 1'b1, reg_in1_p2m_msgbus} ; // Create packet -> | TIME | P2M bit | MSG |
            // pkt_msgbus  =  {global_time, 1'b1, reg_in2_p2m_msgbus} ; // Create packet -> | TIME | P2M bit | MSG |       -- testbench OK, refDesign KO
            trig_write  =  1'b1  ;
          end
          else if (currentstate == M2P_CYCLE_1 || currentstate == M2P_CYCLE_2 || currentstate == M2P_CYCLE_3) begin
            pkt_msgbus  =  {global_time, 1'b0, reg_in1_m2p_msgbus} ; // Create packet -> | TIME | M2P bit | MSG |
            // pkt_msgbus  =  {global_time, 1'b0, reg_in2_p2m_msgbus} ; // Create packet -> | TIME | P2M bit | MSG |       -- testbench OK, refDesign KO
            trig_write  =  1'b1  ;
          end
        end
    end


    //-------------------------------------------------------------------------
    // Write Memory Controller
    //-------------------------------------------------------------------------

    write_memory_controller #(
      .PKT_WIDTH              (PKT_WIDTH                ),
      .MEM_ADDR_WIDTH         (MEM_ADDR_WIDTH           ),
      .MEM_DATA_WIDTH         (MEM_DATA_WIDTH           )
      //-------------------------------------------------------------------------
    ) write_memory_controller_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Write enable and data packet to send
      .trig_write             (trig_write               ),
      .packet_in              (pkt_msgbus               ),
      //  . Write Port
      .scram_wren             (ram_wren                 ),
      .scram_wrbe             (ram_wrbe                 ),
      .scram_wraddr           (ram_wraddr               ),
      .scram_wrdata           (ram_wrdata               )
      //-------------------------------------------------------------------------
    );



endmodule