//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Training Sequence Ordered Set Detector. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Output assignations
//                (3) TS Detector
//                (4) TS Field sampling
//
// Dependency : 
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module eieos_detector #(
    //-------------------------------------------------------------------------------
    //  Shift Registers Parameters
    parameter SHIFT_DATA_WIDTH  = 'd8 ,
    parameter SHIFT_DATAK_WIDTH = 'd1 ,
    parameter SHIFT_DATA_DEPTH  = 'd16
    //-------------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------------
    //  Enable, Resets and Clock
    input  wire                                                en                   ,
    input  wire                                                arstn                , //  Asynchronous Reset
    input  wire                                                srstn                , //  Synchronous Reset
    input  wire                                                pclk                 ,
    //-------------------------------------------------------------------------------
    //  PIPE Signals
    input  wire [3                                       -1:0] pl_rate              ,
    //-------------------------------------------------------------------------------
    //  Parallel Data and DataK
    input  wire [SHIFT_DATA_DEPTH * SHIFT_DATA_WIDTH     -1:0] paral_data           ,
    input  wire [SHIFT_DATA_DEPTH * SHIFT_DATAK_WIDTH    -1:0] paral_datak          ,

    //-------------------------------------------------------------------------------
    //  Detection Signal
    output wire                                                detect_eieos            
    //-------------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    // Detection Flags
    reg                                                 reg_detect_eieos      ;


    //-------------------------------------------------------------------------
    // Output assignations
    //-------------------------------------------------------------------------
    assign detect_eieos = reg_detect_eieos;

    //------------------------------------------------------------------------------------------------------------------
    // EIEOS Detectors
    //  see PCI Express® Base Specification Revision 5.0 Version 1.0 | 4.2.4.3 Electrical Idle Sequences (EIOS) (p 301)
    //------------------------------------------------------------------------------------------------------------------
    always @(posedge pclk or negedge arstn) begin
      if (!arstn) begin
        reg_detect_eieos <= 1'b0;
      end
      else begin
        if(!srstn) begin
          reg_detect_eieos <= 1'b0;
        end
        else if (en) begin
          case (pl_rate)
            `LINK_RATE_5GT: begin
              if ({paral_datak[0], paral_data[7:0]} == `K_COM && // Symbol 0
                  paral_datak[14:1] == {(14){1'b1}} && paral_data[119:8] == {(14){`B_EIE}}  && // Symbol 1-14
                  paral_datak[15] == 1'b0 && paral_data[127:120] == `D_TS1 // Symbol 1-15
              ) begin
                // $display ("[pmc] time=%0t \t EIEOS Detect (5 GT/s)", $time);
                reg_detect_eieos <= 1'b1;
              end
              else reg_detect_eieos <= 1'b0;
            end // LINK_RATE_5GT
            `LINK_RATE_8GT: begin
              if ({paral_data[15:0] , paral_data[47:32], paral_data[79:64], paral_data[111:96]}  == {(8){8'h00}} && // Symbol 0,1,4,5,8,9,12,13
                  {paral_data[31:16], paral_data[63:48], paral_data[95:80], paral_data[127:112]} == {(8){8'hFF}}    // Symbol 2,3,6,7,10,11,14,15
              ) begin
                //$display ("[pmc] time=%0t \t EIEOS Detect (8 GT/s)", $time);
                reg_detect_eieos <= 1'b1;
              end
              else reg_detect_eieos <= 1'b0;
            end // LINK_RATE_8GT
            `LINK_RATE_16GT: begin
              if ({paral_data[31:0] , paral_data[95:64]}  == {(8){8'h00}} && // Symbol 0,1,2,3,8,9,10,11
                  {paral_data[63:32], paral_data[127:96]} == {(8){8'hFF}}    // Symbol 4,5,6,7,12,13,14,15
              ) begin
                //$display ("[pmc] time=%0t \t EIEOS Detect (16 GT/s)", $time);
                reg_detect_eieos <= 1'b1;
              end
              else reg_detect_eieos <= 1'b0;
            end // LINK_RATE_16GT
            default: reg_detect_eieos <= 1'b0;
          endcase
        end
      end
    end
   


endmodule