//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Data Decoder for Single lane configuration. Good luck :)
//
//              This module is composed of:
//                (1) Local declarations
//                (2) Receiver Detection
//                (3) DATA and DATAK Shift Registers
//                (4) TS1 and TS2 Detectors
//                (5) EIOS Detectors
//                (6) EIEOS Detectors
//                (7) SKPOS Detectors
//
// Dependency : pipemc_shift_register, tsos_detector, eios_detector, eieos_detector,
//            skpos_detector
//-----------------------------------------------------------------------------

`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module data_decoder_x1 #(
    //-------------------------------------------------------------------------------
    //  PIPE Parameters
    parameter G_PIPE_WIDTH      = 'h88422      // Gen 4
    //-------------------------------------------------------------------------------
  )(
    //-------------------------------------------------------------------------------
    //  Clocks, EN and Resets
    input  wire                                                en                   ,
    input  wire                                                arstn                , //  Asynchronous Reset
    input  wire                                                srstn                , //  Synchronous Reset
    input  wire                                                pclk                 ,

    //-------------------------------------------------------------------------------
    //  PIPE Signals
    //  . Signals all lanes
    input  wire [4                                        -1:0] pl_powerdown        ,
    input  wire [3                                        -1:0] pl_rate             ,
    input  wire [3                                        -1:0] pl_width            ,

    //  . Signals per lane - TX
    input  wire                                                 pl_txdetectrx       ,
    input  wire [(G_PIPE_WIDTH>>16)*8                     -1:0] pl_txdata           ,
    input  wire [(G_PIPE_WIDTH>>16)                       -1:0] pl_txdatak          ,
    input  wire                                                 pl_txdatavalid      ,

    //  . Signals per lane - RX
    input  wire                                                 pl_phystatus        ,
    input  wire [3                                        -1:0] pl_rxstatus         ,
    input  wire [(G_PIPE_WIDTH>>16)*8                     -1:0] pl_rxdata           ,
    input  wire [(G_PIPE_WIDTH>>16)                       -1:0] pl_rxdatak          ,
    input  wire                                                 pl_rxdatavalid      ,

    //-------------------------------------------------------------------------------
    //  PIPE State Logic signals
    input  wire                                                 data_encoding       ,
    input  wire                                                 detect_txn_ltssm    ,

    //-------------------------------------------------------------------------------
    //  Detection Flags
    output wire                                                 detect_rx           ,
    output wire [2                                        -1:0] detect_ts1          , // | 0 TX | 1 RX |
    output wire [2                                        -1:0] detect_ts2          , // | 0 TX | 1 RX |
    output wire [2                                        -1:0] detect_eios         , // | 0 TX | 1 RX |
    output wire [2                                        -1:0] detect_eieos        , // | 0 TX | 1 RX |
    output wire [2                                        -1:0] detect_skpos        , // | 0 TX | 1 RX |
    output wire [2                                        -1:0] detect_kskpos       , // | 0 TX | 1 RX |

    output wire [4                                        -1:0] detect_chg_nlink    , // | 0 TS1 TX | 1 TS2 TX | 2 TS1 RX | 3 TS RX |
    output wire [4                                        -1:0] detect_chg_nlane    , // | 0 TS1 TX | 1 TS2 TX | 2 TS1 RX | 3 TS RX |
    output wire [4                                        -1:0] detect_chg_idr      , // | 0 TS1 TX | 1 TS2 TX | 2 TS1 RX | 3 TS RX |
    output wire [4                                        -1:0] detect_chg_psets    , // | 0 TS1 TX | 1 TS2 TX | 2 TS1 RX | 3 TS RX |

    //-------------------------------------------------------------------------------
    //  TS Fields
    output wire [8*4                                      -1:0] data_rate_ids         // | 0... DRI TX TS1 ...7 | 8... DRI TX TS2 ...15 | 16... DRI RX TS1 ...23 |24... DRI RX TS2 ...31 | 
    //-------------------------------------------------------------------------------
  );

    //-------------------------------------------------------------------------
    // Local declarations 
    //-------------------------------------------------------------------------
    // TX Shift Registers
    localparam SHIFT_DATA_WIDTH   = 'd8   ;
    localparam SHIFT_DATA_DEPTH   = 'd16  ;
    localparam SHIFT_DATAK_WIDTH  = 'd1   ;
    localparam SHIFT_DATAK_DEPTH  = 'd16  ;

    wire                                                en_datak              ;
    wire [SHIFT_DATA_DEPTH * SHIFT_DATA_WIDTH     -1:0] paral_txdata          ;
    wire [SHIFT_DATAK_DEPTH * SHIFT_DATAK_WIDTH   -1:0] paral_txdatak         ;
    wire [SHIFT_DATA_DEPTH * SHIFT_DATA_WIDTH     -1:0] paral_rxdata          ;
    wire [SHIFT_DATAK_DEPTH * SHIFT_DATAK_WIDTH   -1:0] paral_rxdatak         ;

    // TX Shift Registers

    // DataK Sampling Enable
    assign en_datak = en && ~data_encoding;


    //------------------------------------------------------------------------------------------------------------------
    // Receiver Detection
    //  see PIPE v6.0 | 8.7 Receiver Detection (p 131)
    //------------------------------------------------------------------------------------------------------------------
    assign detect_rx = (pl_txdetectrx && pl_phystatus && (pl_powerdown == `POWER_STATE_P1) && (pl_rxstatus == `RXST_RX_DET) );


    //------------------------------------------------------------------------------------------------------------------
    // DATA and DATAK Shift Registers
    //------------------------------------------------------------------------------------------------------------------   
    pipemc_shift_register #(
      //-------------------------------------------------------------------------
      //  . PIPE Parameters
      .G_PIPE_WIDTH           (G_PIPE_WIDTH         ),
      //  . Shift Registers Parameters
      .SHIFT_WIDTH            (SHIFT_DATA_WIDTH     ),
      .SHIFT_DEPTH            (SHIFT_DATA_DEPTH     )
      //-------------------------------------------------------------------------
    ) pipemc_shift_register_txdata_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . PIPE Signals
      .pl_width               (pl_width                 ),
      //  . Input Data
      .shift_data_in          (pl_txdata                ),
      .shift_datavalid        (pl_txdatavalid           ),
      //  . Parallel Data Output
      .shift_data_out         (paral_txdata             )
      //-------------------------------------------------------------------------
    );

    pipemc_shift_register #(
      //-------------------------------------------------------------------------
      //  . PIPE Parameters
      .G_PIPE_WIDTH           (G_PIPE_WIDTH          ),
      //  . Shift Registers Parameters
      .SHIFT_WIDTH            (SHIFT_DATAK_WIDTH     ),
      .SHIFT_DEPTH            (SHIFT_DATAK_DEPTH     )
      //-------------------------------------------------------------------------
    ) pipemc_shift_register_txdatak_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (en_datak                 ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . PIPE Signals
      .pl_width               (pl_width                 ),
      //  . Input Data
      .shift_data_in          (pl_txdatak               ),
      .shift_datavalid        (pl_txdatavalid           ),
      //  . Parallel Data Output
      .shift_data_out         (paral_txdatak            )
      //-------------------------------------------------------------------------
    );

    pipemc_shift_register #(
      //-------------------------------------------------------------------------
      //  . PIPE Parameters
      .G_PIPE_WIDTH           (G_PIPE_WIDTH         ),
      //  . Shift Registers Parameters
      .SHIFT_WIDTH            (SHIFT_DATA_WIDTH     ),
      .SHIFT_DEPTH            (SHIFT_DATA_DEPTH     )
      //-------------------------------------------------------------------------
    ) pipemc_shift_register_rxdata_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . PIPE Signals
      .pl_width               (pl_width                 ),
      //  . Input Data
      .shift_data_in          (pl_rxdata                ),
      .shift_datavalid        (pl_rxdatavalid           ),
      //  . Parallel Data Output
      .shift_data_out         (paral_rxdata             )
      //-------------------------------------------------------------------------
    );

    pipemc_shift_register #(
      //-------------------------------------------------------------------------
      //  . PIPE Parameters
      .G_PIPE_WIDTH           (G_PIPE_WIDTH          ),
      //  . Shift Registers Parameters
      .SHIFT_WIDTH            (SHIFT_DATAK_WIDTH     ),
      .SHIFT_DEPTH            (SHIFT_DATAK_DEPTH     )
      //-------------------------------------------------------------------------
    ) pipemc_shift_register_rxdatak_inst (
      //-------------------------------------------------------------------------
      //  . Clocks, EN and Resets
      .en                     (en_datak                 ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . PIPE Signals
      .pl_width               (pl_width                 ),
      //  . Input Data
      .shift_data_in          (pl_rxdatak               ),
      .shift_datavalid        (pl_rxdatavalid           ),
      //  . Parallel Data Output
      .shift_data_out         (paral_rxdatak            )
      //-------------------------------------------------------------------------
    );




    //------------------------------------------------------------------------------------------------------------------
    // TS1 and TS2 Detectors
    //  see PCI Express® Base Specification Revision 5.0 Version 1.0 | 4.2.4.1 Training Sequences (p 228)
    //------------------------------------------------------------------------------------------------------------------
    tsos_detector #( //  TS1 TX
      //-------------------------------------------------------------------------
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH          ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH         ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH          ),
      //  . TS Symbolss
      .B_TS                   (`B_TS1     ),
      .D_TS                   (`D_TS1     )
      //-------------------------------------------------------------------------
    ) tsos_detector_ts1_tx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Data Encoding
      .data_encoding          (data_encoding            ),
      //  . Parallel Data and DataK
      .paral_data             (paral_txdata             ),
      .paral_datak            (paral_txdatak            ),
      //  . Detection Signals
      .detect_ts              (detect_ts1       [0]     ),
      .detect_chg_nlink       (detect_chg_nlink [0]     ),
      .detect_chg_nlane       (detect_chg_nlane [0]     ),
      .detect_chg_idr         (detect_chg_idr   [0]     ),
      .detect_chg_psets       (detect_chg_psets [0]     ),
      //  . TS Fields
      .nlink                  (           ),
      .nlane                  (           ),
      .data_rate_id           (data_rate_ids[7:0]       ),
      .presets                (           )
      //-------------------------------------------------------------------------
    );
    
    tsos_detector #(  // TS2 TX
      //-------------------------------------------------------------------------
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH          ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH         ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH          ),
      //  . TS Symbolss
      .B_TS                   (`B_TS2     ),
      .D_TS                   (`D_TS2     )
      //-------------------------------------------------------------------------
    ) tsos_detector_ts2_tx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Data Encoding
      .data_encoding          (data_encoding            ),
      //  . Parallel Data and DataK
      .paral_data             (paral_txdata             ),
      .paral_datak            (paral_txdatak            ),
      //  . Detection Signals
      .detect_ts              (detect_ts2       [0]     ),
      .detect_chg_nlink       (detect_chg_nlink [1]     ),
      .detect_chg_nlane       (detect_chg_nlane [1]     ),
      .detect_chg_idr         (detect_chg_idr   [1]     ),
      .detect_chg_psets       (detect_chg_psets [1]     ),
      //  . TS Fields
      .nlink                  (           ),
      .nlane                  (           ),
      .data_rate_id           (data_rate_ids[15:8]      ),
      .presets                (           )
      //-------------------------------------------------------------------------
    );

    tsos_detector #(  // TS1 RX
      //-------------------------------------------------------------------------
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH         ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH        ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH         ),
      //  . TS Symbolss
      .B_TS                   (`B_TS1                   ),
      .D_TS                   (`D_TS1                   )
      //-------------------------------------------------------------------------
    ) tsos_detector_ts1_rx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Data Encoding
      .data_encoding          (data_encoding            ),
      //  . Parallel Data and DataK
      .paral_data             (paral_rxdata             ),
      .paral_datak            (paral_rxdatak            ),
      //  . Detection Signals
      .detect_ts              (detect_ts1       [1]     ),
      .detect_chg_nlink       (detect_chg_nlink [2]     ),
      .detect_chg_nlane       (detect_chg_nlane [2]     ),
      .detect_chg_idr         (detect_chg_idr   [2]     ),
      .detect_chg_psets       (detect_chg_psets [2]     ),
      //  . TS Fields
      .nlink                  (           ),
      .nlane                  (           ),
      .data_rate_id           (data_rate_ids[23:16]     ),
      .presets                (           )
      //-------------------------------------------------------------------------
    );

    tsos_detector #(  // TS2 RX
      //-------------------------------------------------------------------------
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH         ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH        ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH         ),
      //  . TS Symbolss
      .B_TS                   (`B_TS2                   ),
      .D_TS                   (`D_TS2                   )
      //-------------------------------------------------------------------------
    ) tsos_detector_ts2_rx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Data Encoding
      .data_encoding          (data_encoding            ),
      //  . Parallel Data and DataK
      .paral_data             (paral_rxdata             ),
      .paral_datak            (paral_rxdatak            ),
      //  . Detection Signals
      .detect_ts              (detect_ts2       [1]     ),
      .detect_chg_nlink       (detect_chg_nlink [3]     ),
      .detect_chg_nlane       (detect_chg_nlane [3]     ),
      .detect_chg_idr         (detect_chg_idr   [3]     ),
      .detect_chg_psets       (detect_chg_psets [3]     ),
      //  . TS Fields
      .nlink                  (           ),
      .nlane                  (           ),
      .data_rate_id           (data_rate_ids[31:24]     ),
      .presets                (           )
      //-------------------------------------------------------------------------
    );

    //------------------------------------------------------------------------------------------------------------------
    // EIOS Detectors
    //  see PCI Express® Base Specification Revision 5.0 Version 1.0 | 4.2.4.3 Electrical Idle Sequences (EIOS) (p 301)
    //------------------------------------------------------------------------------------------------------------------
    eios_detector #(  // TX
      //-------------------------------------------------------------------------
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH         ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH        ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH         )
      //-------------------------------------------------------------------------
    ) eios_detector_tx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Data Encoding
      .data_encoding          (data_encoding            ),
      //  . Parallel Data and DataK
      .paral_data             (paral_txdata             ),
      .paral_datak            (paral_txdatak            ),
      //  . Detection Signals
      .detect_eios            (detect_eios [0]          )
      //-------------------------------------------------------------------------
    );

    eios_detector #(  // RX
      //-------------------------------------------------------------------------
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH         ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH        ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH         )
      //-------------------------------------------------------------------------
    ) eios_detector_rx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . Data Encoding
      .data_encoding          (data_encoding            ),
      //  . Parallel Data and DataK
      .paral_data             (paral_rxdata             ),
      .paral_datak            (paral_rxdatak            ),
      //  . Detection Signals
      .detect_eios            (detect_eios [1]          )
      //-------------------------------------------------------------------------
    );


    //------------------------------------------------------------------------------------------------------------------
    // EIEOS Detectors
    //  see PCI Express® Base Specification Revision 5.0 Version 1.0 | 4.2.4.3 Electrical Idle Sequences (EIOS) (p 301)
    //------------------------------------------------------------------------------------------------------------------
    eieos_detector #(  // TX
      //-------------------------------------------------------------------------
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH         ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH        ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH         )
      //-------------------------------------------------------------------------
    ) eieos_detector_tx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . PIPE Signals
      .pl_rate                (pl_rate                  ),
      //  . Parallel Data and DataK
      .paral_data             (paral_txdata             ),
      .paral_datak            (paral_txdatak            ),
      //  . Detection Signals
      .detect_eieos           (detect_eieos [0]         )
      //-------------------------------------------------------------------------
    );

    eieos_detector #(  // RX
      //-------------------------------------------------------------------------
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH         ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH        ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH         )
      //-------------------------------------------------------------------------
    ) eieos_detector_rx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . PIPE Signals
      .pl_rate                (pl_rate                  ),
      //  . Parallel Data and DataK
      .paral_data             (paral_rxdata             ),
      .paral_datak            (paral_rxdatak            ),
      //  . Detection Signals
      .detect_eieos           (detect_eieos [1]         )
      //-------------------------------------------------------------------------
    );



    //------------------------------------------------------------------------------------------------------------------
    // SKPOS Detectors
    //  see PCI Express® Base Specification Revision 5.0 Version 1.0 | 4.2.7 Clock Tolerance Compensation (p 381)
    //------------------------------------------------------------------------------------------------------------------
    skpos_detector #(  // TX
      //-------------------------------------------------------------------------
      //  . SKPOS Detector Parameters
      .TX_RXn_PATTERN          (1'b1                    ),  // If 1, it looks for transmitted SKPOSs, otherwise it detects received SKPOSs
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH         ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH        ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH         )
      //-------------------------------------------------------------------------
    ) skpos_detector_tx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),  
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . PIPE Signals
      .pl_rate                (pl_rate                  ),
      //  . Data Encoding
      .data_encoding          (data_encoding            ),
      //  . Parallel Data and DataK
      .paral_data             (paral_txdata             ),
      .paral_datak            (paral_txdatak            ),
      //  . Detection Signals
      .detect_skpos           (detect_skpos [0]         ),
      .detect_kskpos          (detect_kskpos[0]         )
      //-------------------------------------------------------------------------
    );

    skpos_detector #(  // RX
      //-------------------------------------------------------------------------
      //  . SKPOS Detector Parameters
      .TX_RXn_PATTERN          (1'b0                    ), // If 1, it looks for transmitted SKPOSs, otherwise it detects received SKPOSs
      //  . Shift Registers Parameters
      .SHIFT_DATA_WIDTH       (SHIFT_DATA_WIDTH         ),
      .SHIFT_DATAK_WIDTH      (SHIFT_DATAK_WIDTH        ),
      .SHIFT_DATA_DEPTH       (SHIFT_DATA_DEPTH         )
      //-------------------------------------------------------------------------
    ) skpos_detector_rx_inst (
      //-------------------------------------------------------------------------
      //  . Enable, RST and Clock
      .en                     (en                       ),
      .arstn                  (arstn                    ),
      .srstn                  (srstn                    ),
      .pclk                   (pclk                     ),
      //  . PIPE Signals
      .pl_rate                (pl_rate                  ),
      //  . Data Encoding
      .data_encoding          (data_encoding            ),
      //  . Parallel Data and DataK
      .paral_data             (paral_rxdata             ),
      .paral_datak            (paral_rxdatak            ),
      //  . Detection Signals
      .detect_skpos           (detect_skpos [0]         ),
      .detect_kskpos          (detect_kskpos[0]         )
      //-------------------------------------------------------------------------
    );





endmodule