//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: 
//
// Dependency : -
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"
`include "pipemc_config_h.v"

module uart_backdoor
 #(
    //-------------------------------------------------------------------------
    // Constants
    parameter   AXI_ADDR_WIDTH            =    64,
    parameter   AXI_DATA_WIDTH            =   512,
    parameter   RAM_1_N                   =     1,  // #             RAMs with ADDR WIDTH = MEM_ADDR_WIDTH_MAX
    parameter   RAM_2_N                   =     0,  // # Pairs    of RAMs with ADDR WIDTH = MEM_ADDR_WIDTH_MAX - 1
    parameter   RAM_4_N                   =     0,  // # Quartets of RAMs with ADDR WIDTH = MEM_ADDR_WIDTH_MAX - 2
    parameter   RAM_8_N                   =     0,  // # Octets   of RAMs with ADDR WIDTH = MEM_ADDR_WIDTH_MAX - 4
    parameter   MEM_ADDR_WIDTH_MAX        =     8,   // Max address width amongs the connected RAMs

    parameter G_PIPE_INTF         = 'd1       ,
    parameter G_PCIE_NUM_LANES    = 'd1      ,
    parameter G_PIPE_WIDTH        = 'h88422       // Gen 4
    //-------------------------------------------------------------------------
  ) (
    //-------------------------------------------------------------------------
    // Enable, Clock and Reset
    input wire                                          en                    ,
    input wire                                          arstn                 ,
    input wire                                          srstn                 ,
    input wire                                          clk                   ,

    //-------------------------------------------------------------------------
    // 
    input wire                                          pl_pclk               ,
    
    //-------------------------------------------------------------------------
    // UART Interface
    //  . UART RX
    input wire                                          rx_datas_rdy          ,
    input wire [8                                 -1:0] rx_datas              ,
    input wire                                          parity_error          ,
    input wire                                          start_bit_error       ,
    
    //  . UART TX
    input wire                                          tx_busy,
    output wire                                          tx_datas_start        ,
    output wire [8                                 -1:0] tx_datas              
    //-------------------------------------------------------------------------
  );

  // AXI4 Master 1 Interface
  //  . Read Address Channel
  wire [4                                -1:0] axi4_mst1_arid        ;
  wire [64                               -1:0] axi4_mst1_araddr      ;
  wire [4                                -1:0] axi4_mst1_arregion    ;
  wire [8                                -1:0] axi4_mst1_arlen       ;
  wire [3                                -1:0] axi4_mst1_arsize      ;
  wire [2                                -1:0] axi4_mst1_arburst     ;
  wire                                         axi4_mst1_arlock      ;
  wire [4                                -1:0] axi4_mst1_arcache     ;
  wire [3                                -1:0] axi4_mst1_arprot      ;
  wire [4                                -1:0] axi4_mst1_arqos       ;
  wire                                         axi4_mst1_arvalid     ;
  wire                                         axi4_mst1_arready     ;
  //  . Read Data Channel
  wire [4                               -1:0] axi4_mst1_rid         ;
  wire [AXI_DATA_WIDTH                 -1:0] axi4_mst1_rdata       ;
  wire [2                               -1:0] axi4_mst1_rresp       ;
  wire                                        axi4_mst1_rlast       ;
  wire                                        axi4_mst1_rvalid      ;
  wire                                         axi4_mst1_rready     ;

  //-------------------------------------------------------------------------------
  //AXI Backdoor RAM Ports
  localparam  N_TOT_RAM = (`NRAM_W0 + 2 * `NRAM_W1 + 4 * `NRAM_W1 + 8 * `NRAM_W3) + G_PCIE_NUM_LANES - 2; // It adds the msgbus RAMs as well

  wire [N_TOT_RAM                                   -1:0] shared_ram_rden         ;
  wire [`MEM_ADDR_WIDTH_MAX                         -1:0] shared_ram_rdaddr       ;
  wire [N_TOT_RAM * `MEM_DATAPATH_WIDTH             -1:0] shared_ram_rddata       ;


  //---------------------------------------------------------------------------
  // UART Controller
  //---------------------------------------------------------------------------
  uart_axi_ps #(
        //-------------------------------------------------------------------------
        .G_ADDR_WIDTH                     ('d64      ),
        .G_DATA_WIDTH                     (AXI_DATA_WIDTH        )
        //-------------------------------------------------------------------------
  ) uart_axi_ps_inst (
        //-------------------------------------------------------------------------
        // Enable, Clock and Reset
        .en                               (en                 ),
        .arstn                            (arstn              ),
        .clk                              (clk             ),
        //-------------------------------------------------------------------------
        // UART Interface
        //  . UART RX
        .rx_datas_rdy                     (rx_datas_rdy           ),
        .rx_datas                         (rx_datas               ),
        .parity_error                     (parity_error           ),
        .start_bit_error                  (start_bit_error        ),
        //  . UART TX
        .tx_busy                          (tx_busy                ),
        .tx_datas_start                   (tx_datas_start         ),
        .tx_datas                         (tx_datas               ),
        //  . Read Address Channel     
        .arid                             (axi4_mst1_arid         ),
        .araddr                           (axi4_mst1_araddr       ),
        .arregion                         (axi4_mst1_arregion     ),
        .arlen                            (axi4_mst1_arlen        ),
        .arsize                           (axi4_mst1_arsize       ),
        .arburst                          (axi4_mst1_arburst      ),
        .arlock                           (axi4_mst1_arlock       ),
        .arcache                          (axi4_mst1_arcache      ),
        .arprot                           (axi4_mst1_arprot       ),
        .arqos                            (axi4_mst1_arqos        ),
        .arvalid                          (axi4_mst1_arvalid      ),
        .arready                          (axi4_mst1_arready      ),
        //  . Read Data Channel
        .rid                              (axi4_mst1_rid          ),
        .rdata                            (axi4_mst1_rdata        ),
        .rresp                            (axi4_mst1_rresp        ),
        .rlast                            (axi4_mst1_rlast        ),
        .rvalid                           (axi4_mst1_rvalid       ),
        .rready                           (axi4_mst1_rready       )
        //-------------------------------------------------------------------------
  );

    //---------------------------------------------------------------------------
    // AXI Backdoor Interface
    //---------------------------------------------------------------------------
    axi_ram_backdoor #(
      //-------------------------------------------------------------------------
      .AXI_ADDR_WIDTH                   (`AXI_ADDR_WIDTH        ),
      .AXI_DATA_WIDTH                   (`MEM_DATAPATH_WIDTH    ),
      .RAM_1_N                          (`NRAM_W0 + G_PCIE_NUM_LANES - 2), // It considers the msgbus RAMs as well
      .RAM_2_N                          (`NRAM_W1               ),
      .RAM_4_N                          (`NRAM_W2               ),
      .RAM_8_N                          (`NRAM_W3               ),
      .MEM_ADDR_WIDTH_MAX               (`MEM_ADDR_WIDTH_MAX    )
      //-------------------------------------------------------------------------
    ) axi_ram_backdoor_inst (
      //-------------------------------------------------------------------------
      // AXI4 Slave Interface
      //  . Clock and Resets
      .arstn                            (arstn              ),
      .srstn                            (srstn              ),
      .clk                              (clk             ),
      //  . Read Address Channel     
      .arid                             (axi4_mst1_arid         ),
      .araddr                           (axi4_mst1_araddr       ),
      .arregion                         (axi4_mst1_arregion     ),
      .arlen                            (axi4_mst1_arlen        ),
      .arsize                           (axi4_mst1_arsize       ),
      .arburst                          (axi4_mst1_arburst      ),
      .arlock                           (axi4_mst1_arlock       ),
      .arcache                          (axi4_mst1_arcache      ),
      .arprot                           (axi4_mst1_arprot       ),
      .arqos                            (axi4_mst1_arqos        ),
      .arvalid                          (axi4_mst1_arvalid      ),
      .arready                          (axi4_mst1_arready      ),
      //  . Read Data Channel
      .rid                              (axi4_mst1_rid          ),
      .rdata                            (axi4_mst1_rdata        ),
      .rresp                            (axi4_mst1_rresp        ),
      .rlast                            (axi4_mst1_rlast        ),
      .rvalid                           (axi4_mst1_rvalid       ),
      .rready                           (axi4_mst1_rready       ),
      //-------------------------------------------------------------------------
      //RAM Ports
      .shared_ram_rden                  (shared_ram_rden        ),
      .shared_ram_rdaddr                (shared_ram_rdaddr      ),
      .shared_ram_rddata                (shared_ram_rddata      )
      //-------------------------------------------------------------------------
    );

    //---------------------------------------------------------------------------
    // PIPE Monitor Checker Registers
    //---------------------------------------------------------------------------
    pipemc_reg #(
      //-------------------------------------------------------------------------
      // XR5AXI Layer Constants
      .ADDR_WIDTH              (`MEM_ADDR_WIDTH_REG                 ),
      .DATA_WIDTH              (`MEM_DATAPATH_WIDTH ),
      //-------------------------------------------------------------------------
      // PIPE Configuration
      .G_PIPE_INTF             (G_PIPE_INTF         ),
      .G_PCIE_NUM_LANES        (G_PCIE_NUM_LANES    ),
      .G_PIPE_WIDTH            (G_PIPE_WIDTH        )
      //-------------------------------------------------------------------------
    ) pipemc_reg_inst (
      //-------------------------------------------------------------------------
      // Registers Interface
      //  . Read  Port
      .reg_rdclk              (clk                                       ),
      .reg_rden               (shared_ram_rden[0]                               ),
      .reg_rdaddr             (shared_ram_rdaddr [0     +: 2                  ] ),
      .reg_rddata             (shared_ram_rddata [0     +: `MEM_DATAPATH_WIDTH] ),
      //-------------------------------------------------------------------------
      // RAM Interface
      .reg_wrclk              (pl_pclk                                          ),
      //  . LTSSM RAM
      .dcram_ltssm_wren       (                                 ),
      .dcram_ltssm_wraddr     (                               ),
      //  . Reported Errors RAM
      .dcram_rerr_wren        (                                  ),
      .dcram_rerr_wraddr      (                                ),
      //  . Message Bus RAM
      .dcram_msgbus_wren      (                                ),
      .dcram_msgbus_wraddr    (                              ),
      //  . Data Logic RAM RAM
      .dcram_evntx_wren       (                                 ),
      .dcram_evntx_wraddr     (                               )
      //-------------------------------------------------------------------------
    );



endmodule

`resetall