//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description:  TestBench
//              
//              This module is composed of:
//                (1) Local Declarations
//                (2) Signal generations
//                (3) DUT Instantiation
//
// Dependency: axi_ram_backdoor
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_config_h.v"

module tb ;

// Parameters
localparam  G_PCIE_NUM_LANES  = 'd2  ;
localparam  G_PIPE_WIDTH      = 'h044422  ; // Gen ??
localparam G_DATAPATH_SIZE        = 'd256 ;


//---------------------------------------------------------------------------
// Local Declarations
//---------------------------------------------------------------------------
// Clocks and Reset
reg                                               pmc_en                        ;
reg                                               pmc_arstn                     ;
reg                                               pmc_srstn                     ;
reg                                               pmc_refclk                   ;


//  PIPE Signals | PIPE Configuration
reg [3                                      -1:0] rate                      ;

// UART Interface
//  . UART RX
reg                                         rx_datas_rdy          ;
reg [8                                -1:0] rx_datas              ;
reg                                         parity_error          ;
reg                                         start_bit_error       ;
//  . UART TX
reg                                         tx_busy               ;
reg                                         tx_datas_start        ;
reg [8                                -1:0] tx_datas              ;


// SYNC FSM signals
localparam[8     -1:0]  SYNC_BYTE_0       = 8'hFF,
                        SYNC_BYTE_1       = 8'hFE,
                        SYNC_BYTE_2       = 8'hFD,
                        SYNC_BYTE_3       = 8'hFC;

localparam nbytes = 6;

reg [8*nbytes                                -1:0] stream_rx_datas              ;


//---------------------------------------------------------------------------
// Signal Generation
//---------------------------------------------------------------------------
integer a,b;
initial
begin
  pmc_en          <= 1'b1 ;
  pmc_arstn       <= 1'b0 ;
  pmc_srstn       <= 1'b1 ;
  pmc_refclk     <= 1'b0 ;

  rate        <= 2'd2 ;

  stream_rx_datas <= {8'h03,8'h01, SYNC_BYTE_3, SYNC_BYTE_2, SYNC_BYTE_1, SYNC_BYTE_0};

  #20
  pmc_arstn       <= 1'b1 ;

  
  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps



  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  #240
  $finish;
end


// UART Emulator
localparam  FSM_RST     = 2'h0  ;
localparam  FSM_SEND_RX = 2'h1  ;
localparam  FSM_WAIT_TX = 2'h2  ;

reg [2 -1:0] fsm_uart_state;


always @(posedge pmc_refclk or negedge pmc_arstn) begin
  if (!pmc_arstn) begin
    fsm_uart_state <= FSM_RST;
    rx_datas_rdy <= 1'b0;
    rx_datas <= 8'h0;
    parity_error <= 1'b0;
    start_bit_error <= 1'b0;

    tx_busy <= 1'b0;

    a <= 0;
  end
  else begin
    case (fsm_uart_state)
      FSM_RST: begin
        fsm_uart_state <= FSM_SEND_RX;
    
        rx_datas_rdy <= 1'b0;
        rx_datas <= 8'h0;
        parity_error <= 1'b0;
        start_bit_error <= 1'b0;

        tx_busy <= 1'b0;
        a <= 0;
      end
      FSM_SEND_RX: begin
        if(a < nbytes) begin
          fsm_uart_state <= FSM_WAIT_TX;
    
          rx_datas_rdy <= 1'b1;
          rx_datas <= stream_rx_datas[a*8 +:8];
        end
        else begin
          rx_datas_rdy <= 1'b0;
          rx_datas <= 8'h00;
          #240
          $finish;
        end
        
      end
      FSM_WAIT_TX: begin    
        rx_datas_rdy <= 1'b0;
        rx_datas <= 8'h0;

        if (tx_datas_start || 1'b1) begin
          if (a < nbytes)
            a <= a + 1;

          fsm_uart_state <= FSM_SEND_RX;
        end
        else begin
          fsm_uart_state <= FSM_WAIT_TX;
        end

      end
      default: begin
        fsm_uart_state <= FSM_RST;
    
        rx_datas_rdy <= 1'b0;
        rx_datas <= 8'h0;
        parity_error <= 1'b0;
        start_bit_error <= 1'b0;

        tx_busy <= 1'b0;

        a <= 0;
      end
    endcase
  end
  
end


// Clock
always @(*)
begin
  if      (rate == 2'd3) #1; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #2; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #4; // 125 MHz @  5.0 Gbps
  else                   #8; // 62.5MHz @  2.5 Gbps
  pmc_refclk <= ~pmc_refclk ;
end


//---------------------------------------------------------------------------
// UART Controller
//---------------------------------------------------------------------------
uart_backdoor #(
      //-------------------------------------------------------------------------
      .AXI_ADDR_WIDTH                   (`AXI_ADDR_WIDTH        ),
      .AXI_DATA_WIDTH                   (`MEM_DATAPATH_WIDTH    ),
      .RAM_1_N                          (`NRAM_W0 + G_PCIE_NUM_LANES ), // It considers the msgbus RAMs as well
      .RAM_2_N                          (`NRAM_W1               ),
      .RAM_4_N                          (`NRAM_W2               ),
      .RAM_8_N                          (`NRAM_W3               ),
      .MEM_ADDR_WIDTH_MAX               (`MEM_ADDR_WIDTH_MAX    )
      //-------------------------------------------------------------------------
    ) dut (
      //-------------------------------------------------------------------------
      // Enable, Clock and Reset
      .en                               (pmc_en                 ),
      .arstn                            (pmc_arstn              ),
      .srstn                            (pmc_srstn              ),
      .clk                              (pmc_refclk             ),
      .pl_pclk                           (pmc_refclk             ),
      //-------------------------------------------------------------------------
      // UART Interface
      //  . UART RX
      .rx_datas_rdy                     (rx_datas_rdy           ),
      .rx_datas                         (rx_datas               ),
      .parity_error                     (parity_error           ),
      .start_bit_error                  (start_bit_error        ),
      //  . UART TX
      .tx_busy                          (tx_busy                ),
      .tx_datas_start                   (tx_datas_start         ),
      .tx_datas                         (tx_datas               )
      //-------------------------------------------------------------------------
);



endmodule

`resetall