//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description:  TestBench
//              
//              This module is composed of:
//                (1) Local Declarations
//                (2) Signal generations
//                (3) DUT Instantiation
//
// Dependency: axi_ram_backdoor
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

module tb ;

// Parameters
localparam  G_PCIE_NUM_LANES  = 'd2  ;
localparam  G_PIPE_WIDTH      = 'h044422  ; // Gen ??
localparam G_DATAPATH_SIZE        = 'd256 ;


//---------------------------------------------------------------------------
// Local Declarations
//---------------------------------------------------------------------------
// Clocks and Reset
reg                                               pmc_en                        ;
reg                                               pmc_arstn                     ;
reg                                               pmc_srstn                     ;
reg                                               pmc_refclk                   ;


//  PIPE Signals | PIPE Configuration
reg [3                                      -1:0] rate                      ;

// UART Interface
//  . UART RX
reg                                         rx_datas_rdy          ;
reg [8                                -1:0] rx_datas              ;
reg                                         parity_error          ;
reg                                         start_bit_error       ;
//  . UART TX
reg                                         tx_busy               ;
reg                                         tx_datas_start        ;
reg [8                                -1:0] tx_datas              ;

// AXI4 Master 1 Interface
//  . Read Address Channel
reg [4                                -1:0] axi4_mst1_arid        ;
reg [64                               -1:0] axi4_mst1_araddr      ;
reg [4                                -1:0] axi4_mst1_arregion    ;
reg [8                                -1:0] axi4_mst1_arlen       ;
reg [3                                -1:0] axi4_mst1_arsize      ;
reg [2                                -1:0] axi4_mst1_arburst     ;
reg                                         axi4_mst1_arlock      ;
reg [4                                -1:0] axi4_mst1_arcache     ;
reg [3                                -1:0] axi4_mst1_arprot      ;
reg [4                                -1:0] axi4_mst1_arqos       ;
reg                                         axi4_mst1_arvalid     ;
reg                                         axi4_mst1_arready     ;
//  . Read Data Channel
reg [4                               -1:0] axi4_mst1_rid         ;
reg [G_DATAPATH_SIZE                 -1:0] axi4_mst1_rdata       ;
reg [2                               -1:0] axi4_mst1_rresp       ;
reg                                        axi4_mst1_rlast       ;
reg                                        axi4_mst1_rvalid      ;
reg                                         axi4_mst1_rready     ;

// SYNC FSM signals
localparam[8     -1:0]  SYNC_BYTE_0       = 8'hFF,
                        SYNC_BYTE_1       = 8'hFE,
                        SYNC_BYTE_2       = 8'hFD,
                        SYNC_BYTE_3       = 8'hFC;

reg [8*5                                -1:0] stream_rx_datas              ;


//---------------------------------------------------------------------------
// Signal Generation
//---------------------------------------------------------------------------
integer a,b;
initial
begin
  pmc_en          <= 1'b1 ;
  pmc_arstn       <= 1'b0 ;
  pmc_srstn       <= 1'b1 ;
  pmc_refclk     <= 1'b0 ;

  rate        <= 2'd2 ;

  stream_rx_datas <= {8'h01, SYNC_BYTE_3, SYNC_BYTE_2, SYNC_BYTE_1, SYNC_BYTE_0};

  #20
  pmc_arstn       <= 1'b1 ;

  
  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps



  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  #120
  $finish;
end


// AXI Emulator (super simple)
always @(posedge pmc_refclk or negedge pmc_arstn) begin
  if (!pmc_arstn) begin
    axi4_mst1_arready <= 1'b0;

    axi4_mst1_rdata <= {(G_DATAPATH_SIZE){1'b0}};
    axi4_mst1_rresp <= 2'h0;
    axi4_mst1_rlast <= 1'b0;
    axi4_mst1_rvalid <= 1'b0;
  end
  else begin
    if (axi4_mst1_arvalid) begin
      axi4_mst1_arready <= 1'b1;
    end
    else begin
      axi4_mst1_arready <= 1'b0;
    end

    if (axi4_mst1_rready) begin
      axi4_mst1_rdata <= {(G_DATAPATH_SIZE/16){16'hAACC}};
      axi4_mst1_rlast <= 1'b1;
      axi4_mst1_rvalid <= 1'b1;
    end
    else begin
      axi4_mst1_rdata <= {(G_DATAPATH_SIZE){1'b0}};
      axi4_mst1_rlast <= 1'b0;
      axi4_mst1_rvalid <= 1'b0;
    end
  end
  
end

// UART Emulator
localparam  FSM_RST     = 2'h0  ;
localparam  FSM_SEND_RX = 2'h1  ;
localparam  FSM_WAIT_TX = 2'h2  ;

reg [2 -1:0] fsm_uart_state;


always @(posedge pmc_refclk or negedge pmc_arstn) begin
  if (!pmc_arstn) begin
    fsm_uart_state <= FSM_RST;
    rx_datas_rdy <= 1'b0;
    rx_datas <= 8'h0;
    parity_error <= 1'b0;
    start_bit_error <= 1'b0;

    tx_busy <= 1'b0;

    a <= 0;
  end
  else begin
    case (fsm_uart_state)
      FSM_RST: begin
        fsm_uart_state <= FSM_SEND_RX;
    
        rx_datas_rdy <= 1'b0;
        rx_datas <= 8'h0;
        parity_error <= 1'b0;
        start_bit_error <= 1'b0;

        tx_busy <= 1'b0;
        a <= 0;
      end
      FSM_SEND_RX: begin
        fsm_uart_state <= FSM_WAIT_TX;
    
        rx_datas_rdy <= 1'b1;
        rx_datas <= stream_rx_datas[a*8 +:8];

      end
      FSM_WAIT_TX: begin    
        rx_datas_rdy <= 1'b0;
        rx_datas <= 8'h0;

        if (tx_datas_start || 1'b1) begin
          if (a < (5-1))
            a <= a + 1;

          fsm_uart_state <= FSM_SEND_RX;
        end
        else begin
          fsm_uart_state <= FSM_WAIT_TX;
        end

      end
      default: begin
        fsm_uart_state <= FSM_RST;
    
        rx_datas_rdy <= 1'b0;
        rx_datas <= 8'h0;
        parity_error <= 1'b0;
        start_bit_error <= 1'b0;

        tx_busy <= 1'b0;

        a <= 0;
      end
    endcase
  end
  
end


/*
localparam  READ_REQ   = 2'b00  ;
localparam  READ_WAIT  = 2'b01  ;

reg [2 -1:0] fsm_axi_state;

always @(posedge pmc_refclk or negedge pmc_arstn) begin
  if (!pmc_arstn) begin
    fsm_axi_state <= READ_REQ;
    b = 0;
    //  . Read Address Channel       
    axi4_mst1_arid <= 4'h0;
    axi4_mst1_araddr <= 8'b00000000;
    axi4_mst1_arregion <= 'h0;
    axi4_mst1_arlen <= 8'h0;
    axi4_mst1_arsize <= 'h0;
    axi4_mst1_arburst <= 2'h1;
    axi4_mst1_arlock <= 1'b0;
    axi4_mst1_arcache <= 'h0;
    axi4_mst1_arprot <= 'h0;
    axi4_mst1_arqos <= 'h0;
    axi4_mst1_arvalid <= 1'b0;
    //  . Read Data Channel
    axi4_mst1_rready <= 1'b0;
     
  end
  else begin
    if (!pmc_srstn) begin
      fsm_axi_state <= READ_REQ;
      b = 0;
      //  . Read Address Channel       
      axi4_mst1_arid <= 4'h0;
      axi4_mst1_araddr <= 8'b00000000;
      axi4_mst1_arregion <= 'h0;
      axi4_mst1_arlen <= 8'h0;
      axi4_mst1_arsize <= 'h0;
      axi4_mst1_arburst <= 2'h1;
      axi4_mst1_arlock <= 1'b0;
      axi4_mst1_arcache <= 'h0;
      axi4_mst1_arprot <= 'h0;
      axi4_mst1_arqos <= 'h0;
      axi4_mst1_arvalid <= 1'b0;
      //  . Read Data Channel
      axi4_mst1_rready <= 1'b0;
    end
    else begin
      case (fsm_axi_state)
        READ_REQ :
        begin
          axi4_mst1_araddr <= b + (9'b100000000 * (b%(RAM_1_N + RAM_2_N + RAM_4_N + RAM_8_N))) + 9'b010000000 ;

          axi4_mst1_arlen <= 8'h0;
          axi4_mst1_arlock <= 1'b0;
          axi4_mst1_arvalid <= 1'b1;

          axi4_mst1_rready <= 1'b0;

          if (axi4_mst1_arready && axi4_mst1_arvalid)begin
            fsm_axi_state <= READ_WAIT;
            axi4_mst1_arvalid <= 1'b0;
            b = b + 1;
          end
          else begin
            fsm_axi_state <= READ_REQ;
          end 
        end

        READ_WAIT :
        begin
          axi4_mst1_arvalid <= 1'b0;
          axi4_mst1_rready <= 1'b1;

          if(axi4_mst1_rready && axi4_mst1_rvalid && axi4_mst1_rlast) begin
            axi4_mst1_rready <= 1'b0;
            axi4_mst1_arvalid <= 1'b1;
            fsm_axi_state <= READ_REQ;
          end
          else begin
            fsm_axi_state <= READ_WAIT;
          end 
        end
      endcase

    end// else srstn
  end // else arstn
end // always

*/

// Clock
always @(*)
begin
  if      (rate == 2'd3) #1; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #2; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #4; // 125 MHz @  5.0 Gbps
  else                   #8; // 62.5MHz @  2.5 Gbps
  pmc_refclk <= ~pmc_refclk ;
end


//---------------------------------------------------------------------------
// UART Controller
//---------------------------------------------------------------------------
uart_axi_ps #(
      //-------------------------------------------------------------------------
      .G_ADDR_WIDTH                     ('d64      ),
      .G_DATA_WIDTH                     (G_DATAPATH_SIZE        )
      //-------------------------------------------------------------------------
    ) dut (
      //-------------------------------------------------------------------------
      // Enable, Clock and Reset
      .en                               (pmc_en                 ),
      .arstn                            (pmc_arstn              ),
      .clk                              (pmc_refclk             ),
      //-------------------------------------------------------------------------
      // UART Interface
      //  . UART RX
      .rx_datas_rdy                     (rx_datas_rdy           ),
      .rx_datas                         (rx_datas               ),
      .parity_error                     (parity_error           ),
      .start_bit_error                  (start_bit_error        ),
      //  . UART TX
      .tx_busy                          (tx_busy                ),
      .tx_datas_start                   (tx_datas_start         ),
      .tx_datas                         (tx_datas               ),
      //  . Read Address Channel     
      .arid                             (axi4_mst1_arid         ),
      .araddr                           (axi4_mst1_araddr       ),
      .arregion                         (axi4_mst1_arregion     ),
      .arlen                            (axi4_mst1_arlen        ),
      .arsize                           (axi4_mst1_arsize       ),
      .arburst                          (axi4_mst1_arburst      ),
      .arlock                           (axi4_mst1_arlock       ),
      .arcache                          (axi4_mst1_arcache      ),
      .arprot                           (axi4_mst1_arprot       ),
      .arqos                            (axi4_mst1_arqos        ),
      .arvalid                          (axi4_mst1_arvalid      ),
      .arready                          (axi4_mst1_arready      ),
      //  . Read Data Channel
      .rid                              (axi4_mst1_rid          ),
      .rdata                            (axi4_mst1_rdata        ),
      .rresp                            (axi4_mst1_rresp        ),
      .rlast                            (axi4_mst1_rlast        ),
      .rvalid                           (axi4_mst1_rvalid       ),
      .rready                           (axi4_mst1_rready       )
      //-------------------------------------------------------------------------
);



endmodule

`resetall