//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor/Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: PIPE Error Generator
//
//              This module aims to introduce errors in the PIPE interface to
//              simulate possible scenarios and test the PIPE Monitor/Checker
//
// Dependency : 
//-----------------------------------------------------------------------------

`timescale 1 ns / 1 ps

module pipe_err_generator #(
    parameter                                          G_PIPE_INTF      = 1 ,
    parameter                                          G_PCIE_NUM_LANES = 16,
    parameter                                          G_PIPE_WIDTH     = 'h044422 
  )(
    //-------------------------------------------------------------------------
    // Module Indipendent Signals
    //-------------------------------------------------------------------------
    // input wire                                          peg_rstn              ,
    
    //-------------------------------------------------------------------------
    // Control Signals
    //-------------------------------------------------------------------------
    // input wire                                          k_endelay             ,

    //-------------------------------------------------------------------------
    // XpressRICH IP - PIPE Interface
    //-------------------------------------------------------------------------
    //  . Clock and Resets
    output wire                                         pl_pclk               ,
    output wire                                         pl_rstn               ,
    output wire                                         pl_rstnp              ,
    output wire                                         pl_srst               ,
    output wire                                         pl_npor               ,
    output wire                                         pl_spor               ,
    input  wire                                         pl_rstn_srst_out      ,
    input  wire [4                                -1:0] pl_exit               ,

    //  . PIPE signals (common to all lanes)
    input  wire [4                                -1:0] pl_powerdown          ,
    input  wire [3                                -1:0] pl_rate               ,
    input  wire [5                                -1:0] pl_ltssm              ,    
    input  wire [2                                -1:0] pl_equ_phase          ,

    //  . PIPE signals (interface width/clock rate change)
    input  wire [                                  2:0] pl_width              ,
    input  wire [                                  4:0] pl_pclk_rate          ,
    output wire [G_PCIE_NUM_LANES                 -1:0] pl_pclk_change_ok     ,
    input  wire [G_PCIE_NUM_LANES                 -1:0] pl_pclk_change_ack    ,
    input  wire [                                  4:0] pl_pll_rate           ,
    output wire                                         pl_pll_ack            ,
    input  wire                                         pl_esm_cal_request    ,
    output wire                                         pl_esm_cal_complete   ,
    input  wire [                                 31:0] pl_esm_rate           ,
    
    //  . PIPE signals (per lane)
    input  wire [64*G_PCIE_NUM_LANES                     -1:0] pl_sideband_out,
    input  wire [G_PCIE_NUM_LANES                        -1:0] pl_txdetectrx  ,
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8   -1:0] pl_txdata      ,
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)     -1:0] pl_txdatak     ,
    input  wire [G_PCIE_NUM_LANES                        -1:0] pl_txdatavalid ,
    input  wire [G_PCIE_NUM_LANES                        -1:0] pl_txstartblock,
    input  wire [G_PCIE_NUM_LANES*2                      -1:0] pl_txsyncheader,
    input  wire [G_PCIE_NUM_LANES                        -1:0] pl_txelecidle  ,
    input  wire [G_PCIE_NUM_LANES                        -1:0] pl_txcompliance,
    input  wire [G_PCIE_NUM_LANES                        -1:0] pl_rxstandby   ,
    
    output wire [64*G_PCIE_NUM_LANES                     -1:0] pl_sideband_in ,
    output wire [G_PCIE_NUM_LANES                        -1:0] pl_phystatus   ,
    output wire [G_PCIE_NUM_LANES*3                      -1:0] pl_rxstatus    ,
    output wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8   -1:0] pl_rxdata      ,
    output wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)     -1:0] pl_rxdatak     ,
    output wire [G_PCIE_NUM_LANES                        -1:0] pl_rxdatavalid ,
    output wire [  (1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0] pl_rxstartblock,
    output wire [  (1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0] pl_rxstartbhigh,
    output wire [2*(1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0] pl_rxsyncheader,
    output wire [G_PCIE_NUM_LANES                        -1:0] pl_rxvalid     ,
    output wire [G_PCIE_NUM_LANES                        -1:0] pl_rxelecidle  ,
    
    //  . PHY-MAC interface (per lane)
    output wire [G_PCIE_NUM_LANES*8               -1:0] pl_p2m_msgbus         ,
    input  wire [G_PCIE_NUM_LANES*8               -1:0] pl_m2p_msgbus         ,
    
    //  . WAKE# pin
    input  wire                                         pl_wake_oen           ,
    output wire                                         pl_wake_in            ,

    //  . CLKREQ# management signals
    input  wire                                         pl_clkreq_oen         ,
    output wire                                         pl_clkrem_allow       ,

    //-------------------------------------------------------------------------
    // PHY IP - PIPE Interface
    //-------------------------------------------------------------------------
    //  . Clock and Resets
    input  wire                                         phy_pclk               ,
    input  wire                                         phy_rstn               , //
    input  wire                                         phy_rstnp              , //
    input  wire                                         phy_srst               , //
    input  wire                                         phy_npor               , //
    input  wire                                         phy_spor               , //
    output wire                                         phy_rstn_srst_out      , //
    output wire [4                                -1:0] phy_exit               , //

    //  . PIPE signals (common to all lanes)
    output wire [4                                -1:0] phy_powerdown          ,
    output wire [3                                -1:0] phy_rate               ,
    output wire [5                                -1:0] phy_ltssm              , // 
    output wire [2                                -1:0] phy_equ_phase          , //

    //  . PIPE signals (interface width/clock rate change)
    output wire [                                  2:0] phy_width              , //
    output wire [                                  4:0] phy_pclk_rate          , //
    input  wire [G_PCIE_NUM_LANES                 -1:0] phy_pclk_change_ok     , //
    output wire [G_PCIE_NUM_LANES                 -1:0] phy_pclk_change_ack    , //
    output wire [                                  4:0] phy_pll_rate           , //
    input  wire                                         phy_pll_ack            , //
    output wire                                         phy_esm_cal_request    , //
    input  wire                                         phy_esm_cal_complete   , //
    output wire [                                 31:0] phy_esm_rate           , //

    //  . PIPE signals (per lane)
    output wire [64*G_PCIE_NUM_LANES                     -1:0] phy_sideband_out, //
    output wire [G_PCIE_NUM_LANES                        -1:0] phy_txdetectrx  , //
    output wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8   -1:0] phy_txdata      , // phy_
    output wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)     -1:0] phy_txdatak     , // phy_
    output wire [G_PCIE_NUM_LANES                        -1:0] phy_txdatavalid ,
    output wire [G_PCIE_NUM_LANES                        -1:0] phy_txstartblock,
    output wire [G_PCIE_NUM_LANES*2                      -1:0] phy_txsyncheader,
    output wire [G_PCIE_NUM_LANES                        -1:0] phy_txelecidle  ,
    output wire [G_PCIE_NUM_LANES                        -1:0] phy_txcompliance,
    output wire [G_PCIE_NUM_LANES                        -1:0] phy_rxstandby   , //
    input  wire [64*G_PCIE_NUM_LANES                     -1:0] phy_sideband_in , //
    input  wire [G_PCIE_NUM_LANES                        -1:0] phy_phystatus   , //
    input  wire [G_PCIE_NUM_LANES*3                      -1:0] phy_rxstatus    ,
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8   -1:0] phy_rxdata      , // phy_
    input  wire [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)     -1:0] phy_rxdatak     , // phy_
    input  wire [G_PCIE_NUM_LANES                        -1:0] phy_rxdatavalid ,
    input  wire [  (1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0] phy_rxstartblock, // phy_
    input  wire [  (1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0] phy_rxstartbhigh, //
    input  wire [2*(1+G_PIPE_WIDTH[20])*G_PCIE_NUM_LANES -1:0] phy_rxsyncheader, //
    input  wire [G_PCIE_NUM_LANES                        -1:0] phy_rxvalid     ,
    input  wire [G_PCIE_NUM_LANES                        -1:0] phy_rxelecidle  ,

    //  . PHY-MAC interface (per lane)
    input  wire [G_PCIE_NUM_LANES*8                      -1:0] phy_p2m_msgbus  , //
    output wire [G_PCIE_NUM_LANES*8                      -1:0] phy_m2p_msgbus  , //
    
    //  . WAKE# pin
    output wire                                                phy_wake_oen    , //
    input  wire                                                phy_wake_in     , //

    //  . CLKREQ# management signals
    output wire                                                phy_clkreq_oen  , //
    input  wire                                                phy_clkrem_allow  //

    // side-band signals
    // .phy_rxpolarity        (pl_rxpolarity),
    // .phy_txmargin          (pl_txmargin[2:0]),
    // .phy_txswing           (pl_txswing[0]),
    // .phy_txdeemph          (pl_txdeemph[0])
    );

    wire     k_endelay = 1'b0 ;
    wire     peg_rstn         ;

    //-------------------------------------------------------------------------
    // Signals forcing
    //-------------------------------------------------------------------------


    // Delay Signals
    localparam  RXDETECT_DELAY      =      16     * 4; // ns
    localparam  RX_DELAY            =      16     * 4; // ns

    reg [G_PCIE_NUM_LANES                        -1:0] delayed_phystatus   ;
    reg [G_PCIE_NUM_LANES*3                      -1:0] delayed_rxstatus    ;

    reg [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8   -1:0] delayed_rxdata      ;
    reg [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)     -1:0] delayed_rxdatak     ;
    reg [G_PCIE_NUM_LANES                        -1:0] delayed_rxdatavalid ;

    always @(phy_phystatus    ) delayed_phystatus         <= #(RXDETECT_DELAY ) phy_phystatus    ;
    always @(phy_rxstatus     ) delayed_rxstatus          <= #(RXDETECT_DELAY ) phy_rxstatus     ;

    always @(phy_rxdata       ) delayed_rxdata            <= #(RX_DELAY       ) phy_rxdata       ;
    always @(phy_rxdatak      ) delayed_rxdatak           <= #(RX_DELAY       ) phy_rxdatak      ;
    always @(phy_rxdatavalid  ) delayed_rxdatavalid       <= #(RX_DELAY       ) phy_rxdatavalid  ;

    /*
    assign pl_phystatus   = delayed_phystatus   ;
    assign pl_rxstatus    = delayed_rxstatus    ;

    assign pl_rxdata      = delayed_rxdata      ;
    assign pl_rxdatak     = delayed_rxdatak     ;
    assign pl_rxdatavalid = delayed_rxdatavalid ;
    */

    

    //-------------------------------------------------------------------------
    // Mechanisms to Prevent Infinit Simulation
    //-------------------------------------------------------------------------
    localparam STATE_WIDTH    =  5;
    localparam STATES_NUMBER  =  9;

    wire  statechange       ;
    wire  isempty           ;

    wire  twostatesloop     ;
    wire  threestatesloop   ;

    reg [STATE_WIDTH  -1:0] ltssm_shiftreg  [STATES_NUMBER  -1:0] ;
    
    assign statechange  = (ltssm_shiftreg[0] != pl_ltssm) && phy_rstn;
    assign isempty      = (ltssm_shiftreg[0] == 0) && (ltssm_shiftreg[1] == 0) && (ltssm_shiftreg[2] == 0) && (ltssm_shiftreg[3] == 0) && (ltssm_shiftreg[4] == 0) && (ltssm_shiftreg[5] == 0);


    // Identification of unplanned reset
    integer r   = 0;
    always @(posedge phy_rstn) begin
      r <= r + 1;
    end

    assign peg_rstn = (r > 0) ? 1'b1 : phy_rstn;

    always @(posedge phy_pclk) begin
      if (r > 1) begin
        $display("%t : %m : ERROR : UNEXPECTED ASSERTION OF RSTN", $time);
        $finish(0);
      end

    end

    // Shift Register that stores ltssm states
    integer i;
    always @(posedge phy_pclk, negedge peg_rstn) begin
      if (! peg_rstn) begin
        for (i = 0; i < STATES_NUMBER; i = i+1) begin
          ltssm_shiftreg[i] <= 0;
        end

      end else if (statechange) begin
        for (i = 1; i < STATES_NUMBER; i = i+1) begin
          ltssm_shiftreg[i] <= ltssm_shiftreg[i-1];
        end

        ltssm_shiftreg[0] <= pl_ltssm;
      end
    end


    // Identification of Loop
    assign twostatesloop    = (ltssm_shiftreg[0] == ltssm_shiftreg[2] && ltssm_shiftreg[0] == ltssm_shiftreg[4]) && (ltssm_shiftreg[1] == ltssm_shiftreg[3] && ltssm_shiftreg[3] == ltssm_shiftreg[5]) && !isempty;
    assign threestatesloop  = (ltssm_shiftreg[0] == ltssm_shiftreg[3] && ltssm_shiftreg[1] == ltssm_shiftreg[4] && ltssm_shiftreg[2] == ltssm_shiftreg[5]) && !isempty;
    

    //-------------------------------------------------------------------------
    // Output Signal Update
    //-------------------------------------------------------------------------
        //  . Clock and Resets
    assign pl_pclk              =   phy_pclk            ;
    assign pl_rstn              =   phy_rstn            ;
    assign pl_rstnp             =   phy_rstnp           ;
    assign pl_srst              =   phy_srst            ;
    assign pl_npor              =   phy_npor            ;
    assign pl_spor              =   phy_spor            ;
    assign phy_rstn_srst_out    =   pl_rstn_srst_out    ;
    assign phy_exit             =   pl_exit             ;

    //  . PIPE_top signals (common to all lanes)
    assign phy_powerdown        =   pl_powerdown        ;
    assign phy_rate             =   pl_rate             ;
    assign phy_ltssm            =   pl_ltssm            ;
    assign phy_equ_phase        =   pl_equ_phase        ;

    //  . PIPE signals (interface width/clock rate change)
    assign phy_width            =   pl_width            ;
    assign phy_pclk_rate        =   pl_pclk_rate        ;
    assign pl_pclk_change_ok    =   phy_pclk_change_ok  ;
    assign phy_pclk_change_ack  =   pl_pclk_change_ack  ;
    assign phy_pll_rate         =   pl_pll_rate         ;
    assign pl_pll_ack           =   phy_pll_ack         ;
    assign phy_esm_cal_request  =   pl_esm_cal_request  ;
    assign pl_esm_cal_complete  =   phy_esm_cal_complete;
    assign phy_esm_rate         =   pl_esm_rate         ;
    
    //  . PIPE signals (per lane)
    assign phy_sideband_out     =   pl_sideband_out     ;
    assign phy_txdetectrx       =   pl_txdetectrx       ;
    assign phy_txdata           =   pl_txdata           ;
    assign phy_txdatak          =   pl_txdatak          ;
    assign phy_txdatavalid      =   pl_txdatavalid      ;
    assign phy_txstartblock     =   pl_txstartblock     ;
    assign phy_txsyncheader     =   pl_txsyncheader     ;
    assign phy_txelecidle       =   pl_txelecidle       ;
    assign phy_txcompliance     =   pl_txcompliance     ;
    assign phy_rxstandby        =   pl_rxstandby        ;

    assign pl_sideband_in       =   phy_sideband_in     ;
    assign pl_phystatus         =   k_endelay ? delayed_phystatus   : phy_phystatus   ;
    assign pl_rxstatus          =   k_endelay ? delayed_rxstatus    : phy_rxstatus    ;
    assign pl_rxdata            =   k_endelay ? delayed_rxdata      : phy_rxdata      ;
    assign pl_rxdatak           =   k_endelay ? delayed_rxdatak     : phy_rxdatak     ;
    assign pl_rxdatavalid       =   k_endelay ? delayed_rxdatavalid : phy_rxdatavalid ;
    assign pl_rxstartblock      =   phy_rxstartblock    ;
    assign pl_rxstartbhigh      =   phy_rxstartbhigh    ;
    assign pl_rxsyncheader      =   phy_rxsyncheader    ;
    assign pl_rxvalid           =   phy_rxvalid         ;
    assign pl_rxelecidle        =   phy_rxelecidle      ;
    
    //  . PHY-MAC interface (per lane)
    assign pl_p2m_msgbus        =   phy_p2m_msgbus      ;
    assign phy_m2p_msgbus       =   pl_m2p_msgbus       ;
    
    //  . WAKE# pin
    assign phy_wake_oen         =   pl_wake_oen         ;
    assign pl_wake_in           =   phy_wake_in         ;

    //  . CLKREQ# management signals
    assign phy_wake_oen         =   pl_wake_oen         ;
    assign pl_wake_in           =   phy_wake_in         ;


endmodule