//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Reported Error Logic TestBench
//              
//              This module is composed of:
//                (1) Local Declarations
//                (2) Signal generations
//                (3) DUT Instantiation
//
// Dependency: axi_ram_backdoor
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

module tb ;

// Parameters
localparam  G_PCIE_NUM_LANES  = 'd2  ;
localparam  G_PIPE_WIDTH      = 'h044422  ; // Gen ??
localparam G_DATAPATH_SIZE        = 'd256 ;

localparam  G_PKT_WIDTH       = 'd16  ;     // Packet Width [bits]
localparam  G_MEM_ADDR_WIDTH  = 'd8   ;
localparam  G_MEM_DATA_WIDTH  = 'd32  ;


localparam   RAM_1_N                   =     2; // #             RAMs with ADDR WIDTH = RAM_MAX_ADDR_WIDTH
localparam   RAM_2_N                   =     1; // # Pairs    of RAMs with ADDR WIDTH = RAM_MAX_ADDR_WIDTH - 1
localparam   RAM_4_N                   =     0; // # Quartets of RAMs with ADDR WIDTH = RAM_MAX_ADDR_WIDTH - 2
localparam   RAM_8_N                   =     0; // # Octets   of RAMs with ADDR WIDTH = RAM_MAX_ADDR_WIDTH - 4
localparam   RAM_MAX_ADDR_WIDTH        =    'd8;
localparam   AXI_DATA_WIDTH            =    G_DATAPATH_SIZE;

//---------------------------------------------------------------------------
// Local Declarations
//---------------------------------------------------------------------------
// Clocks and Reset
reg                                               pmc_en                        ;
reg                                               pmc_arstn                     ;
reg                                               pmc_srstn                     ;
reg                                               pmc_refclk                   ;

//  Timer
reg [(G_PKT_WIDTH - 1 - 8)                  -1:0] global_time               ;

//  PIPE Signals | PIPE Configuration
reg [3                                      -1:0] rate                      ;


// AXI4 Master 1 Interface


//  . Read Address Channel
reg [4                                -1:0] axi4_mst1_arid        ;
reg [64                               -1:0] axi4_mst1_araddr      ;
reg [4                                -1:0] axi4_mst1_arregion    ;
reg [8                                -1:0] axi4_mst1_arlen       ;
reg [3                                -1:0] axi4_mst1_arsize      ;
reg [2                                -1:0] axi4_mst1_arburst     ;
reg                                         axi4_mst1_arlock      ;
reg [4                                -1:0] axi4_mst1_arcache     ;
reg [3                                -1:0] axi4_mst1_arprot      ;
reg [4                                -1:0] axi4_mst1_arqos       ;
reg                                         axi4_mst1_arvalid     ;
wire                                         axi4_mst1_arready     ;
//  . Read Data Channel
wire [4                                -1:0] axi4_mst1_rid         ;
wire [G_DATAPATH_SIZE                  -1:0] axi4_mst1_rdata       ;
wire [2                                -1:0] axi4_mst1_rresp       ;
wire                                         axi4_mst1_rlast       ;
wire                                         axi4_mst1_rvalid      ;
reg                                         axi4_mst1_rready      ;

reg [(RAM_1_N + 2 * RAM_2_N + 4 * RAM_4_N + 8 * RAM_8_N) * AXI_DATA_WIDTH   -1:0] shared_ram_rddata  ;

// 
reg [(32*RAM_MAX_ADDR_WIDTH)                               -1:0] rom_addr_list      ;
reg [(32*AXI_DATA_WIDTH)                               -1:0] rom_data_list      ;

//---------------------------------------------------------------------------
// Signal Generation
//---------------------------------------------------------------------------
integer a,b;
initial
begin
  pmc_en          <= 1'b1 ;
  pmc_arstn       <= 1'b0 ;
  pmc_srstn       <= 1'b1 ;
  pmc_refclk     <= 1'b0 ;

  global_time <=  'd0 ;


  for (a=0; a<32; a=a+1) begin
    rom_addr_list [a*RAM_MAX_ADDR_WIDTH +:RAM_MAX_ADDR_WIDTH] <= a;
    rom_addr_list [a*AXI_DATA_WIDTH +:AXI_DATA_WIDTH] <= a*4;
  
    

  end

  for (a=0; a<(RAM_1_N + 2 * RAM_2_N + 4 * RAM_4_N + 8 * RAM_8_N); a=a+1) begin
    // RAM Ports
    shared_ram_rddata[a*G_DATAPATH_SIZE +:G_DATAPATH_SIZE] <= a * 2 + 1;
  end

  rate        <= 2'd2 ;

  #20
  pmc_arstn       <= 1'b1 ;

  
  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps



  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  #120
  $finish;
end

// AXI 4 Controller
localparam  READ_REQ   = 2'b00  ;
localparam  READ_WAIT  = 2'b01  ;

reg [2 -1:0] fsm_axi_state;

always @(posedge pmc_refclk or negedge pmc_arstn) begin
  if (!pmc_arstn) begin
    fsm_axi_state <= READ_REQ;
    b = 0;
    //  . Read Address Channel       
    axi4_mst1_arid <= 4'h0;
    axi4_mst1_araddr <= 8'b00000000;
    axi4_mst1_arregion <= 'h0;
    axi4_mst1_arlen <= 8'h0;
    axi4_mst1_arsize <= 'h0;
    axi4_mst1_arburst <= 2'h1;
    axi4_mst1_arlock <= 1'b0;
    axi4_mst1_arcache <= 'h0;
    axi4_mst1_arprot <= 'h0;
    axi4_mst1_arqos <= 'h0;
    axi4_mst1_arvalid <= 1'b0;
    //  . Read Data Channel
    axi4_mst1_rready <= 1'b0;
     
  end
  else begin
    if (!pmc_srstn) begin
      fsm_axi_state <= READ_REQ;
      b = 0;
      //  . Read Address Channel       
      axi4_mst1_arid <= 4'h0;
      axi4_mst1_araddr <= 8'b00000000;
      axi4_mst1_arregion <= 'h0;
      axi4_mst1_arlen <= 8'h0;
      axi4_mst1_arsize <= 'h0;
      axi4_mst1_arburst <= 2'h1;
      axi4_mst1_arlock <= 1'b0;
      axi4_mst1_arcache <= 'h0;
      axi4_mst1_arprot <= 'h0;
      axi4_mst1_arqos <= 'h0;
      axi4_mst1_arvalid <= 1'b0;
      //  . Read Data Channel
      axi4_mst1_rready <= 1'b0;
    end
    else begin
      case (fsm_axi_state)
        READ_REQ :
        begin
          axi4_mst1_araddr <= b + (9'b100000000 * (b%(RAM_1_N + RAM_2_N + RAM_4_N + RAM_8_N))) + 9'b010000000 ;

          axi4_mst1_arlen <= 8'h0;
          axi4_mst1_arlock <= 1'b0;
          axi4_mst1_arvalid <= 1'b1;

          axi4_mst1_rready <= 1'b0;

          if (axi4_mst1_arready && axi4_mst1_arvalid)begin
            fsm_axi_state <= READ_WAIT;
            axi4_mst1_arvalid <= 1'b0;
            b = b + 1;
          end
          else begin
            fsm_axi_state <= READ_REQ;
          end 
        end

        READ_WAIT :
        begin
          axi4_mst1_arvalid <= 1'b0;
          axi4_mst1_rready <= 1'b1;

          if(axi4_mst1_rready && axi4_mst1_rvalid && axi4_mst1_rlast) begin
            axi4_mst1_rready <= 1'b0;
            axi4_mst1_arvalid <= 1'b1;
            fsm_axi_state <= READ_REQ;
          end
          else begin
            fsm_axi_state <= READ_WAIT;
          end 
        end
      endcase

    end// else srstn
  end // else arstn
end // always


always @(*)
begin
  if      (rate == 2'd3) #1; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #2; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #4; // 125 MHz @  5.0 Gbps
  else                   #8; // 62.5MHz @  2.5 Gbps
  pmc_refclk <= ~pmc_refclk ;
end

always @(posedge pmc_refclk or negedge pmc_arstn)
begin
  if  (!pmc_arstn)
      global_time    <=  'd0 ;
  else
      global_time    <=  global_time + 'd1 ;
end


//---------------------------------------------------------------------------
// AXI RAM Backdoor Access
//---------------------------------------------------------------------------
axi_ram_backdoor #(
      //-------------------------------------------------------------------------
      .AXI_ADDR_WIDTH                     ('d64      ),
      .AXI_DATA_WIDTH                     (G_DATAPATH_SIZE      ),
      .RAM_1_N                            (RAM_1_N  ),
      .RAM_2_N                            (RAM_2_N  ),
      .RAM_4_N                            (RAM_4_N  ),
      .RAM_8_N                            (RAM_8_N  ),
      .RAM_MAX_ADDR_WIDTH                 (8  )
      //-------------------------------------------------------------------------
    ) dut (
      //-------------------------------------------------------------------------
      // AXI4 Slave Interface
      //  . Clock and Resets
      .arstn                            (pmc_arstn              ),
      .srstn                            (pmc_srstn              ),
      .clk                              (pmc_refclk             ),
      //  . Read Address Channel     
      .arid                             (axi4_mst1_arid         ),
      .araddr                           (axi4_mst1_araddr       ),
      .arregion                         (axi4_mst1_arregion     ),
      .arlen                            (axi4_mst1_arlen        ),
      .arsize                           (axi4_mst1_arsize       ),
      .arburst                          (axi4_mst1_arburst      ),
      .arlock                           (axi4_mst1_arlock       ),
      .arcache                          (axi4_mst1_arcache      ),
      .arprot                           (axi4_mst1_arprot       ),
      .arqos                            (axi4_mst1_arqos        ),
      .arvalid                          (axi4_mst1_arvalid      ),
      .arready                          (axi4_mst1_arready      ),
      //  . Read Data Channel
      .rid                              (axi4_mst1_rid          ),
      .rdata                            (axi4_mst1_rdata        ),
      .rresp                            (axi4_mst1_rresp        ),
      .rlast                            (axi4_mst1_rlast        ),
      .rvalid                           (axi4_mst1_rvalid       ),
      .rready                           (axi4_mst1_rready       ),
      //-------------------------------------------------------------------------
      //RAM Ports
      .shared_ram_rden                  (        ),
      .shared_ram_rdaddr                (      ),
      .shared_ram_rddata                (shared_ram_rddata      )
      //-------------------------------------------------------------------------
);



endmodule

`resetall