//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Reported Error Logic TestBench
//              
//              This module is composed of:
//                (1) Local Declarations
//                (2) Signal generations
//                (3) DUT Instantiation
//
// Dependency : reported_error_logic
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

module tb ;

// Parameters
localparam  G_PCIE_NUM_LANES    = 'd2  ;
localparam  G_PIPE_WIDTH        = 'h044422  ; // Gen ??

localparam  G_PKT_WIDTH         = 'd32  ;     // Packet Width [bits]
localparam  G_ERR_ID_WIDTH      = 'd3   ;
localparam  G_MEM_ADDR_WIDTH    = 'd8   ;
localparam  G_MEM_DATA_WIDTH    = 'd32  ;

//  . RxStatus Codes
localparam DECODE_ERR           = 3'b100  ;
localparam ELSTC_BUFF_OVFL_ERR  = 3'b101  ;
localparam ELSTC_BUFF_UNFL_ERR  = 3'b110  ;
localparam RX_DISPARITY_ERR     = 3'b111  ;

//  . Symbols Codes
localparam EDB_SYM              = 8'hFE   ; // [K30.7]  EnD Bad -> Marks the end of a nullified TLP

//---------------------------------------------------------------------------
// Local Declarations
//---------------------------------------------------------------------------

// Clocks and Reset
reg                                               en                        ;
reg                                               arstn                     ;
reg                                               srstn                     ;
reg                                               pl_pclk                   ;

//  Timer
reg [(G_PKT_WIDTH - G_ERR_ID_WIDTH - G_PCIE_NUM_LANES)     -1:0] global_time               ;

//  PIPE Signals | PIPE Configuration
reg [3                                      -1:0] rate                      ;

//  PIPE RX Signals
reg [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8  -1:0] pl_rxdata                 ;
reg [G_PCIE_NUM_LANES*3                     -1:0] pl_rxstatus               ;
reg [G_PCIE_NUM_LANES                       -1:0] pl_rxvalid                ;

//  Reported Error RAM
wire                                                 scrambe_rerr_wren      ;
wire [G_MEM_DATA_WIDTH/8                       -1:0] scrambe_rerr_wrbe      ;
wire [G_MEM_ADDR_WIDTH                         -1:0] scrambe_rerr_wraddr    ;
wire [G_MEM_DATA_WIDTH                         -1:0] scrambe_rerr_wrdata    ;




//---------------------------------------------------------------------------
// Signal Generation
//---------------------------------------------------------------------------

initial
begin
  en          <= 1'b1 ;
  arstn       <= 1'b0 ;
  srstn       <= 1'b1 ;
  pl_pclk        <= 1'b0 ;

  global_time <=  'd0 ;

  rate        <= 2'd2 ;

  pl_rxdata   <= 'h0 ;
  pl_rxstatus <= 'h0 ;
  pl_rxvalid  <= 'h0 ;

  #80
  arstn       <= 1'b1 ;

  #80
  pl_rxvalid <= 2'b11;
  pl_rxstatus[5:0] <= {3'b000, DECODE_ERR};
  pl_rxdata[7:0]    <= EDB_SYM;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_rxstatus[5:0] <= {3'b000, DECODE_ERR};
  pl_rxdata[7:0]   <= 8'h00;
  pl_rxdata[15:8]  <= EDB_SYM;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_rxstatus[5:0] <= {DECODE_ERR, 3'b000};
  pl_rxdata[15:8]  <= 8'h00;
  pl_rxdata[39:32]   <= EDB_SYM;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_rxstatus[5:0] <= {ELSTC_BUFF_OVFL_ERR, 3'b000};
  pl_rxdata[39:32]   <= EDB_SYM;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_rxstatus[5:0] <= {ELSTC_BUFF_UNFL_ERR, 3'b000};
  pl_rxdata[39:32]   <= EDB_SYM;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_rxstatus[5:0] <= {RX_DISPARITY_ERR, 3'b000};
  pl_rxdata[39:32]   <= 8'h00;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_rxstatus <= 'h0;
  pl_rxdata <= 'h0;

  #80
  $finish;
end


always
begin
  if      (rate == 2'd3) #1; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #2; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #4; // 125 MHz @  5.0 Gbps
  else                   #8; // 62.5MHz @  2.5 Gbps
  pl_pclk <= ~pl_pclk ;
end

always @(posedge pl_pclk or negedge arstn)
begin
  if  (!arstn)
      global_time    <=  'd0 ;
  else
      global_time    <=  global_time + 'd1 ;
end


//---------------------------------------------------------------------------
// Reported Error Logic
//---------------------------------------------------------------------------

reported_error_logic #(
  //-------------------------------------------------------------------------
  //  . PIPE Parameters
  .G_PCIE_NUM_LANES (G_PCIE_NUM_LANES     ),
  .G_PIPE_WIDTH     (G_PIPE_WIDTH         ),
  //  . Module and Memory Parameters
  .PKT_WIDTH       (G_PKT_WIDTH           ),
  .MEM_ADDR_WIDTH  (G_MEM_ADDR_WIDTH      ),
  .MEM_DATA_WIDTH  (G_MEM_DATA_WIDTH      )
  //-------------------------------------------------------------------------
)dut (
  //-------------------------------------------------------------------------
  // Inputs
  //  . Clocks, EN and Resets
  .en                       (en                      ),
  .arstn                    (arstn                   ),
  .srstn                    (srstn                   ),
  .pclk                     (pl_pclk                 ),

  //  . Timer
  .global_time              (global_time             ),

  //  . PIPE RX Signals
  .pl_rxdata                (pl_rxdata               ),
  .pl_rxstatus              (pl_rxstatus             ),
  .pl_rxvalid               (pl_rxvalid              ),

  //-------------------------------------------------------------------------
  // Output
  //  . RAM Port
  .scrambe_wren             (scrambe_rerr_wren       ),
  .scrambe_wrbe             (scrambe_rerr_wrbe       ),
  .scrambe_wraddr           (scrambe_rerr_wraddr     ),
  .scrambe_wrdata           (scrambe_rerr_wrdata     )
  //-------------------------------------------------------------------------
);

//  . Reported Error RAM
xr5axi_scrambe #(
  //-------------------------------------------------------------------------
  // XR5AXI Layer Constants
  .DATA_WIDTH              (G_MEM_DATA_WIDTH          ),
  .ADDR_WIDTH              (G_MEM_ADDR_WIDTH          )
  //-------------------------------------------------------------------------
) xr5axi_scrambe_rerr_inst (
  //-------------------------------------------------------------------------
  // Buffer Interface
  //  . Clock
  .scram_clk               (pl_pclk                   ),
  //  . Write Port
  .scram_wren              (scrambe_rerr_wren         ),
  .scram_wrbe              (scrambe_rerr_wrbe         ),
  .scram_wraddr            (scrambe_rerr_wraddr       ),
  .scram_wrdata            (scrambe_rerr_wrdata       ),
  //  . Read  Port      
  .scram_rden              (                ),
  .scram_rdaddr            (                ),
  .scram_rddata            (                )
  //-------------------------------------------------------------------------
  );


endmodule

`resetall