//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: PIPE State Logic TestBench
//              
//              This module is composed of:
//                (1) Local Declarations
//                (2) Signal generations
//                (3) DUT Instantiation
//
// Dependency : pipe_state_logic
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

module tb ;

// Parameters
localparam  G_PKT_WIDTH         = 'd16  ;     // Packet Width [bits]
localparam  G_MEM_ADDR_WIDTH    = 'd8   ;
localparam  G_MEM_DATA_WIDTH    = 'd32  ;

//---------------------------------------------------------------------------
// Local Declarations
//---------------------------------------------------------------------------

// Clocks and Reset
reg                                           en                    ;
reg                                           arstn                 ;
reg                                           srstn                 ;
reg                                           pclk                  ;

//  Time
reg [(G_PKT_WIDTH - 5)                  -1:0] global_time           ;

//  PIPE Signals | PIPE State
reg [5                                  -1:0] ltssm                 ;
reg [2                                  -1:0] equ_phase             ;

//  PIPE Signals | PIPE Configuration
reg [3                                  -1:0] rate                  ;
reg [3                                  -1:0] width                 ;
reg [5                                  -1:0] pclk_rate             ;

//  Output
reg                                           psl_encod_128b_130b   ;


//---------------------------------------------------------------------------
// Signal Generation
//---------------------------------------------------------------------------

initial
begin
  en          <= 1'b1 ;
  arstn       <= 1'b0 ;
  srstn       <= 1'b1 ;
  pclk        <= 1'b0 ;

  global_time <=  'd0 ;

  ltssm       <= 5'h0 ;
  equ_phase   <= 2'h0 ;

  rate        <= 2'd2 ;
  width       <= 3'd2 ;
  pclk_rate   <= 2'd2 ;

  #80;
  arstn       <= 1'b1 ;
  ltssm       <= 5'h0 ;
  #80;
  ltssm       <= 5'h1 ;
  #80;
  ltssm       <= 5'h2 ;
  #80;
  ltssm       <= 5'h3 ;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  ltssm     <= 5'h4 ;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  ltssm     <= 5'h5 ;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  ltssm     <= 5'h0c ;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  equ_phase     <= 2'h1 ;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  equ_phase     <= 2'h2 ;

  #80
  $finish;
end


always
begin
  if      (rate == 2'd3) #1; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #2; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #4; // 125 MHz @  5.0 Gbps
  else                   #8; // 62.5MHz @  2.5 Gbps
  pclk <= ~pclk ;
end

always @(posedge pclk or negedge arstn)
begin
  if  (!arstn)
      global_time    <=  'd0 ;
  else
      global_time    <=  global_time + 'd1 ;
end


//---------------------------------------------------------------------------
// PIPE State Logic Instance
//---------------------------------------------------------------------------

pipe_state_logic #(
  .PKT_WIDTH       (G_PKT_WIDTH           ),
  .MEM_ADDR_WIDTH  (G_MEM_ADDR_WIDTH      ),
  .MEM_DATA_WIDTH  (G_MEM_DATA_WIDTH      )
)dut (
  //-------------------------------------------------------------------------
  // Inputs
  //  . Clocks, EN and Resets
  .en                       (en                      ),
  .arstn                    (arstn                   ),
  .srstn                    (srstn                   ),
  .pclk                     (pclk                    ),

  //  . Time
  .global_time              (global_time             ),

  //  . PIPE Signals | PIPE State
  .pl_ltssm                 (ltssm                   ),
  .pl_equ_phase             (equ_phase               ),

  //  . PIPE Signals | PIPE Configuration
  .pl_rate                 (rate                     ),
  .pl_width                (width                    ),
  .pl_pclk_rate            (pclk_rate                ),

  //-------------------------------------------------------------------------
  // Output
  .psl_encod_128b_130b    (                         ),
  //  . RAM Port
  .scrambe_wren           (       ),
  .scrambe_wrbe           (       ),
  .scrambe_wraddr         (     ),
  .scrambe_wrdata         (     )
  //-------------------------------------------------------------------------
);


endmodule

`resetall