//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Reported Error Logic TestBench
//              
//              This module is composed of:
//                (1) Local Declarations
//                (2) Signal generations
//                (3) DUT Instantiation
//
// Dependency : msg_bus_logic
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

module tb ;

// Parameters
localparam  G_PCIE_NUM_LANES  = 'd2  ;
localparam  G_PIPE_WIDTH      = 'h044422  ; // Gen ??

localparam  G_PKT_WIDTH       = 'd16  ;     // Packet Width [bits]
localparam  G_MEM_ADDR_WIDTH  = 'd8   ;
localparam  G_MEM_DATA_WIDTH  = 'd32  ;

//  . Message Bus Commands
localparam  NOP               = 4'b0000  ;
localparam  WRITE_UNCOMMITED  = 4'b0001  ;
localparam  WRITE_COMMITED    = 4'b0010  ;
localparam  READ              = 4'b0011  ;
localparam  READ_C0MPL        = 4'b0100  ;
localparam  WRITE_ACK         = 4'b0101  ;


//---------------------------------------------------------------------------
// Local Declarations
//---------------------------------------------------------------------------

// Clocks and Reset
reg                                               en                        ;
reg                                               arstn                     ;
reg                                               srstn                     ;
reg                                               pl_pclk                   ;

//  Timer
reg [(G_PKT_WIDTH - 1 - 8)                  -1:0] global_time               ;

//  PIPE Signals | PIPE Configuration
reg [3                                      -1:0] rate                      ;

//  PIPE RX Signals
reg [G_PCIE_NUM_LANES*8                     -1:0] pl_p2m_msgbus             ;
reg [G_PCIE_NUM_LANES*8                     -1:0] pl_m2p_msgbus             ;

//  . Message Bus RAM
wire [G_PCIE_NUM_LANES                      -1:0] scrambe_msgbus_wren       ;
wire [G_PCIE_NUM_LANES * G_MEM_DATA_WIDTH/8 -1:0] scrambe_msgbus_wrbe       ;
wire [G_PCIE_NUM_LANES * G_MEM_ADDR_WIDTH   -1:0] scrambe_msgbus_wraddr     ;
wire [G_PCIE_NUM_LANES * G_MEM_DATA_WIDTH   -1:0] scrambe_msgbus_wrdata     ;



//---------------------------------------------------------------------------
// Signal Generation
//---------------------------------------------------------------------------

initial
begin
  en          <= 1'b1 ;
  arstn       <= 1'b0 ;
  srstn       <= 1'b1 ;
  pl_pclk     <= 1'b0 ;

  global_time <=  'd0 ;

  rate        <= 2'd2 ;

  pl_p2m_msgbus   <= 'h0 ;
  pl_m2p_msgbus   <= 'h0;

  #40
  arstn       <= 1'b1 ;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_p2m_msgbus[7:4] <= WRITE_UNCOMMITED;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_p2m_msgbus[7:0] <= 8'hFF;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_p2m_msgbus[7:0] <= 8'hF0;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  
  pl_p2m_msgbus[7:0] <= 8'h00;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_m2p_msgbus[7:4] <= WRITE_ACK;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_m2p_msgbus[7:0] <= 8'h00;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  pl_m2p_msgbus[7:0] <= 8'h00;

  if      (rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps

  #40
  $finish;
end


always
begin
  if      (rate == 2'd3) #1; // 500 MHz @ 16.0 Gbps
  else if (rate == 2'd2) #2; // 250 MHz @  8.0 Gbps
  else if (rate == 2'd1) #4; // 125 MHz @  5.0 Gbps
  else                   #8; // 62.5MHz @  2.5 Gbps
  pl_pclk <= ~pl_pclk ;
end

always @(posedge pl_pclk or negedge arstn)
begin
  if  (!arstn)
      global_time    <=  'd0 ;
  else
      global_time    <=  global_time + 'd1 ;
end


//---------------------------------------------------------------------------
// Message Bus Logic
//---------------------------------------------------------------------------

msg_bus_logic #(
  //-------------------------------------------------------------------------
  //  . PIPE Parameters
  .G_PCIE_NUM_LANES (G_PCIE_NUM_LANES     ),
  .G_PIPE_WIDTH     (G_PIPE_WIDTH         ),
  //  . Module and Memory Parameters
  .PKT_WIDTH       (G_PKT_WIDTH           ),
  .MEM_ADDR_WIDTH  (G_MEM_ADDR_WIDTH      ),
  .MEM_DATA_WIDTH  (G_MEM_DATA_WIDTH      )
  //-------------------------------------------------------------------------
) dut (
  //-------------------------------------------------------------------------
  // Inputs
  //  . Clocks, EN and Resets
  .en                       (en                      ),
  .arstn                    (arstn                   ),
  .srstn                    (srstn                   ),
  .pclk                     (pl_pclk                 ),

  //  . Timer
  .global_time              (global_time             ),

  //  . PIPE RX Signals
  .pl_p2m_msgbus            (pl_p2m_msgbus           ),
  .pl_m2p_msgbus            (pl_m2p_msgbus           ),

  //-------------------------------------------------------------------------
  // Output
  //  . RAM Port
  .scrambe_wren             (scrambe_msgbus_wren     ),
  .scrambe_wrbe             (scrambe_msgbus_wrbe     ),
  .scrambe_wraddr           (scrambe_msgbus_wraddr   ),
  .scrambe_wrdata           (scrambe_msgbus_wrdata   )
  //-------------------------------------------------------------------------
);

//  . Message Bus RAMs
genvar i;

generate
  for (i = 0; i < G_PCIE_NUM_LANES; i = i + 1) begin
    xr5axi_scrambe #(
      //-------------------------------------------------------------------------
      // XR5AXI Layer Constants
      .DATA_WIDTH           (G_MEM_DATA_WIDTH          ),
      .ADDR_WIDTH           (G_MEM_ADDR_WIDTH          )
      //-------------------------------------------------------------------------
    ) xr5axi_scrambe_msgbus_inst (
      //-------------------------------------------------------------------------
      // Buffer Interface
      //  . Clock
      .scram_clk            (pl_pclk                   ),
      //  . Write Port
      .scram_wren           (scrambe_msgbus_wren   [i                                            ]),
      .scram_wrbe           (scrambe_msgbus_wrbe   [i*G_MEM_DATA_WIDTH/8     +:G_MEM_DATA_WIDTH/8]),
      .scram_wraddr         (scrambe_msgbus_wraddr [i*G_MEM_ADDR_WIDTH       +:G_MEM_ADDR_WIDTH  ]),
      .scram_wrdata         (scrambe_msgbus_wrdata [i*G_MEM_DATA_WIDTH       +:G_MEM_DATA_WIDTH  ]),
      //  . Read  Port      
      .scram_rden           (                ),
      .scram_rdaddr         (                ),
      .scram_rddata         (                )
      //-------------------------------------------------------------------------
    );
  end
endgenerate



endmodule

`resetall