//-----------------------------------------------------------------------------
// This confidential and proprietary software may be used only as authorized by
// a licensing agreement from PLDA. In the event of publication, a copyright
// notice must be reproduced on all authorized copies.
//
//-----------------------------------------------------------------------------
// Project : PIPE Monitor Checker
// Author  : nrigotti
//-----------------------------------------------------------------------------
// Description: Reported Error Logic TestBench
//              
//              This module is composed of:
//                (1) Local Declarations
//                (2) Signal generations
//                (3) DUT Instantiation
//
// Dependency : data_logic
//-----------------------------------------------------------------------------


`default_nettype none
`timescale 1 ns / 1 ps

`include "pipemc_constant_h.v"

module tb ;

// Parameters
localparam  G_PCIE_NUM_LANES    = 'd1  ;
//localparam  G_PIPE_WIDTH     = 'h044422  ;   // Gen ??
localparam  G_PIPE_WIDTH        = 'h88422  ;   // Gen 4

localparam  G_PKT_WIDTH         = 'd32  ;     // Packet Width [bits]
localparam  G_ERR_ID_WIDTH      = 'd3   ;
localparam  G_MEM_ADDR_WIDTH    = 'd8   ;
localparam  G_MEM_DATA_WIDTH    = 'd32  ;



//---------------------------------------------------------------------------
// Local Declarations
//---------------------------------------------------------------------------

// Clocks and Reset
reg                                               en                        ;
reg                                               arstn                     ;
reg                                               srstn                     ;
reg                                               pl_pclk                   ;

//  Timer
reg [(G_PKT_WIDTH - 5)     -1:0] global_time               ;
reg                                               data_encoding             ;
reg                                               detect_txn_ltssm          ;

//  PIPE Signals | PIPE Configuration
reg [3                                      -1:0] pl_rate                   ; 

//---------------------------------------------------------------------------
//  PIPE Signals
//  . Signals all lanes
reg [5                                        -1:0] pl_ltssm                ;
reg [2                                        -1:0] pl_equ_phase            ;
reg [3                                        -1:0] pl_width                ;

//  . Signals per lane - TX
reg [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8    -1:0] pl_txdata               ;
reg [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)      -1:0] pl_txdatak              ;
reg [G_PCIE_NUM_LANES                         -1:0] pl_txdatavalid          ;

//  . Signals per lane - RX
reg [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)*8    -1:0] pl_rxdata               ;
reg [G_PCIE_NUM_LANES*(G_PIPE_WIDTH>>16)      -1:0] pl_rxdatak              ;
reg [G_PCIE_NUM_LANES                         -1:0] pl_rxdatavalid          ;

//  Reported Error RAM
wire                                                 scrambe_evntx_wren     ;
wire [G_MEM_DATA_WIDTH/8                       -1:0] scrambe_evntx_wrbe     ;
wire [G_MEM_ADDR_WIDTH                         -1:0] scrambe_evntx_wraddr   ;
wire [G_MEM_DATA_WIDTH                         -1:0] scrambe_evntx_wrdata   ;


//---------------------------------------------------------------------------
// Signal Generation
//---------------------------------------------------------------------------

initial
begin
  en            <= 1'b1 ;
  arstn         <= 1'b0 ;
  srstn         <= 1'b1 ;
  pl_pclk       <= 1'b0 ;
  
  global_time   <=  'd0 ;

  pl_rate       <= 2'd2 ;
  data_encoding <= 1'b0 ;

  pl_ltssm      <= `LTSSM_POL_ACT ;
  pl_equ_phase  <= 2'h0 ;
  pl_width      <= `PIPE_WIDTH_16 ;

  pl_txdata         <= 'h0 ;
  pl_txdatak        <= 'h0;
  pl_txdatavalid    <= {(G_PCIE_NUM_LANES){1'b0}};

  pl_rxdata         <= 'h0 ;
  pl_rxdatak        <= 'h0;
  pl_rxdatavalid    <= {(G_PCIE_NUM_LANES){1'b0}};

  #40
  arstn       <= 1'b1 ;
  pl_txdatavalid   <= {(G_PCIE_NUM_LANES){1'b1}};

  // TS1
  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000f7f7bc ;
  pl_txdatak  <= 8'h07;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h00000000000020f7 ;
  pl_txdatak  <= 8'h01;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h000000000000001e ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000004a4a ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000004a4a ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'hf7bc000000004a4a ;
  pl_txdatak  <= 8'hc0;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h00f7f7bc00004a4a ;
  pl_txdatak  <= 8'h70;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h000000f7f7bc4a4a ;
  pl_txdatak  <= 8'h1c;

  //TS1
  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000f7f7bc ;
  pl_txdatak  <= 8'h07;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h00000000000020f7 ;
  pl_txdatak  <= 8'h01;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h000000000000001e ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000004a4a ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000004a4a ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'hf7bc000000004a4a ;
  pl_txdatak  <= 8'hc0;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h00f7f7bc00004a4a ;
  pl_txdatak  <= 8'h70;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h000000f7f7bc4a4a ;
  pl_txdatak  <= 8'h1c;

  //TS2
  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000f7f7bc ;
  pl_txdatak  <= 8'h07;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h00000000000020f7 ;
  pl_txdatak  <= 8'h01;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h000000000000001e ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000004545 ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h0000000000004545 ;
  pl_txdatak  <= 8'h00;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'hf7bc000000004545 ;
  pl_txdatak  <= 8'hc0;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h00f7f7bc00004545 ;
  pl_txdatak  <= 8'h70;

  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 64'h000000f7f7bc4545 ;
  pl_txdatak  <= 8'h1c;


  if      (pl_rate == 2'd3) #2; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #4; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #8; // 125 MHz @  5.0 Gbps
  else                   #16; // 62.5MHz @  2.5 Gbps
  pl_txdata   <= 'h0 ;
  pl_txdatak  <= 'h00;
  #40
  $finish;
end


always
begin
  if      (pl_rate == 2'd3) #1; // 500 MHz @ 16.0 Gbps
  else if (pl_rate == 2'd2) #2; // 250 MHz @  8.0 Gbps
  else if (pl_rate == 2'd1) #4; // 125 MHz @  5.0 Gbps
  else                   #8; // 62.5MHz @  2.5 Gbps
  pl_pclk <= ~pl_pclk ;
end

always @(posedge pl_pclk or negedge arstn)
begin
  if  (!arstn)
      global_time    <=  'd0 ;
  else
      global_time    <=  global_time + 'd1 ;
end


//---------------------------------------------------------------------------
// Data Logic
//---------------------------------------------------------------------------

data_logic #(
  //-------------------------------------------------------------------------
  //  . PIPE Parameters
  .G_PCIE_NUM_LANES (G_PCIE_NUM_LANES     ),
  .G_PIPE_WIDTH     (G_PIPE_WIDTH         ),
  //  . Module and Memory Parameters
  .PKT_WIDTH       (G_PKT_WIDTH           ),
  .MEM_ADDR_WIDTH  (G_MEM_ADDR_WIDTH      ),
  .MEM_DATA_WIDTH  (G_MEM_DATA_WIDTH      )
  //-------------------------------------------------------------------------
) dut (
  //-------------------------------------------------------------------------
  // Inputs
  //  . Clocks, EN and Resets
  .en                       (en                   ),
  .arstn                    (arstn                ),
  .srstn                    (srstn                ),
  .pclk                     (pl_pclk              ),

  //  . Timer
  .global_time              (global_time          ),

  //-------------------------------------------------------------------------
  // PIPE Signals
  //  . Signals all lanes
  .pl_powerdown           (              ),
  .pl_rate                (pl_rate                ),
  .pl_ltssm               (pl_ltssm               ),
  .pl_equ_phase           (pl_equ_phase           ),

  .pl_width               (pl_width               ),

   //  . Signals per lane - TX
  .pl_txdetectrx          (                ),
  .pl_txdata              (pl_txdata              ),
  .pl_txdatak             (pl_txdatak             ),
  .pl_txdatavalid         (pl_txdatavalid         ),

  //  . Signals per lane - RX
  .pl_phystatus           (                ),
  .pl_rxstatus            (                ),
  .pl_rxdata              (pl_rxdata              ),
  .pl_rxdatak             (pl_rxdatak             ),
  .pl_rxdatavalid         (pl_rxdatavalid         ),

  //-------------------------------------------------------------------------
  // PIPE Monitor Checker internal signals
  .data_encoding          (data_encoding          ),
  .detect_txn_ltssm       (detect_txn_ltssm       ),
  .prev_ltssm             (                       ),

  //-------------------------------------------------------------------------
  // Output
  //  . RAM Port
  .scrambe_wren           (scrambe_evntx_wren     ),
  .scrambe_wrbe           (scrambe_evntx_wrbe     ),
  .scrambe_wraddr         (scrambe_evntx_wraddr   ),
  .scrambe_wrdata         (scrambe_evntx_wrdata   )
  //-------------------------------------------------------------------------
);

//  . Reported Error RAM
xr5axi_scrambe #(
  //-------------------------------------------------------------------------
  // XR5AXI Layer Constants
  .DATA_WIDTH              (G_MEM_DATA_WIDTH          ),
  .ADDR_WIDTH              (G_MEM_ADDR_WIDTH          )
  //-------------------------------------------------------------------------
) xr5axi_scrambe_evntx_inst (
  //-------------------------------------------------------------------------
  // Buffer Interface
  //  . Clock
  .scram_clk               (pl_pclk                   ),
  //  . Write Port
  .scram_wren              (scrambe_evntx_wren        ),
  .scram_wrbe              (scrambe_evntx_wrbe        ),
  .scram_wraddr            (scrambe_evntx_wraddr      ),
  .scram_wrdata            (scrambe_evntx_wrdata       ),
  //  . Read  Port      
  .scram_rden              (                ),
  .scram_rdaddr            (                ),
  .scram_rddata            (                )
  //-------------------------------------------------------------------------
  );


endmodule

`resetall