#!/bin/sh
#-*-tcl-*-
# the next line restarts using wish \
exec wish "$0" -- ${1+"$@"}


########################################################################
#
# Read arguments
#
########################################################################
set num_arg    $argc
set simulator  ""
set exit_f     0
set nolog      0


# Detect if arguments can be recovered from argv
set no_argv 1
if {[llength $argv]>0} {
  set no_argv 0
  foreach arg $argv {
    # check -do or -gui is not found
    if {([regexp {^-do.*} $arg]==1) || ([regexp {^-gui.*} $arg]==1)} {
      set no_argv 1
    }
  }
}

for {set i 0} {$i<$num_arg} {incr i} {
  if {$no_argv==1} {
    set arg $1
    shift
  } else {
    set arg [lindex $argv $i]
  }

  # Simulator
  if {[regexp {model.*} $arg]==1} { set simulator "modelsim"}
  if {[regexp {ncsim.*} $arg]==1} { set simulator "ncsim"   }
  if {[regexp {vcs.*}   $arg]==1} { set simulator "vcs"     }
}

if {($num_arg==0) || ($simulator=="")} {
  puts "--------------------------------------------------------------------------------"
  puts "Usage: <scriptname> <model|ncsim|vcs> <x1|x4|x8> <m0|m1> <128B|256B|512B> <gen> <7gt|v7xt|kus|ku4|vusp|zusp> <pipe|serial>"
  puts ""
  puts "Mandatory Switches"
  puts "   model,ncsim,vcs        : select simulator"
  puts ""
  puts "Optional Switches"
  puts ""
  puts "--------------------------------------------------------------------------------"
  exit
}

if {(![info exists ::env(XILINX_LIB)])} {
  puts "Note : if XILINX_LIB environment variable is not defined, libraries will be compiled"
}
if {(![info exists ::env(XILINX_VIVADO)])} {
puts "ERROR: XILINX_VIVADO environment variable must be set in order to run serial simulation (eg: setenv XILINX_VIVADO /tools/xilinx/vivado)"
exit
}



########################################################################
#
# Get simulator version
#
########################################################################

if {$simulator=="modelsim"} {
  set ppipe [open "| vsim -version" "r"]
  set vstring [gets $ppipe]
  close $ppipe
  regexp {vsim\s+([0-9a-zA-Z_.]+)} $vstring _ version
  set questa [string first "Questa" $vstring]
  if {$questa==-1} {
    set xilinxmti "mti_se"
  } else {
    set xilinxmti "questa"
  }
}
if {$simulator=="ncsim"} {
  set ppipe [open "| ncsim -version" "r"]
  set vstring [gets $ppipe]
  close $ppipe
  regexp {ncsim\S*\s+(.+)} $vstring _ version
}
if {$simulator=="vcs"} {
  set ppipe [open "| vcs -ID" "r"]
  while { [gets $ppipe vstring] >= 0 } {
    regexp {VCS(-MX)* (.+)} $vstring _ version
  }
  close $ppipe
}


########################################################################
#
# Define path
#
########################################################################

set product_name          "Clock Frequency Monitor"
set core_lib_name         "pipe_monitor"
set refdesign_name        "None"
set unit_name             "uart_backdoor"

set core_path            "../source"
set refdesign_path       "../../"
set include_path         "../source/include"


########################################################################
#
# Display information and set licensing
#
########################################################################

puts "----------------------------------------------------------------"
puts "-- $product_name $refdesign_name Simulation Environment"
puts "----------------------------------------------------------------"
puts ""
puts "----------------------------------------------------------------"
puts "-- Platform  : $tcl_platform(platform)"
puts "-- Simulator : $simulator $version"
puts "----------------------------------------------------------------"
puts ""

########################################################################
#
# Prepare
#
########################################################################

file delete -force "libs"
file mkdir "libs"

# Prepare cds.lib & hdl.var
file delete -force cds.lib
file delete -force hdl.var
if [regexp {^ncsim.*} $simulator] {
  set fi [open "cds.lib" "w"]
  puts $fi {include $CDS_INST_DIR/tools/inca/files/cds.lib}
  close $fi
  set fi [open "hdl.var" "w"]
  close $fi
}

########################################################################
#
# modelsim
#
########################################################################

if {$simulator=="modelsim"} {

  exec vlib "libs/work"
  vmap work "libs/work"


  puts "..compile DUT"
  exec vlog -quiet -work work +incdir+$include_path \
        $core_path/design/memory/xr5axi_scrambe.v \
        $core_path/design/memory/write_memory_controller.v \
        $core_path/design/logic/core/msg_bus_monitor_x1.v \
        $core_path/design/logic/core/pipemc_shift_register.v \
        $core_path/design/logic/core/tsos_detector.v \
        $core_path/design/logic/core/eieos_detector.v \
        $core_path/design/logic/core/data_decoder_x1.v \
        $core_path/design/uart/uart_axi_ps.v \
        $core_path/design/memory/axi_ram_backdoor.v \
        $core_path/design/memory/pipemc_reg.v \
        $core_path/bench/uart/$unit_name.v
  

  puts "..compile Test Environment"
  exec vlog -quiet -work work -sv +incdir+$include_path \
        $core_path/bench/uart/tb_$unit_name.v
  vsim -quiet -voptargs="+acc" work.tb


  onerror { resume }
  onbreak { resume }

  puts "..disable Memory WildcardFilter"
  set WildcardFilter [lsearch -not -all -inline $WildcardFilter Memory]


  set wave_path /tb/dut/
  
  if {$nolog==0} {
    puts "..load Waveform"
    log -r $wave_path/*
    do wave_modelsim.tcl
  }

  puts "..start Simulation"
  run -all
  if {$exit_f==1} {
    quit -f
  }

}

########################################################################
#
# End
#
########################################################################

puts ".. done"
exit