%
% Add the padding bits to reach the desired size that is a multiple
% of two. We require a number of element, M, such that M >= 2p - 3.
% In our case as we want to perform a FFT of a sequence that is a
% power of two, we pick M = NextPow2(2p-3). To reach this size we
% add padding bits, in the following way:
%   - for a_data we insert NSamples - p null bits between the first
%     element and the second element of the original vector.
%   - for b_data  we cyclical repeat b_data to reach the desired
%     size. To do so we first calculate how many time we can fit 
%     b_data vector in the NSamples, given this we calcualate the remaining bits.
%     This bits are the first n-th bit of b_data.
% The result that the cyclic convolution produce have size
% NextPow2(2p) and we are only consider the first p bits of it.
% The example below explain the situation:
%   - a_data_ext = [a_data(1), 0, 0, ..., a_data(2:p)]
%   - b_data_ext = [b_data, b_data, ..., b_data(1:leftBits)]
%
function [a_padded, b_padded, NSamples] = BitPadder(a_data, b_data, p)   
    % Skip all number that are arleady a power of two. Just to be sure
    if 2^(MyLog2(p)) == p 
        a_padded = a_data;
        b_padded = b_data;
        NSamples = p;
    else
        NSamples = 2^(nextpow2(2*p-3));

        % Extend the a_data vector
        a_padded = [a_data(1) zeros(1, NSamples - p) a_data(2:end)];

        % extend the b_data vector
        numDataFrame = floor(NSamples / p);
        leftBits = NSamples - numDataFrame*p;

        % Replicate the input signal numDataFrame times to add pad bits
        b_input_rep = repmat(b_data, 1, numDataFrame);
        b_padded = [b_input_rep b_data(1:leftBits)];
    end 
end