#!/bin/sh

# Script used to extract the frames from a captured video or from an existing
# one.
# Device, resolution, frame rate and capture format can be configured for the
# camera, as well as the duration of the video and the format of the output
# frames

# Video and frame resolution
RES="640x480"


# Frame rate, high value to set to the max for the camera
FPS=30

# Camera device
DEV="/dev/video4"

# Camera video capture format
FORMAT="mjpeg"

# Duration of the video in seconds
TIME=10

# Frame extraction format, pgm is raw grayscale
OUTEXT=".pgm"

# Video and frame output directory"
MDIR=$(readlink -f $(dirname $0))
VIDEODIR="$MDIR/video"
FRAMEDIR="$MDIR/frames"

##########################################################
# DO NOT EDIT BELOW THIS POINT                           #
##########################################################

if [ "$FORMAT" = "mjpeg" ] 
then
	OUTPUT=${VIDEODIR}/video_${FPS}_${RES}.mjpg
elif [ "$FORMAT" = "yuyv422" ] 
then
	OUTPUT=${VIDEODIR}/video_${FPS}_${RES}.yuv
else 
	echo "Please specify a valid format: mjpeg|yuyv422"
	exit 1
fi

if [ -z $1 ]
then
	if [ ! -d $VIDEODIR ] 
	then
		mkdir $VIDEODIR
	elif [ -f $OUTPUT ] 
	then
		mv $OUTPUT ${OUTPUT}.old
	fi
fi

if [ -d $FRAMEDIR ] 
then
	if [ -d ${FRAMEDIR} ]
	then
		rm -r ${FRAMEDIR}
	fi
fi
mkdir $FRAMEDIR

# Print available formats and resolutions
#ffmpeg -hide_banner -f v4l2 -list_formats all -i $DEV

if [ -z $1 ] 
then

	if [ $FORMAT = "mjpeg" ]
	then
		ffmpeg -hide_banner \
			-f v4l2 \
			-y \
			-framerate $FPS \
			-video_size $RES \
			-input_format $FORMAT \
			-i $DEV \
			-t $TIME \
			-q:v 2 \
			"$OUTPUT"

		# Convert to frames
		ffmpeg -y -hide_banner -i $OUTPUT "$FRAMEDIR/frame_%04d${OUTEXT}"
		
	elif [ $FORMAT = "yuyv422" ]
	then
		ffmpeg -hide_banner \
			-f v4l2 \
			-y \
			-framerate $FPS \
			-video_size $RES \
			-input_format $FORMAT \
			-i $DEV \
			-t $TIME \
			-c:v copy \
			"$OUTPUT"

		# Convert to frames
		ffmpeg -y -hide_banner -video_size $RES -i $OUTPUT "$FRAMEDIR/frame_%04d${OUTEXT}"
	fi	


elif [ -f "$1" ]
then
	
	# Convert to frames the given video
	if [ -z $2 ]
	then
		#ffmpeg -y -hide_banner -i "$1" -filter:v "crop=362:480:0:0" "$FRAMEDIR/frame_%04d${OUTEXT}"
		ffmpeg -y -hide_banner -i "$1" "$FRAMEDIR/frame_%04d${OUTEXT}"
		if [ ! $? == 0 ] 
		then 
			ffmpeg -y -hide_banner -video_size $RES -i "$1" "$FRAMEDIR/frame_%04d${OUTEXT}"
		fi
	else
		# The target resolution can be specified using the second command line argument
		#ffmpeg -y -hide_banner -i "$1" -vf "scale=$2:-1,crop=97:120:10:0" "$FRAMEDIR/frame_%04d${OUTEXT}"
		ffmpeg -y -hide_banner -i "$1" -vf "scale=$2:-1" "$FRAMEDIR/frame_%04d${OUTEXT}"
		if [ ! $? == 0 ] 
		then 
			ffmpeg -y -hide_banner -video_size $RES -i "$1" -vf "scale=$2:-1" "$FRAMEDIR/frame_%04d${OUTEXT}"
		fi
	fi

	# Remove duplicated frames excluding the first 100 frames (which usually are static)
	#[ ! -d /tmp/framesbk ] && mkdir /tmp/framesbk
	#find "${FRAMEDIR}" -name "frame_00*" -exec mv {} /tmp/framesbk/ \;
	#fdupes -d --noprompt --quiet "${FRAMEDIR}/" > /dev/null
	#find "/tmp/framesbk" -name "frame_00*" -exec mv {} "${FRAMEDIR}" \; 

else 
	echo "Usage:"
	echo "    $0				Capture camera and extract frames"
	echo "    $0 filename		Extract the frames from the given video"
fi

