#ifndef SERIAL_H
#define SERIAL_H

/**
 * @file serial.h
 * @author Luca Favario
 * @date Aug 2021
 * @brief Serial UART output functions
 *
 * In this file are defined the funcions related to the serial UART. Writing
 * the displacement to UART is an easy way to provide the result to all kind of
 * devices.
 *
 * The output can be in 'CHAR' format or 'RAW' format, selected with
 * SERIALFORMAT:
 *	- if CHAR then the displacement data is converted to a string of characters
 *	terminated by '\n\r'. The transmitted data is easily readable by another
 *	device or computer. For example using `cat /dev/ttyUSB0` on a linux
 *	machine, where /dev/ttyUSB0 is the receiver UART device. This format
 *	requires more bytes to be transmitted (>=8 depending on the value) 
 *	- if RAW is selected then the displacement is sent as bytes and the
 *	receiver needs to khow the format of the transmitted data to be able to
 *	read it. This method requires less bytes to be transmitted (4 in case of
 *	float)
 *
 * The baude rate can be set with SERIALSPEED in the config file, while the
 * other configurations are fixed:
 *	- 8 bit
 *	- 1 stop bit
 *	- no parity bit
 *	- non-canonical mode: no special processing is done, and the terminal
 *		driver returns individual characters.
 
 */

#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <termios.h>
#include <sys/ioctl.h>
#include "common.h"

/**
 * @def CHAR
 * @brief Write to serial a string of chars
 */
#define CHAR 0

/**
 * @def RAW
 * @brief Write to serial the raw data, for example in case of float 4 bytes
 */
#define RAW 1

/**
 * @brief Initialize and open the serial interface.
 * @param fd Integer file descriptor pointer, written by this function.
 * @return Returns an integer, 0 in case of success and 1 in case of error.
 */
int serial_open(int *fd);

/**
 * @brief Close the serial interface.
 * @param fd Integer file descriptor 
 * @return Returns an integer, 0 in case of success and 1 in case of error
 */
void serial_close(int fd);

/**
 * @brief Write a float to the serial port as a string.
 * @param fd Integer file descriptor.
 * @param data Floating point data to be written to uart.
 * @return Returns an integer, 0 in case of success and 1 in case of error.
 */
int serial_write(int fd, float data);

#endif
