#include "serial.h"

/* Open the UART device for writing, then use the termios.h library to
 * configure the connection.
 */
int serial_open(int *fd) {

	struct termios tty;
	int file_desc;

	// Open the UART device in write mode
	if ((file_desc = open(SERIALDEV, O_WRONLY | O_NOCTTY | O_NONBLOCK)) == -1) 
		return 1;

	// Check if SERIALDEV is a serial device
	if (!isatty(file_desc)) {
		close(file_desc);
		return 2;
	}
	
	// Read the config of SERIALDEV
	if (tcgetattr(file_desc, &tty) < 0) {
		close(file_desc);
		return 3;
	}

	// Set baud rate to the defined SERIALSPEED
	cfsetospeed(&tty, (speed_t) SERIALSPEED);

	// Disable character processing
	// 8 bit, no parity bit, 1 stop bit, no hw flow control
	tty.c_cflag &= ~(CSIZE | PARENB | CSTOPB | CRTSCTS);
	tty.c_cflag |= (CLOCAL | CS8);

	// Disable output processing
	tty.c_oflag = 0;

	// Disable line processing
	tty.c_lflag &= ~(ECHO | ECHONL | ICANON | IEXTEN | ISIG);

	// Apply the edited config
	if (tcsetattr(file_desc, TCSANOW, &tty) < 0) {
		close(file_desc);
		return 4;
	}

	// Update the file descriptor
	*fd = file_desc;

	return 0;
}

// Close the serial device
void serial_close(int fd) {
	close(fd);
}

// Write data to serial
int serial_write(int fd, float data) {

#if SERIALFORMAT == CHAR
	// Data is converted to string and sent as a sequence of chars.
	// Requires an higher speed because in case of large data many bytes have to be
	// transmitted
	char str[32];
	unsigned int len;

	sprintf(str, "%.5f\n\r", data);
	len = strlen(str);

	if (write(fd, str, len) != len)
		return 1;

#elif SERIALFORMAT == RAW
	// Data is sent as raw bytes. It requires the receiver to be written
	// accordingly and to know the format of data. In case of float 4 bytes are
	// sent, which is less expensive than the string option.
	if (write(fd, &data, sizeof(data)) != sizeof(data))
		return 1;
	
#endif

	return 0;
}
