#include <stdio.h>
#include <signal.h>
#include <sys/time.h>
#include "fft.h"
#include "misc.h"

#ifdef FILECAMERA
#include "camera_file.h"
#else
#include "camera.h"
#endif

#ifdef SERIAL
#include "serial.h"
#endif

#ifdef BENCHMARK
#include "benchmark.h"
#endif

// Global variable used to stop the main loop when SIGINT is received
volatile int flag = 1;

void intHandler(int unused);

#ifdef DEBUG
static void print_vector(int cycle, float *vect, char * filename, unsigned int length);
#endif

int main() {

	/****************************
	 * Initialize system        *
	 ****************************/

	// FFTW plans forward and inverse FFT
	fftwf_plan pfor, pback;

	// Reference frame, captured frame and cross-correlation in frequency
	// domain. Length RESX/2+1 because Hermitian redundancy is exploited.
	// f1_fft and f2_fft are used as double buffer to store the reference frame
	// and the captured frames.
	fftwf_complex *f1_fft = NULL, *f2_fft = NULL, *cross = NULL;
	
	// Pointers for the double buffer, reference and current frame
	fftwf_complex *ref, *curr, *tmp;

	// Cross-correlation (not upsampled) in time domain
	float *sig = NULL;

	// Vector with the sum of the columns of the captured frame. 
	float *vect;

	// Conversion factor from pixel to millimeters
	float pix_to_mm;

	// Number of peaks (white bands) in the first reference image. This is used
	// to compute the threshold for the reference image.
	unsigned int peakcount;

	// Threshold used to decide when to update the reference image. If the
	// displacement is larger than this threshold the image is updated.
	float dpp, deltath;

	// Maximum displacement allowed with respect to the reference frame.
	unsigned int deltarange;

	// Total displacement with respect to the first captured frame in pixel units
	float res;

	// Output displacement in pixel units or millimeters (the latter if
	// PIXTOMMOUT is set)
	float output;

	// Position of the reference frame with respect to the first captured frame
	float ref_position;

	// Displacement in the upsampled neighborhood of the peak, the unit is
	// upsampled pixel. 
	// Can be fractional: if there are multiple adjacent points with the same
	// max value the returned position is the center of the max interval.
	float win_delta;

	// Displacement in pixel units, without upsampling, float because in case
	// of multiple peaks the mean is returned
	float delta; 

	// Struct used by camera.c to store informations about the device
	struct camera_dev_t dev;

	// Catch SIGINT
	struct sigaction sa;

	// Flag set when an error occurs computing the upsampled position
	unsigned char do_not_update_ref = 0;

	// Return value of the program 0 in case of success, 1 in case of error
	unsigned int retval = 0;

	// Variables used to evaluate computation time and numerate frames
#ifdef DEBUG
	unsigned int count = 0;
	struct timeval start, stop;
#endif

#ifdef SERIAL
	int fd, val;

	fprintf(stderr, "Opening and configuring UART...");
	fflush(stderr);
	if ((val = serial_open(&fd))) {
		fprintf(stderr, "Fail\nFailed to open serial %s, check device name and permissions (%d)\n\n", SERIALDEV, val);
		retval = 1;
		goto cleanup_serial;
	}
	fprintf(stderr, "Done\n");
#endif

	// Signal handler SIGINT
	sa.sa_handler = intHandler;
	sa.sa_flags = 0;
	sigaction(SIGINT, &sa, NULL);

	// Camera initialization
	fprintf(stderr, "Preparing camera...");
	fflush(stderr);
	if (camera_init(&dev) < 0) {
		fprintf(stderr, "Fail\nCan not open the specified device.\n\n");
		retval = 1;
		goto cleanup_camera_init;
	}
	fprintf(stderr, "Done\n");

	// Fft initialization
	fprintf(stderr, "Preparing plans...");
	fflush(stderr);

	// Allocate the vector to ensure SIMD compatible alignment.
	vect = fft_malloc_r(RESX);

	// Allocate the spatial cross correlation vector to be sure that the
	// alignment is correct for FFTW SIMD.
	sig = fft_malloc_r(ZPRU);

	// Allocate the vectors allowing them to be aligned for SIMD in FFTW
	f1_fft = fft_malloc_c(RESX2+1);
	f2_fft = fft_malloc_c(RESX2+1);
	cross = fft_malloc_c(ZPRU2+1);

	// Update reference and current frame pointers
	ref = f1_fft;
	curr = f2_fft;

	// Read wisdom file
	fft_read_wisdom();

	// Initialize plans
	pfor = fft_init_fw(vect, f1_fft);
	pback = fft_init_bw(cross, sig);

	// Save wisdom
	fft_save_wisdom();
	fprintf(stderr, "Done\n");
	
	// Setup the camera
	fprintf(stderr, "Starting the camera capture...");
	fflush(stderr);
	if (camera_start(&dev) < 0) {
		fprintf(stderr, "Fail\n");
		retval = 1;
		goto cleanup_camera_start;
	}
	fprintf(stderr, "Done\n");
	

	/****************************
	 * Reference frame          *
	 ****************************/
	
	fprintf(stderr, "Preparing reference vector...");
	fflush(stderr);

	// Get the vector of the referece frame. 
	// Throw away the first frames waiting for the camera to adjust after
	// startup. Sometimes brightness changes in the first few frames captured.
	for (int i = 0; i < 20; i++) {
		if (camera_get_frame(&dev, &vect) < 0) {
			fprintf(stderr, "Fail\nFailed to get frame\n");
			retval = 1;
			goto cleanup_camera_start;
		}
	}

#ifdef DEBUG
	unsigned int refcount = 0;
	print_vector(0, vect, "vector", RESX);
#endif

#ifdef PIXTOMMOUT
	// Compute the conversion factor from pixels to millimeters and the number of peaks
	if (pixel_to_mm_factor(vect, &pix_to_mm, &peakcount)) {
		fprintf(stderr, "Fail");
		fprintf(stderr, "\nFailed to evaluate conversion factor, check that at least one triplet is always present in the frame (narrow, medium and wide black bands)\n");
		retval = 1;
		goto cleanup_camera_start;
	}
#else
	// If PIXTOMMOUT is not set only the number of peaks is used
	pixel_to_mm_factor(vect, &pix_to_mm, &peakcount);
#endif 

#ifdef ZEROTH
	threshold_apply(vect);
#endif

	// Compute the displacement threshold for the new reference image 
	dpp = (peakcount) ? RESX / peakcount : 0;
	deltath = DELTATHFACT * dpp;

	// If deltath is too large set deltath to a known save value (if the number
	// of white bands is too low).
	if (deltath == 0 || deltath > DELTATHFALLBACK) deltath = DELTATHFALLBACK;

	// Maximum allowed displacement with respect to the reference frame
	deltarange = ZPUPS * (int)(dpp * DELTARANGEFACT);
	if (deltarange > (ZPRU >> 1) || dpp == 0) deltarange = (ZPRU >> 1);

	// Compute the Fourier transform of the reference frame
	fft_execute_fw(pfor, vect, ref);
	fprintf(stderr, "Done\n");
	fflush(stderr);

	// Initialize the position of the reference frame
	ref_position = 0.0;
	
	/****************************
	 * Main loop				*
	 ****************************/
	
	// Initialize the position 
	res = 0;
	win_delta = 0;
	output = 0;

#ifndef BENCHMARK

	// Run until flag is true (until SIGINT is received)
	while (flag) {
#else 
	if (bench_open()) {
		fprintf(stderr, "Failed to open bench file\n");
		flag = 0;
	}
	bench_start();
	// Run BENCHN times or until flag is true
	for (int i = 0; i < BENCHN && flag; i++) {
#endif

#ifdef DEBUG
		// Debug mode, get a frame on key press
		fprintf(stderr, "Press enter to continue\n");
		getchar();
#endif

		// Get the vector computed summing the columns the camera frame buffer
		if (camera_get_frame(&dev, &vect) < 0) {
			fprintf(stderr, "Failed to get frame\n");
			retval = 1;
			goto cleanup_camera_start;
		}

#ifdef DEBUG
		// In debug mode the conversion is recomputed each iteration to check
		// the correct behavior, even when the flag is not set
		pixel_to_mm_factor(vect, &pix_to_mm, &peakcount);

		// Debug, save the captured frame
		print_vector(count, vect, "vector", RESX);
		gettimeofday(&start, NULL);

#elif defined BENCHMARK
		bench_start_time();
#endif

#ifdef ZEROTH
		threshold_apply(vect);
#endif

		// Compute the FFT of the captured frame
		fft_execute_fw(pfor, vect, curr);

		// Cross correlation between reference frame and captured frame and
		// apply the zero-padding if required
		fft_xcorr(ref, curr, cross);

		// Inverse FFT
		fft_execute_bw(pback);
	
#ifdef DEBUG_FFT
		print_vector(count, sig, "cross", ZPRU);
#endif
		
		// Compute the displacement, position of peak with respect to the
		// center.
		if (fft_displacement(sig, deltarange, &delta)) {
#ifdef DEBUG
			fprintf(stderr, "Warning: error computing the pixel displacement, multiple peaks found. The current frame is skipped and the previous position is returned. Please check camera lighting, speed, focus and bands verticality. The current frame will be skipped\n");
#endif
			do_not_update_ref = 1;

		} else {
			// Compute the delta in pixel units
			delta = delta / (float) ZPUPS;

			// Inverse DFT, upsampled displacement in the neighborhood of the peak
			// Returns 1 if the window is not centered correctly and in this
			// case does not update the reference image.
			if (fft_dft_bw(cross, delta, &win_delta)) {
#ifdef DEBUG
				fprintf(stderr, "Warning: peak not centered in the window, the reference image won't be updated\n");
#endif
				do_not_update_ref = 1;
			}
			 
			// Compute the position with respect to the first captured frame
			res = ref_position + delta + win_delta / (float) UPS;

#ifdef PIXTOMMOUT
			// Output in mm evaluated using the factor computed using the
			// transitions of the first frame
			output = res * pix_to_mm;

#elif defined (PIXTOMMCAL) 
			// Output in mm using the manually provided factor
			output = res * (float) PIXTOMMCAL;

#else
			// Output in pixels
			output = res;
#endif

		}

#ifdef DEBUG
		// Debug information
		gettimeofday(&stop, NULL);
		fprintf(stderr, "Computation time: %f\n", (stop.tv_sec - start.tv_sec) + (stop.tv_usec - start.tv_usec) / 1e6);
		fprintf(stderr, "Delta: %f, Win delta: %f, Tot delta: %lf, Position in pixels: %lf\n", delta, win_delta, delta + win_delta/UPS, res);
		fprintf(stderr, "Current frame: %u, Reference frame: %u\n", count, refcount);
		fprintf(stderr, "Pixel to mm conversion factor: %f\n", pix_to_mm);
#elif defined BENCHMARK
		bench_stop_time();
		bench_write_computation();
#endif

#ifdef STDOUT
		// Print the displacement with respect to the first frame to stdout
		fprintf(stdout, "%f\n", output);
#endif

#ifdef SERIAL
		// Write the displacement to serial
		serial_write(fd, output);
#endif

		// If the displacement in pixel units is larger than the defined
		// threshold then update the reference vector in the frequency domain
		// and the position of the reference frame.
		if (!do_not_update_ref && ( delta > deltath || delta < -deltath)) {

			// Update the reference vector in frequency domain using the buffers
			tmp = ref;
			ref = curr;
			curr = tmp;

			// Update the position of the reference frame with respect of the
			// first frame
			ref_position = res;

#ifdef DEBUG
			// Debug info
			fprintf(stderr, "Updated reference position: %lf, Count: %u\n", ref_position, count);
			refcount = count;
#endif
		}

		// Reset flag
		do_not_update_ref = 0;

#ifdef DEBUG
		// Update counter for average time computation
		count++;
#endif
	}

#ifdef BENCHMARK 
	bench_stop();
	bench_close();
#endif


	/****************************
	 *  Uninitialize            *
	 ****************************/

	// Camera stop capture
cleanup_camera_start:
	fprintf(stderr, "\nStopping camera capture...");
	fflush(stderr);
	camera_stop(&dev);
	fprintf(stderr, "Done\n");

	// Destroy FFTW plans
	fprintf(stderr, "Destroying FFTW plans...");
	fflush(stderr);
	fft_destroy(&pfor);
	fft_destroy(&pback);

	// Free allocated FFTW vectors
	fft_free((void **)&f1_fft);
	fft_free((void **)&f2_fft);
	fft_free((void **)&cross);
	fft_free((void **)&sig);
	fft_free((void **)&vect);
	fprintf(stderr, "Done\n");

	// Camera uninitialize
cleanup_camera_init:
	fprintf(stderr, "Closing camera...");
	fflush(stderr);
	camera_close(&dev);
	fprintf(stderr, "Done\n");

#ifdef SERIAL
cleanup_serial:
	fprintf(stderr, "Closing UART...");
	fflush(stderr);
	serial_close(fd);
	fprintf(stderr, "Done\n");
#endif

	return retval;
}

// SIGINT handler, set the flag to false to stop the loop
void intHandler(int unused) {
	flag = 0;
}

#ifdef DEBUG
// Debug, save vector to file
static void print_vector(int cycle, float *vect, char * filename, unsigned int length){
	char str[64];
	FILE *fp;
	sprintf(str, "%s/%s_%d", DEBUGDIR, filename, cycle);
	fp = fopen(str, "w+");
	if (fp != NULL) {
		for (int i = 0; i < length; i++) {
			fprintf(fp, "%lf\n", vect[i]);
		}
		fclose(fp);
	}
}
#endif
