#include "misc.h"

/* Assume that in a captured frame there is always at least one complete
 * triplet (narrow black band, narrow white, medium black, narrow white, large
 * black, narrow white).
 * The width of one triplet is known since it is defined in the postscript file
 * in millimeters.
 * The factor is computed as the number of pixels of a full triplet, obtained
 * counting 7 edges, over the defined width of the triplet.
 * Hysteresis is used to check the transitions because, despite the filtering
 * of the sum of the columns, the transitions of the signal can have a
 * non-monotone behavior.
 * The number of peaks is also returned, which is used by the main program to
 * compute the threshold displacement for the reference image.
 * Returns 0 in case of success and 1 in case of error;
 */
int pixel_to_mm_factor(float * vect, float * fact, unsigned int * peakcount) {

	unsigned int i, startid, stopid, count;
	float * p, minval, maxval, thresholdrise, thresholdfall;

	// Variable used to keep track if the current element is in a white (peak)
	// or black band
	unsigned char iswhiteband;

	// First find the max value. 50% of the max value is used as a threshold
	// value for the rise/fall transitions
	p = vect;
	maxval = 0;
	minval = 3.4e38;
	for (i = 0; i < RESX; i++) {
		if (*p > maxval) maxval = *p;
		if (*p < minval) minval = *p;
		p++;
	}

	// Thresholds to identify transitions
	thresholdrise = minval + (maxval - minval) * 0.55;
	thresholdfall = minval + (maxval - minval) * 0.35;

	// Iterate the vector counting the transitions and the peaks
	count = 0;
	p =	vect;
	*peakcount = 0;
	iswhiteband = (*p > thresholdrise) ? 1 : 0;
	for (i = 0; i < RESX-1; i++) {

		// Rising edge
		if (*p <= thresholdrise && *(p+1) > thresholdrise && !iswhiteband) {
			iswhiteband = 1;

			if (!count) startid = i;
			if (count == 6) stopid = i;
			count++;

		// Falling edge
		} else if (*p >= thresholdfall && *(p+1) < thresholdfall && iswhiteband) {
			(*peakcount)++;
			iswhiteband = 0;

			if (!count) startid = i;
			if (count == 6) stopid = i; 
			count++;

		// End of frame. If the frame finishes before the last white band it is
		// counted as a peak too
		} else if (i == RESX-1 && iswhiteband) {
			(*peakcount)++;
		}

		p++;
	}

	// Not enough transitions to evaluate the factor to convert pixels to mm
	if (count < 7) {
		*fact = 0;
		return 1;
	}

	// Evaluate the factor
	*fact = ((float) TRIPLETWIDTH) / ((float)(stopid - startid));
	
	return 0;
}

/* Compute and apply the threshold to the frame vector
 */
void threshold_apply(float * vect) {
	float min, max, th, *p;

	// Compute the threshold as the mean value between the max and the min
	max = 0;
	min = 3.4e38;
	p = vect;
	for (unsigned int i = 0; i < RESX; i++) {
		if (*p > max) max = *p;
		if (*p < min) min = *p;
		p++;
	}
	th = min + (max - min) * 0.75;

	// Apply the threshold to the vector
	p = vect;
	for (unsigned int i = 0; i < RESX; i++) {
		*p = (*p > th) ? *p : 0;
		p++;
	}
}
		
