#ifndef FFT_H
#define FFT_H

/**
 * @file fft.h
 * @author Luca Favario
 * @date Aug 2021
 * @brief FFT related computations
 */

#include <math.h>
#include <fftw3.h>
#include "common.h"

/**
 * @brief Allocate a real vector for FFTW. This ensures that the alignment is
 * correct and that SIMD is used.
 * @param n Number of elements in the vector
 * @return float pointer to the vector
 */
float * fft_malloc_r(unsigned int n);

/**
 * @brief Allocate a complex vector for FFTW. This ensures that the alignment
 * is correct and that SIMD is used.
 * @param n Number of elements in the vector
 * @return fftw_complex pointer to the vector
 */
fftwf_complex * fft_malloc_c(unsigned int n);

/**
 * @brief Free the allocated vector.
 * @param v vector to be freed
 */
void fft_free(void ** v);

/**
 * @brief Returns the fftwf plan for the forward fft. 
 *
 * Wrapper around fftwf_plan_dft_r2c_1d real to complex 1D fft.
 *
 * @param in input real array spacial domain
 * @param out output complex array frequency domain
 * @return fftwf plan to compute the fft
 */
fftwf_plan fft_init_fw(float *in, fftwf_complex *out);

/**
 * @brief Returns the fftwf plan for the inverse fft. 
 *
 * Wrapper around fftwf_plan_dft_c2r_1d complex to real 1D fft.
 * This plan can be used because the complex vector satisfies the Hermitian
 * condition.
 *
 * @param in input complex array frequency domain 
 * @param out output real array spatial domain
 * @return fftwf plan to compute the inverse fft
 */
fftwf_plan fft_init_bw(fftwf_complex *in, float *out);

/**
 * @brief Compute the cross correlation between two vector and apply padding to
 * the resulting vector when required (ZPUPS > 1).
 *
 * Since the two image vectors satisfy the Hermitian condition, then the cross
 * correlation satisfies it too. For this reason only half of the vector is
 * computed.
 * After the cross correlation the zero padding is applied with upsampling
 * factor ZPUPS.
 *
 * @param v1 Input complex array, Fourier image vector 1
 * @param v2 Input complex array, Fourier image vector 2
 * @param out First half vector of the cross correlation in frequency domain
 */
void fft_xcorr(fftwf_complex *v1, fftwf_complex *v2, fftwf_complex *out);

/**
 * @brief Return the displacement of the maximum of the cross correlation in
 * the spacial domain with respect to the center.
 *
 * When computing the displacement takes into account the fact that the cross
 * correlation vector has the first half and the second half swapped (due to
 * fft and cross correlation operation).
 * Moreover only the displacement interval -deltarange:+deltarange is
 * considered. This solution allows to solve the problems where a wrong higher
 * peak is found further than the correct one due to the cyclic behavior of the
 * cross correlation.
 * The result is returned in upsampled pixels if padding is applied (ZPUPS).
 *
 * @param in Input cross correlation in spacial domain.
 * @param deltarange Maximum displacement allowed with respect to the reference
 * frame.
 * @param index Float pointer used to return the computed displacement in
 * upsampled pixels if zero-padding is used(ZPUPS)
 * @return 0 in case of success, 1 in case of error
 */
int fft_displacement(float *in, unsigned int deltarange, float *index);

/**
 * @brief Execute the inverse fft using the given fftwf plan
 *
 * Wrapper around fftwf_execute(p). There is no need to specify the output
 * since there is only one vector to which this is applied.
 *
 * @param p Fftw plan to be executed
 */
void fft_execute_bw(fftwf_plan p);

/**
 * @brief Execute the fft using the given fftwf plan
 *
 * Wrapper of the fftwf function fftwf_execute_dft.
 * The input and output have to be specified because the same plan can be used
 * on the vector of reference or current frame.
 *
 * @param p Fftw plan to be executed
 * @param in Input real vector of an image
 * @param out Output complex vector Fourier transform of the input
 */
void fft_execute_fw(fftwf_plan p, float *in, fftwf_complex *out);

/**
 * @brief Inverse DFT, compute the upsampled displacement in the window around
 * the peak.
 *
 * This function uses the inverse DFT to compute the interpolated spatial cross
 * correlation in a neighborhood of the peak.
 * Theory concepts are explained in the report.
 *
 * @param cross Cross correlation in frequency domain
 * @param delta Displacement in pixel units computed without upsampling using
 * fft_displacement
 * @param win_delta Return value, displacement in upsampled units
 * @return 0 success, 1 in case of error
 */
int fft_dft_bw(fftwf_complex *cross, float delta, float *win_delta);

/**
 * @brief Destroy a fftwf plan 
 *
 * Wrapper of the fftwf function fftwf_destroy_plan
 *
 * @param p Fftw plan to be destroyed
 */
void fft_destroy(fftwf_plan *p);

/**
 * @brief Read fftwf wisdom
 *
 * Read the fftwf wisdom file. If it is available, then it largely decrease the
 * plan creation time. 
 * It is a wrapper of fftwf function fftwf_import_wisdom_from_filename.
 * Filename defined in config.h
 *
 * @return Integer different from zero in case of success
 */
int fft_read_wisdom();

/**
 * @brief Save fftwf wisdom
 *
 * Write the fftwf wisdom file.
 * It is a wrapper of fftwf function fftwf_export_wisdom_to_filename.
 * Filename defined in config.h
 *
 * @return Integer different from zero in case of success
 */
int fft_save_wisdom();

#endif
