#ifndef COMMON_H 
#define COMMON_H

#include "config.h"

/**
 * @file common.h
 * @author Luca Favario
 * @date Sep 2021
 * @brief Common definitions used by different parts of the program. This file
 * should not be edited by the user.
 * To enable DEBUG, BENCHMARK and FILECAMERA use the make file.
 */

/******************************
 *       DO NOT EDIT          *
 ******************************/

/******************************
 *          DEBUG             * 
 ******************************/
/**
 * @def DEBUG
 * @brief Enable debug output and messages. Automatically disabled by BENCHMARK
 */
//#define DEBUG

/**
 * @def DEBUG_FFT
 * @brief Additional files are written with FFT computation results. Allows to
 * easily plot the computed vectors.
 */
//#define DEBUG_FFT

/**
 * @def DEBUGDIR
 * @brief Directory where logs and data is written.
 */
#define DEBUGDIR "/tmp/"


/******************************
 *          BENCHMARK         * 
 ******************************/
/** 
 * @def BENCHMARK
 * @brief flag to enable benchmark mode: loop only BENCHN times and print
 * computation times in an easy to parse format in the file
 * "bench_RESXxRESY_ZPUPS_UPS". 
 * DEBUG is disabled if this macro is defined, THREAD should be disabled to get
 * the full data.
 */
//#define BENCHMARK 

/**
 * @def BENCHN
 * @brief How many times the main loop iterates in benchmark mode
 */
#define BENCHN 20

/**
 * @def BENCHDIR
 * @brief Directory where the benchmark logs should be saved
 */
#define BENCHDIR "/tmp/"

/******************************
 *       VIDEO FILE           * 
 ******************************/
/**
 * @def FILECAMERA
 * @brief Read the frames from a video file instead of from the camera.
 * Automatically disables THREAD, since threads not implemented in
 * camera_file.h
 * This mode requires to compile with camera_file.h instead of camera.h,
 * automatically done with `make file`.
 * The frames are read and transmitted to this program with a named pipe. The
 * transmitter is the python program frames2pipe.py.
 */
//#define FILECAMERA

/**
 * @def FILEBUFSIZE
 * @brief Size in bytes of the buffer used to communicate on the named pipe
 * with the python program. 
 */
#if defined (BRESX) && defined (BENCHMARK)
#	define FILEBUFSIZE BRESX
#else
#	define FILEBUFSIZE RESX
#endif

/******************************
 *     COMPILATION OPTIONS    *
 ******************************/

// If benchmark is set disable DEBUG and the pixel conversion
#ifdef BENCHMARK
#	undef DEBUG
#	undef PIXTOMMOUT
#endif

// DEBUG_FFT is enabled only if DEBUG is enabled too
#ifndef DEBUG
#	undef DEBUG_FFT
#endif

// Benchmark update resolution and upsample with command line arguments
#if defined (BRESX) && defined (BENCHMARK)
#	undef RESX
#	define RESX BRESX
#endif

#if defined (BRESY) && defined (BENCHMARK)
#	undef RESY
#	define RESY BRESY
#endif

#if defined (BUPS) && defined (BENCHMARK)
#	undef UPS
#	define UPS BUPS
#endif

// Disable threads if the sensor_file is being compiled. Threads are not
// implemented.
#ifdef FILECAMERA
#	undef THREAD
#endif

/******************************
 *           FFT              * 
 ******************************/
/**
 * @def ZPUPS 
 * @brief Upscaling factor used by the zero padding which is applied before the
 * inverse FFT (before the DFT). 
 * This allows to identify the delta used to shift the window with better
 * precision and to reduce the width of the window, hence improving the
 * performance in certain conditions.
 * For UPS < 32 There is no real advantage in using the zero padding, so it is
 * fine to set ZPUPS to 1 and reduce memory usage.
 * When increasing UPS >= 32 the benefit becomes larger because many points
 * have to be computed by the DFT and using a narrower window has a big effect.
 * If there are not strict memory constraint a value of 4 allows to reduce the
 * complexity with larger UPS, at the cost of 4x the memory required by the
 * cross correlation in both spacial and frequency domain.
 * On the other hand larger values of ZPUPS do not provide a real performance
 * benefit, while they heavily impact the used memory.
 * In benchmark mode ZPUPS is overwritten by the shell variable.
 */
#if defined (BZPUPS) && defined (BENCHMARK)
#	define ZPUPS BZPUPS
#else
#	if UPS < 32
#		define ZPUPS 1
#	else
#		define ZPUPS 2
#	endif
#endif

/** 
 * @def WIN
 * @brief Size of the window in upsampled pixels. The constant factor can be
 * smaller with larger ZPUPS.
 * The higher ZPUPS, the narrower the window is allowed. The smallest value
 * which are found to correctly center the window are: 
 *  - ZPUPS=1 => UPS*2
 *  - ZPUPS=2 => UPS*1
 *  - ZPUPS=4 => UPS*0.5
 */
#if ZPUPS == 1
#	define WIN ((int)(UPS*2))
#elif ZPUPS == 2
#	define WIN ((int)(UPS*1))
#endif

/**
 * @def DELTARANGEFACT
 * @brief Factor that, multiplied by the threshold value for the reference
 * image displacement deltath, is used to compute the maximum allowed
 * displacement deltarange.
 */
#define DELTARANGEFACT 1.3

/** 
 * @dev ZEROTH
 * @brief If defined then the frame vectors are compared against a threshold
 * and the values below the threshold are set to 0.
 * The threshold is defined as the mean between the min and the max of the
 * signal.
 * Setting the lower part of the signal to 0 reduces the effect of the black
 * bands when computing the cross correlation.
 */
//#define ZEROTH

/******************************
 *          CAMERA            * 
 ******************************/
/**
 * @def BUFN
 * @brief Number of buffers to be used by the USB camera.
 * Recommended values: 
 *	- 2 for non-threaded version
 *	- 4 for threaded
 */
#ifdef THREAD
#	define BUFN 4
#else
#	define BUFN 2
#endif

/**
 * @def SUMROWN
 * @brief Number of rows to sum when computing the sum of the columns of the
 * frame in the frame vector.
 */
#define SUMROWN RESY >> 3


/******************************
 *          SENSOR            * 
 ******************************/
/**
 * @def DELTATHFALLBACK
 * @brief Fallback displacement threshold to update the reference frame.
 * The displacement threshold is computed using the number of peaks in the
 * frame (white bands) as deltath = RESX/peakcount. In case deltath is larger
 * than DELTATHFALLBACK, which is the case when there are too few peaks in the
 * frame, then deltath is set to DELTATHFALLBACK which is a known safe value.
 */
#define DELTATHFALLBACK (RESX/4)

/**
 * @def DELTATHFACT 
 * @brief Factor multiplied by the average peak distance in the reference frame
 * to compute the reference frame displacement threshold.
 */
#define DELTATHFACT 0.8

/******************************
 *       EXTRA                *
 ******************************/

/** @def RESX2
 * @brief Define half of the horizontal resolution
 */
#define RESX2 ((unsigned int) RESX >> 1)

/** @def ZPRU
 * @brief Horizontal resolution multiplied by the zero-padding upsampling factor. 
 * This value corresponds to the length of the padded cross correlation in
 * frequency and spatial domains.
 */
#define ZPRU ((unsigned int)((unsigned int)RESX * (unsigned int)ZPUPS))

/** @def ZPRU2
 * @brief Define half of ZPRU
 */
#define ZPRU2 ((unsigned int)(ZPRU >> 1))

#endif
