#include "camera_file.h"
#define TESTFILE "/tmp/image"
#define FIFO "/tmp/fifopipe"

/* This file implements an simulation of a camera interface in which the data
 * is received as parallel data and the data can be read by the CPU before the
 * whole frame is available in memory.
 * In this case a named pipe is used to communicate with a python program, which
 * reads the frame previously extracted by a video and send them to this
 * program with the pipe.
 * This method has a memory advantage with respect to saving the whole frame to
 * memory since there is no need to save one or more frames in memory. It could
 * be used with modules which have a i2c interface to configure the camera and a
 * parallel port which is used to transfer the raw data (OV7670, OV2640).
 * Using DMA the data can be written to a small buffer and accessed by the
 * processor before the whole complete frame is available, and there is no need
 * to store the complete frame in memory.
 * In this implementation 1 or more bytes can be read in a buffer (get_buffer),
 * which is then read to compute the sum of the columns (camera_get_frame).
 * The buffer in case of the DMA would be filled automatically and a thread
 * could be used to handle the buffers and sum the columns.
 * Note: in this implementation only RAW gray scale frames are supported.
 */

// Fifo file descriptor
static int fifo_fd;

// Get FILEBUFSIZE bytes from buffer
static int get_buffer (unsigned char *ch) {
	return read(fifo_fd, ch, FILEBUFSIZE);
}

int camera_init(struct camera_dev_t *dev) {
	// Create serial fifo
	if (mkfifo(FIFO, 0600) < 0) {
		if (errno != EEXIST) {
			return -1;
		}
	}
	
	return 0;
}

int camera_start(struct camera_dev_t *dev) {
	// Open reading serial fifo
	if ((fifo_fd = open(FIFO, O_RDONLY)) < 0) 
		return 1;

	return 0;
}

void camera_stop(struct camera_dev_t *dev) {
	// Close serial fifo
	if (fifo_fd) {
		close(fifo_fd);
		fifo_fd = 0;
	}
}

void camera_close(struct camera_dev_t *dev) {
}

/* Read the gray scale raw frame from the buffer, FILEBUFSIZE bytes at each
 * iteration.
 * The sum of the columns is computed as data is available instead of waiting
 * for the whole frame to be available in memory. 
 */
int camera_get_frame(struct camera_dev_t *dev, float **vectptr) {
	unsigned int tmp_vect[RESX], *vint;
	// Buffer
	unsigned char ch[FILEBUFSIZE], *chptr;
	float *v;

	// First row
	vint = tmp_vect;
	for (int x = 0; x < RESX; x+=FILEBUFSIZE) {
		if (!get_buffer(ch)) 
			return -1;
		chptr = ch;
		for (int i = 0; i < FILEBUFSIZE; i++) {
			*vint = *(chptr++);
			vint++;
		}
	}

	// Middle rows
	for (int y = 1; y < RESY-1; y++) {
		vint = tmp_vect;
		for (int x = 0; x < RESX; x+=FILEBUFSIZE) {
			if (!get_buffer(ch)) 
				return -1;
			if (y < SUMROWN) {
				chptr = ch;
				for (int i = 0; i < FILEBUFSIZE; i++) {
					*vint += *(chptr++);
					vint++;
				}
			}
		}
	}

	// Last row
	v =	*vectptr;
	vint = tmp_vect;
	for (int x = 0; x < RESX; x+=FILEBUFSIZE) {
		if (!get_buffer(ch)) 
			return -1;
		chptr = ch;
		for (int i = 0; i < FILEBUFSIZE; i++) {
			*(v++) = *(vint++) + *(chptr++);
		}
	}

	return 0;
}
