#ifndef CAMERA_H
#define CAMERA_H

/**
 * @file camera.h
 * @author Luca Favario
 * @date Aug 2021
 * @brief Camera capture and frame vector computation.
 */

#include <libv4l2.h>
#include <linux/videodev2.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <string.h>
#include <turbojpeg.h>
#include <fcntl.h>
#include <sys/time.h>
#include <errno.h>
#include "common.h"

#ifdef THREAD
#include <pthread.h>
#endif

#ifdef BENCHMARK
#include "benchmark.h"
#endif

/**
 * @def YUYV 
 * @brief YUYV422 raw format for camera capture, used in common.h
 */
#define YUYV 0

/**
 * @def MJPEG 
 * @brief JPEG compressed format for camera capture, used in common.h 
 */
#define MJPEG 1

/** 
 * @brief Struct used to store the address and the length of the buffers
 */
struct camera_buffer {
	/** @var start 
	 * Pointer to the start of the buffer */
	void *start;
	/** @var length 
	 * Length of the buffer */
	unsigned int length;
};

/** 
 * @brief Struct contaning mutex for multithreading, file descriptor of the
 * camera device and buffers information.
 */
struct camera_dev_t {

#ifdef THREAD
	/** @var mutex 
	 * Mutex to lock access to the buffer shared between the threads
	 */
	pthread_mutex_t mutex;
	/** @var mutex_sync 
	 * This mutex is unlocked when a buffer is made available by the refreshing
	 * thread
	 */
	pthread_mutex_t mutex_sync;
	/** @var th 
	 * Thread id of the thread which refresh the stored frames */
	pthread_t th;
	/** @var bufs
	 * Buffers which are exchanged between the two threads
	 */
	struct v4l2_buffer bufs[3];
	/** @var newbuf
	 * Pointer the the last extracted frame by the updating thread
	 */
	struct v4l2_buffer *newbuf;
	/** @var currbuf
	 * Pointer to the buffer used by the main thread
	 */
	struct v4l2_buffer *currbuf;
	/** @var shbuf
	 * Pointer to the buffer shared between the threads
	 */
	struct v4l2_buffer *shbuf;
#endif

	/** @var fd 
	 * File descriptor of the camera device */
	int fd;
	/** @var running 
	 * Flag used to stop the endless loops */
	int running;

#if FORMAT == MJPEG
	/** @var tjInstance 
	 * Turbojpeg decompression object */
	tjhandle tjInstance;
#endif

	/** @var buffers 
	 * Buffers information */
	struct camera_buffer buffers[BUFN];
};

/** 
 * @brief Open and setup the video device 
 * @param dev fd and buffers are updated
 * @return 0 if successful, negative in case of error
 */
int camera_init(struct camera_dev_t *dev);

/**
 * @brief Start the stream of the camera as well as the frame updating thread
 * if THREAD is defined
 * @param dev contains mutex and file descriptor
 * @return 0 if success, negative if error
 */
int camera_start(struct camera_dev_t *dev);

/**
 * @brief Stop the stream of the camera and the frame updating thread if THREAD
 * is defined.
 * @param dev contains mutex and file descriptor
 * @return 0 if success, negative if error
 */
void camera_stop(struct camera_dev_t *dev);

/**
 * @brief Uninitialize and close the video device
 * @param dev fd and buffers
 * @return 0 if success, negative if error
 */
void camera_close(struct camera_dev_t *dev);

/**
 * @brief Get frame, decompress it (in case of MJPEG), sum the columns into a vector.
 * @param dev File descriptor, mutex and buffers
 * @param vectptr pointer to sum of the columns vector
 * @return 0 if success, negative if error
 */
int camera_get_frame(struct camera_dev_t *dev, float **vectptr);

#endif
