#include "camera.h"

/* Total number of pixels in the image
 */
#define RESXY ((int) RESX * RESY)

/* In case of MJPEG format the sum of the columns is computed using the
 * decompressed image. In this case the matrix is grayscale and the width of
 * the matrix is equal to the width of the vector and the horizontal
 * resolution, hence the sum of each column is computed defining DELTA = 1.
 * In case of YUYV (yuyv422) format the frame is in raw format, but it is not
 * grayscale. The image is converted to grayscale when computing the sum of the
 * clumns by ignoring the columns which contain color information and keeping
 * only the luminance values.
 * The data in yuyv422 format is:
 *	Y Cb Y Cr Y Cb 
 * Where Y are luminance values and Cx are chrominance values.
 * To compute the grayscale vector only the sum of Y columns is computed, for
 * this reason DELTA is set to 2.
 */
#if FORMAT == MJPEG
#define DELTA 1
#elif FORMAT == YUYV 
#define DELTA 2
#endif

#if defined DEBUG 
static void camera_save_frame(void *bufstart, void *filename);
#endif

#ifdef THREAD
static void* camera_buffers_thread(void *devp);
#endif

static void camera_process_frame(struct camera_dev_t *dev, unsigned char *matrix, unsigned int bytesused, float *vect);

/* Open and setup the video device 
 *	- Open file descriptor
 *	- Set video format
 *	- Inform the device about buffer
 *	- Allocate buffer
 */
int camera_init(struct camera_dev_t *dev) {
	int fd;
	unsigned int i;
	struct v4l2_buffer buf;
	struct v4l2_format format;
	struct v4l2_requestbuffers bufreq;

	// Open file descriptor of the camera
	if ((fd = open(DEVICE, O_RDWR )) < 0) {
		return -1;
	}

	// Update dev struct file descriptor
	dev->fd = fd;

	// Set video format
	format.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
#if FORMAT == YUYV
	format.fmt.pix.pixelformat = V4L2_PIX_FMT_YUYV;
#elif FORMAT == MJPEG
	format.fmt.pix.pixelformat = V4L2_PIX_FMT_MJPEG;
	dev->tjInstance = NULL;
#endif
	format.fmt.pix.width = RESX;
	format.fmt.pix.height = RESY;
	if (ioctl(fd, VIDIOC_S_FMT, &format) < 0) {
		return -2;
	}
	
	// Inform the device about the buffers 
	bufreq.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	bufreq.memory = V4L2_MEMORY_MMAP;
	bufreq.count = BUFN;
	if (ioctl(fd, VIDIOC_REQBUFS, &bufreq) < 0) {
		return -3;
	}
	
	// Allocate the buffers
	for (i = 0; i < BUFN; i++) {
		memset(&buf, 0, sizeof(buf));
		buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buf.memory = V4L2_MEMORY_MMAP;
		buf.index = i;
		if (ioctl(fd, VIDIOC_QUERYBUF, &buf) < 0) {
			return -4;
		}

		// Memory mapping the buffer
		(dev->buffers[i]).start = mmap(NULL, buf.length, PROT_READ | PROT_WRITE, MAP_SHARED, fd,
						buf.m.offset);
		if (dev->buffers[i].start == MAP_FAILED) {
			return -5;
		}
		memset(dev->buffers[i].start, 0, buf.length);
		dev->buffers[i].length = buf.length;
	}

	return 0;
}


/* Start the camera stream
 *	- QBUF exchange buffer with the driver
 *	- STREAMON start the capture
 * If thread is enabled then initialize the mutexes and start the thread.
 */
int camera_start(struct camera_dev_t *dev) {
	enum v4l2_buf_type type;
	struct v4l2_buffer buf;
	unsigned int i;

	// Put the buffers in the incoming queue
	for (i = 0; i < BUFN; i++) {
		memset(&buf, 0, sizeof(buf));
		buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buf.memory = V4L2_MEMORY_MMAP;
		buf.index = i;
		if (ioctl(dev->fd, VIDIOC_QBUF, &buf) < 0) {
			return -1;
		}
	}

	// Start the capture
	type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (ioctl(dev->fd, VIDIOC_STREAMON, &type) < 0) {
		return -2;
	}

#if FORMAT == MJPEG
	// Initializing jpeg decompressor
	dev->tjInstance = tjInitDecompress();
#endif

#ifdef THREAD

	// Initialize the buffer pointers
	dev->currbuf = &dev->bufs[0];
	dev->shbuf = &dev->bufs[1];
	dev->newbuf = &dev->bufs[2];

	// Initialize newbuf
	memset(dev->newbuf, 0, sizeof(*(dev->currbuf)));
	dev->newbuf->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	dev->newbuf->memory = V4L2_MEMORY_MMAP;

	// Dequeue a buffer for the first currbuf
	// Initialize the buffer
	memset(dev->currbuf, 0, sizeof(*(dev->currbuf)));
	dev->currbuf->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	dev->currbuf->memory = V4L2_MEMORY_MMAP;
	// Dequeue buffer
	if (ioctl(dev->fd, VIDIOC_DQBUF, dev->currbuf) == -1) {
		return -1;
	}

	// Dequeue a buffer for the first shbuf 
	// Initialize the buffer
	memset(dev->shbuf, 0, sizeof(*(dev->shbuf)));
	dev->shbuf->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	dev->shbuf->memory = V4L2_MEMORY_MMAP;
	// Dequeue buffer
	if (ioctl(dev->fd, VIDIOC_DQBUF, dev->shbuf) == -1) {
		return -1;
	}

	// Update buffers thread
	pthread_attr_t attr;
	dev->running = 1;
	pthread_attr_init(&attr);
	// Init mutexes
	pthread_mutex_init(&(dev->mutex), NULL);
	pthread_mutex_init(&(dev->mutex_sync), NULL);
	pthread_mutex_lock(&(dev->mutex_sync));
	// Create thread
	pthread_create(&(dev->th), &attr, camera_buffers_thread, dev);
#endif

	return 0;
}

/* Stop the camera stream and the thread
 */
void camera_stop(struct camera_dev_t *dev) {
	enum v4l2_buf_type type;

#ifdef THREAD
	// Destroy the mutex
	pthread_mutex_destroy(&(dev->mutex));
	pthread_mutex_destroy(&(dev->mutex_sync));
	
	// Wait for the thread to exit
	dev->running = 0;
	pthread_join(dev->th, NULL);
#endif

#if FORMAT == MJPEG
	// Destroy turbojpeg transformer
	if (dev->tjInstance) {
		tjDestroy(dev->tjInstance);
		dev->tjInstance = NULL;
	}
#endif

	// Stop the capture
	type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	ioctl(dev->fd, VIDIOC_STREAMOFF, &type); 
}

/* Uninitialize and close the video device
 */
void camera_close(struct camera_dev_t *dev) {
	unsigned int i;

	// Unmap the buffers
	for (i = 0; i < BUFN; i++) {
		munmap(dev->buffers[i].start, dev->buffers[i].length);
	}

	// Close device
	close(dev->fd);
}


/* Get a frame and process the image.
 * Sum the columns of a frame in a vector. In case of jpeg the frame is
 * converted to grayscale and decompressed using turbojpeg.
 * If THREAD is defined then update the new frame using the vector double
 * buffers.
 */
int camera_get_frame(struct camera_dev_t *dev, float **vectptr) {

#ifndef THREAD
	
	struct v4l2_buffer buf;

	// Dequeue a buffer 
	memset(&buf, 0, sizeof(buf));
	buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	buf.memory = V4L2_MEMORY_MMAP;

	// Dequeue buffer
	if (ioctl(dev->fd, VIDIOC_DQBUF, &buf) == -1) {
		return -1;
	}

	// Sum frame columns and decompress when format is MJPEG
	camera_process_frame(dev, dev->buffers[buf.index].start, buf.bytesused, *vectptr);

	// Requeue the buffer when the vector is computed
	if (ioctl(dev->fd, VIDIOC_QBUF, &buf) == -1) {
		return -1;
	}
	
#else 
	struct v4l2_buffer *tmp;

	if (!dev->running) return -1;

	// Wait for a buffer to be available
	pthread_mutex_lock(&(dev->mutex_sync));

	// Lock while the pointers are swapped. Single access to shbuf
	pthread_mutex_lock(&(dev->mutex));

	// Swap the buffer used to compute the displacement and the buffer updated
	// by the buffers thread
	tmp = dev->currbuf;
	dev->currbuf = dev->shbuf;
	dev->shbuf = tmp;

	// Unlock access to shbuf
	pthread_mutex_unlock(&(dev->mutex));

	// Sum frame columns and decompress when format is MJPEG
	camera_process_frame(dev, dev->buffers[dev->currbuf->index].start, dev->currbuf->bytesused, *vectptr);

#endif

	return 0;
}


#ifdef THREAD
/* Camera thread, constantly dequeue and queue buffers.
 */
static void* camera_buffers_thread(void *devp) {
	struct camera_dev_t *dev;
	struct v4l2_buffer *tmp;

	// Pointer to camera_dev_t struct
	dev = (struct camera_dev_t *) devp;

	// Thread loop to update the frames
	while (dev->running) {

		// Dequeue new v4l2 buffer
		if (ioctl(dev->fd, VIDIOC_DQBUF, dev->newbuf) == -1) {
			dev->running = 0;
			pthread_mutex_unlock(&(dev->mutex_sync));
			pthread_mutex_unlock(&dev->mutex);
			pthread_exit(NULL);
		} 
		
		// Lock access to shbuf
		pthread_mutex_lock(&(dev->mutex));

		// Re-queue old used buffer
		if (ioctl(dev->fd, VIDIOC_QBUF, dev->shbuf) == -1) {
			dev->running = 0;
			pthread_mutex_unlock(&(dev->mutex_sync));
			pthread_mutex_unlock(&(dev->mutex));
			pthread_exit(NULL);
		}

		// swap the pointer of the old buffer shbuf with the one of the new
		// buffer newbuf
		tmp = dev->shbuf;
		dev->shbuf = dev->newbuf;
		dev->newbuf = tmp;

		// Unlock synchronization mutex, new buffer is available in shbuf
		pthread_mutex_unlock(&(dev->mutex_sync));

		// Unlock access to shbuf
		pthread_mutex_unlock(&(dev->mutex));

	}

	return NULL;
}
#endif


/* Process the frame captured by the camera, as stored in the last dequeued
 * buffer. 
 * If the format is MJPEG it is decompressed first.
 * Then the columns of the matrix are summed into a vector.
 * If DEBUG is defined then the frame is saved and timing
 * information about the sum of the columns and the decompression are printed.
 */
static void camera_process_frame(struct camera_dev_t *dev, unsigned char *matrix, unsigned int bytesused, float *vect) {
	float *vf;
	unsigned int x, y, *v, tmp_vect[RESX];
	unsigned char *p;

#if defined DEBUG 
	// Structs to compute timing
	struct timeval start, stop;
#endif

#if FORMAT == YUYV
	// Frame pointer
	p = matrix;

#elif FORMAT == MJPEG
	unsigned char matdc[RESXY];

#if defined DEBUG 
	// Decompress start time
	gettimeofday(&start, NULL);
#elif defined BENCHMARK 
	bench_start_time();
#endif

	// Decompressing jpeg image
	tjDecompress2(dev->tjInstance, matrix, bytesused, matdc, RESX, 0, RESY, TJPF_GRAY, 0);

	// Frame pointer
	p = matdc;

#if defined DEBUG 
	// Decompress stop time
	gettimeofday(&stop, NULL);
	fprintf(stderr, "Decompress time: %f\n", (stop.tv_sec - start.tv_sec) + (stop.tv_usec - start.tv_usec) / 1e6);
#elif defined BENCHMARK 
	bench_stop_time();
	bench_write_decompress();
#endif

#endif

	// Sum the columns
	// Integer vector pointer
	v =	tmp_vect;
	// Result vector pointer
	vf = (float *) vect;

#if defined DEBUG 
	// Save the frame captured by the camera
	camera_save_frame((void*) p, "/tmp/frame.pgm");
	// Sum columns start time
	gettimeofday(&start, NULL);
#elif defined BENCHMARK 
	bench_start_time();
#endif

	// Init integer vector with first row values
	for (x = 0; x < RESX; x++) {
		*(v++) = *p;
		p += DELTA;
	}

	// Add the columns as integers into the temporary vector
	// Sum only SUMROWN rows
	v = tmp_vect;
	for (y = 1; y < SUMROWN; y++) {
		for (x = 0; x < RESX; x++) {
			*(v++) += *p;
			p += DELTA;
		}
		v = tmp_vect;
	}

	// Convert to float
	v = tmp_vect;
	for (x = 0; x < RESX; x++) {
		*(vf++) = (float) *(v++);
	}

#if defined DEBUG 
	// Sum columns stop time
	gettimeofday(&stop, NULL);
	fprintf(stderr, "Sum columns time: %f\n", (stop.tv_sec - start.tv_sec) + (stop.tv_usec - start.tv_usec) / 1e6);
#elif defined BENCHMARK 
	bench_stop_time();
	bench_write_sum();
#endif

}

#if defined DEBUG 
/* Save the frame to a pgm image in the filename location
 */
static void camera_save_frame(void *bufstart, void *filename) {
	FILE *fp =	fopen((char*)filename, "w+");

	int i;
	fprintf(fp, "P5\n%d %d\n255\n", RESX, RESY);

#if FORMAT == YUYV
	// Convert to grayscale and write pgm
	for (i = 0; i < (RESXY << 1); i+=2) 
		fprintf(fp, "%c", *((unsigned char*)bufstart + i));

#elif FORMAT == MJPEG
	// Already decompressed image
	for (i = 0; i < RESXY; i++) 
		fprintf(fp, "%c", *((unsigned char*)bufstart + i));

#endif

	fclose(fp);

}
#endif
