#include "benchmark.h"

/* File descriptor for the log file
 */
static FILE *bench_desc;

/* Flag used to enable or disable the logging
 */
static unsigned char running;

/* Timing structures
 */
static struct timeval times[2], totaltimes[2];

/* Open the benchmark file
 */
int bench_open() {
	struct stat st;
	char filename[64];

	// Check if the path exists
	if (stat(BENCHDIR, &st) == -1) {
		if (errno == ENOENT)
			fprintf(stderr, "Path does not exists: %s\n", BENCHDIR);
		else
			fprintf(stderr, "Could not stat %s\n", BENCHDIR);
		return 1;
	} 

	// Check if the path is a directory
	if ((st.st_mode & S_IFMT) != S_IFDIR) {
		fprintf(stderr, "The given path is not a directory: %s\n", BENCHDIR);
		return 1;
	}

	// Open BENCHDIR_RESXxRESY_UPS
	sprintf(filename, "%s/bench_%dx%d_%d_%d", BENCHDIR, RESX, RESY, ZPUPS, UPS);
	if ((bench_desc = fopen(filename, "w")) == NULL) {
		fprintf(stderr, "Error trying to open the file: %s\n", filename);
		return 1;
	}

	// Start with logging disabled
	running = 0;

	return 0;
}

// Write the average and max values of each timing field at the end of the
// file, then close the file
int bench_close() {
	char filename[64];
	double mean[3];
	float max[3], tmp[3], val;

	if (bench_desc == NULL) return 1;

	// Disable logging
	running = 0;

	// Close file writing mode
	if (fclose(bench_desc) == EOF) {
		fprintf(stderr, "Error closing the file: %s\n", BENCHDIR);
		return 1;
	}

	// Reopen the file in reading mode
	sprintf(filename, "%s/bench_%dx%d_%d_%d", BENCHDIR, RESX, RESY, ZPUPS, UPS);
	if ((bench_desc = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Error trying to open the file: %s\n", filename);
		return 1;
	}

	// Init max and mean values
	for (int j = 0; j < 3; j++) {
		max[j] = 0;
		mean[j] = 0;
	}

	// Scan the file, reading the timing values and updating the mean and max
	// values.
	for (int i = 0; i < BENCHN; i++) {
		unsigned int col_count;

		if (!(col_count = fscanf(bench_desc, "%f%*[ ]%f%*[ ]%f", tmp, tmp+1, tmp+2)))
			return 1;

		if (col_count == 1) {
			// only fft/dft
			tmp[2] = tmp[0];
			tmp[0] = 0;
			tmp[1] = 0;
		} else if (col_count == 2) {
			// sum columns and fft/dft
			tmp[2] = tmp[1];
			tmp[1] = tmp[0];
			tmp[0] = 0;
		}

		for (int j = 0; j < 3; j++) {
			max[j] = (tmp[j] > max[j]) ? tmp[j] : max[j];
			mean[j] += tmp[j];
		}
	}

	// Close the file in reading mode
	fclose(bench_desc);

	// Reopen the file in append mode
	if ((bench_desc = fopen(filename, "a")) == NULL) {
		fprintf(stderr, "Error trying to open the file: %s\n", filename);
		return 1;
	}

	// Evaluate the average times
	for (int j = 0; j < 3; j++) 
		mean[j] /= BENCHN;

	// Evaluate FPS, sensor rate
	val = (float) BENCHN / ((float)(totaltimes[1].tv_sec - totaltimes[0].tv_sec) +
		(totaltimes[1].tv_usec - totaltimes[0].tv_usec) / 1e6);
	

	// Print the average and max values at the end of the file
	fprintf(bench_desc, "-----------------------------------------------------------\n\n");
	fprintf(bench_desc, "RESX:        \t%d\n", RESX);
	fprintf(bench_desc, "RESY:        \t%d\n", RESY);
	fprintf(bench_desc, "ZPUPS:       \t%d\n", ZPUPS);
	fprintf(bench_desc, "UPS:         \t%d\n\n", UPS);
	fprintf(bench_desc, "Computed FPS:\t%f\n\n", val);
	fprintf(bench_desc, "         \tDecompress \tSum cols \tFFT/DFT \tTotal time (sum + FFT) \n");
	fprintf(bench_desc, "Average: \t%.8f \t%.8f \t%.8f \t%.8f\n", mean[0], mean[1], mean[2], mean[1]+mean[2]);
	fprintf(bench_desc, "Max:     \t%.8f \t%.8f \t%.8f \t%.8f\n", max[0], max[1], max[2], max[1]+max[2]);

	// Close the file
	fclose(bench_desc);
	
	return 0;
}

// Set the running flag, enabling the writing to the log
void bench_start() {
	running = 1;
	gettimeofday(&totaltimes[0], NULL);
}

// Set the running flag, disabling the writing to the log
void bench_stop() {
	running = 0;
	gettimeofday(&totaltimes[1], NULL);
}

void bench_start_time() {
	gettimeofday(&times[0], NULL);
}

void bench_stop_time() {
	gettimeofday(&times[1], NULL);
}


// Write decompress timing
int bench_write_decompress() {
	float val;

	// Do not write if running is 0
	if (!running) return 0;

	val = (float)(times[1].tv_sec - times[0].tv_sec) + (times[1].tv_usec - times[0].tv_usec) / 1e6;
	if (fprintf(bench_desc, "%.8f ", val) <= 0)
		return 1;

	return 0;
}

// Write sum of columns timing
int bench_write_sum() {
	float val;

	// Do not write if running is 0
	if (!running) return 0;

	val = (float)(times[1].tv_sec - times[0].tv_sec) + (times[1].tv_usec - times[0].tv_usec) / 1e6;
	if (fprintf(bench_desc, "%.8f ", val) <= 0)
		return 1;

	return 0;
}

// Write computation timing (FFT, DFT)
int bench_write_computation() {
	float val;

	// Do not write if running is 0
	if (!running) return 0;

	val = (float)(times[1].tv_sec - times[0].tv_sec) + (times[1].tv_usec - times[0].tv_usec) / 1e6;
	if (fprintf(bench_desc, "%.8f\n", val) <= 0){
		return 1;
	}

	return 0;
}

