#############################################
##      Filename:     targets_tree.py      ##
##      Date:         21/04/2020           ##
##      Author:       Ayman HATOUM         ##
#############################################
"""This file contains the class definition of targetsTree(). A subclass
of QTreeWidget() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
import random
from PyQt5.QtCore import Qt
from PyQt5.QtCore import pyqtSignal
from PyQt5.QtWidgets import QTreeWidget
from PyQt5.QtWidgets import QMenu
from PyQt5.QtWidgets import QTreeWidgetItem
from PyQt5.QtGui import QDrag
from src.constants import colorNames
from src.dialogs.modify_target_dialog import modifyTargetDialog
from src.windows.targets.tree.target_tree_item import targetTreeItem
from src.windows.targets.tree.task_mime_data import taskMimeData

#----------------------------------------------------------------------------#

class targetsTree(QTreeWidget) :
    "Targets tree class"
    targetRemoved = pyqtSignal(targetTreeItem)
    modifiedCanID = pyqtSignal(int)
    def __init__(self, parent) :
        logging.getLogger(__name__).debug("Constructing targets tree...")
        super().__init__(parent)
        
        self.mainWindow = parent
        self.headers = ["Name", "Inputs", "Outputs"]
        self.colorList = []
        self.canIDList = []
        self.labelList = []

        self.setHeaderLabels(self.headers)
        self.setAlternatingRowColors(True)
        self.setDragEnabled(True)
        self.setMinimumWidth(self.header().length())
        self.setMinimumHeight(500)

        self.itemCollapsed.connect(self.targetsTreeModified)
        self.itemExpanded.connect(self.targetsTreeModified)

    def getTargetName(self, canID) :
        logging.getLogger(__name__).debug("Getting target name of CAN ID [{}]...".format(canID))
        for index in range(self.topLevelItemCount()) :
            item = self.topLevelItem(index)
            if item.canID == canID :
                return item.data(0, Qt.DisplayRole)
        return ""

    def getTarget(self, canID) :
        logging.getLogger(__name__).debug("Getting target tree item of CAN ID [{}]...".format(canID))
        for index in range(self.topLevelItemCount()) :
            item = self.topLevelItem(index)
            if item.canID == canID :
                return item.targetData
        return None

    def getTask(self, targetIndex, taskName) :
        logging.getLogger(__name__).debug("Getting '{}' task tree item from target index [{}]...".format(taskName, targetIndex))
        target = self.topLevelItem(int(targetIndex))
        for task, index in zip(target.targetData.tasks, range(len(target.targetData.tasks))) :
            if task.name == taskName :
                break
        return target.child(index)

    def targetsTreeModified(self) :
        if self.isWindowModified() :
            return 
        self.setWindowModified(True)
        self.mainWindow.setWindowModified(True)

    def saveTargets(self) :
        logging.getLogger(__name__).debug("Saving targets on tree...")
        self.setWindowModified(False)
        data = {}
        data["targets"] = []
        count = self.topLevelItemCount()
        if count :
            for index in range(count) :
                itemData = {}
                item = self.topLevelItem(index)
                itemData["name"] = item.targetData.name
                itemData["hw"] = item.targetData.hw
                itemData["sw"] = item.targetData.sw
                itemData["label"] = item.label
                itemData["can id"] = item.canID
                itemData["color"] = item.color
                itemData["expanded"] = item.isExpanded()
                data["targets"].append(itemData)
        return data

    def updateSmallestCanIDAvailable(self, value) :
        logging.getLogger(__name__).debug("Updating smallest CAN ID available...")
        while value in self.canIDList :
            value = value + 1
        return value

    def setupColor(self) :
        color = random.choice(colorNames)
        while color in self.colorList :
            color = random.choice(colorNames)
        return color

    def clearTargets(self) :
        logging.getLogger(__name__).debug("Clearing targets from tree...")
        self.colorList = []
        self.canIDList = []
        self.labelList = []
        self.clear()
        self.setWindowModified(False)        

    def loadTarget(self, target, label, canID, color, expanded) :
        logging.getLogger(__name__).debug("Loading target with targetData: {}, label: '{}', CAN ID: [{}], color: '{}'...".format(target, label, canID, color))
        item = targetTreeItem(target, label , canID, color)
        self.colorList.append(color)
        self.canIDList.append(canID)
        self.labelList.append(label)
        self.canIDList.sort()
        self.insertTopLevelItem(self.canIDList.index(canID), item)
        self.resizeColumnToContents(0)
        item.setExpanded(expanded)
                
    def addTarget(self, target, label, canID) :
        color = self.setupColor()
        logging.getLogger(__name__).debug("Adding target with targetData: {}, label: '{}', CAN ID: [{}], color: '{}'...".format(target, label, canID, color))
        item = targetTreeItem(target, label , canID, color)
        self.colorList.append(color)
        self.canIDList.append(canID)
        self.labelList.append(label)
        self.canIDList.sort()
        self.insertTopLevelItem(self.canIDList.index(canID), item)
        self.resizeColumnToContents(0)
        self.targetsTreeModified()
        
    def contextMenuEvent(self, event) :
        logging.getLogger(__name__).debug("Context menu event invoked...")
        contextMenu = QMenu(self)        
        modifyTargetAction = contextMenu.addAction("Modify Target")
        modifyTargetAction.triggered.connect(self.modifyTarget)
        modifyTargetAction.setEnabled(False)
        removeTargetAction = contextMenu.addAction("Remove Target")     
        removeTargetAction.triggered.connect(self.removeTarget)     
        removeTargetAction.setEnabled(False)   
        contextMenu.addSeparator()
        contextMenu.addAction(self.mainWindow.allActions.newTargetAction)
        
        self.contextMenuItem = self.itemAt(event.pos())
        if self.contextMenuItem :
            if self.contextMenuItem.type() == QTreeWidgetItem.UserType :
                modifyTargetAction.setEnabled(True)
                removeTargetAction.setEnabled(True)  
                
        contextMenu.exec(event.globalPos())
        
    def modifyTarget(self) :
        logging.getLogger(__name__).debug("Modify target action triggered on {}...".format(self.contextMenuItem))
        color = self.contextMenuItem.color
        canID = self.contextMenuItem.canID
        label = self.contextMenuItem.label
        self.colorList.remove(color)
        self.canIDList.remove(canID)
        self.labelList.remove(label)
        
        dialog = modifyTargetDialog(self.mainWindow, label, canID, color)
        if dialog.exec() :
            logging.getLogger(__name__).debug("Modify target dialog accepted with {}...".format(dialog.data))
            self.contextMenuItem.color = dialog.data["Painting color"]
            self.contextMenuItem.canID = dialog.data["CAN ID"]
            self.contextMenuItem.label = dialog.data["Target label"]
            self.colorList.append(dialog.data["Painting color"])
            self.canIDList.append(dialog.data["CAN ID"])
            self.labelList.append(dialog.data["Target label"])
            self.canIDList.sort()
            expandState = self.contextMenuItem.isExpanded()
            item = self.takeTopLevelItem(self.indexOfTopLevelItem(self.contextMenuItem))
            self.insertTopLevelItem(self.canIDList.index(item.canID), item)
            item.setExpanded(expandState)
            self.modifiedCanID.emit(canID)
            self.targetsTreeModified()
        else :
            logging.getLogger(__name__).debug("Modify target dialog canceled...")
            self.colorList.append(color)
            self.canIDList.append(canID)
            self.labelList.append(label)
            self.canIDList.sort()       
        
    def removeTarget(self) :
        logging.getLogger(__name__).debug("Remove target action triggered on {}...".format(self.contextMenuItem))
        self.targetRemoved.emit(self.contextMenuItem)
        item = self.takeTopLevelItem(self.indexOfTopLevelItem(self.contextMenuItem))
        self.colorList.remove(item.color)
        self.canIDList.remove(item.canID)
        self.labelList.remove(item.label)
        self.targetsTreeModified()
        
    def startDrag(self, dropAction) :
        logging.getLogger(__name__).debug("Starting task tree item drag action...")
        task = self.selectedItems()[0]
        if task :
            drag = QDrag(self)
            drag.setMimeData(taskMimeData(task))
            drag.exec(Qt.CopyAction)
            
#----------------------------------------------------------------------------#