###############################################
##      Filename:     targets_window.py      ##
##      Date:         18/04/2020             ##
##      Author:       Ayman HATOUM           ##
###############################################
"""This file contains the class definition of targetsWindow(). A subclass
of QDockWidget() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import json
import jsonschema
import logging
import pathlib
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QDockWidget
from src.constants import targetsPath
from src.constants import schemaPath
from src.constants import whatsThisMsg
from src.windows.targets.tree.targets_tree import targetsTree
from src.windows.targets.target_class import targetClass

#----------------------------------------------------------------------------#

class targetsWindow(QDockWidget) :
    "Targets class window"
    def __init__(self, parent) :
        logging.getLogger(__name__).debug("Constructing targets window...")
        super().__init__(parent)
        
        self.setWindowTitle("Targets")
        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        
        self.mainWindow = parent

        #Fetch available targets
        self.fetchTargetsData()
        self.smallestCanIDAvailable = 0
        
        self.tree = targetsTree(parent)
        self.setWidget(self.tree)
        self.tree.targetRemoved.connect(self.targetRemoved)
        self.tree.modifiedCanID.connect(self.targetModified)

        self.setWhatsThis(whatsThisMsg.format("Targets Window", "Here you can add targets and display all their parameters and child tasks. <strong>Drag</strong> task items and <strong>Drop</strong> them in <i>Builder</i> for building your project."))

    def resetTargets(self) :
        logging.getLogger(__name__).debug("Resetting targets window...")
        self.tree.clearTargets()
        self.smallestCanIDAvailable = 0

    def targetModified(self, value) :
        logging.getLogger(__name__).debug("Target modified invoked...")
        if value < self.smallestCanIDAvailable :
            self.smallestCanIDAvailable = value
        self.smallestCanIDAvailable = self.tree.updateSmallestCanIDAvailable(self.smallestCanIDAvailable)

    def targetRemoved(self, item) :
        logging.getLogger(__name__).debug("Target removed invoked for freeing CAN ID...")
        if item.canID < self.smallestCanIDAvailable :
            self.smallestCanIDAvailable = item.canID

    def targetConfigSchemaCheck(self, config, file) :
        try :
            with open(schemaPath+"target_config_schema.json", "r", encoding = "utf-8") as outFile :
                schema = json.load(outFile)
        except json.JSONDecodeError as err :
            logging.getLogger(__name__).error("While loading target config schema, JSON decoder failed!\n\n{} in {}.\n".format(str(err), schemaPath+"target_config_schema.json"))
            return False
        except :
            logging.getLogger(__name__).critical("While loading target config schema, UNEXPECTED exception occurred!\n\n", exc_info = True)
            return False

        try :
            jsonschema.validate(config, schema)
        except jsonschema.SchemaError as err :
            logging.getLogger(__name__).error("While loading target config schema!\n\n{}\n".format(err))
            return False
        except jsonschema.ValidationError as err :
            instance = "On instance"
            for item in err.path :
                instance = instance+"[{}]".format(item) if type(item) == int else instance+"['{}']".format(item)
            logging.getLogger(__name__).warning("Target config {} doesn't meet schema requirements!\n\n{}.\n\n{}:\n{}\n".format(file, err.message, instance, json.dumps(err.instance, indent = 2, ensure_ascii = False)))
            return False
        else :
            logging.getLogger(__name__).debug("Target config {} schema check was successfully passed...".format(file))
            return True  

    def fetchTargetsData(self) :
        self.targetsData = {}
        targetsFiles = list(pathlib.Path(targetsPath).glob("*.json"))
        logging.getLogger(__name__).debug("Fetching targets data from {}...".format(targetsFiles))
        if targetsFiles == [] :
            logging.getLogger(__name__).warning("There are NO target's config files (.json) in {}!".format(targetsPath))
            self.mainWindow.allActions.newTargetAction.setEnabled(False)
        else :
            for file in targetsFiles :
                try :
                    with open(file, "r", encoding = "utf-8") as f :
                        data = json.load(f)
                        if self.targetConfigSchemaCheck(data, file) :
                            self.targetsData[data["name"]] = targetClass(data)
                        else :
                            logging.getLogger(__name__).error("Could NOT fetch target's config file {}!".format(file))
                except json.JSONDecodeError as err :
                    logging.getLogger(__name__).error("While fetching target's config file, JSON decoder failed!\n\n{} in {}.\n".format(str(err), file))
                except :
                    logging.getLogger(__name__).critical("While fetching target's config file, UNEXPECTED exception occurred!\n\n", exc_info = True)
            if self.targetsData == {} :
                self.mainWindow.allActions.newTargetAction.setEnabled(False)
                logging.getLogger(__name__).warning("NO target's config file were fetched!")
            else :
                self.mainWindow.allActions.newTargetAction.setEnabled(True)
                logging.getLogger(__name__).info("Fetched targets: {}.".format(list(self.targetsData.keys())))

    def loadTargets(self, targets) :
        logging.getLogger(__name__).debug("Loading targets...")
        for item in targets :
            target = self.targetsData[item["name"]]
            label = item["label"]
            canID = item["can id"]
            color = item["color"]
            expanded = item["expanded"]
            self.tree.loadTarget(target, label, canID, color, expanded)
        self.smallestCanIDAvailable = self.tree.updateSmallestCanIDAvailable(self.smallestCanIDAvailable)
        self.tree.setWindowModified(False)
    
    def targetHwSwMatch(self, name, hw, sw) :
        logging.getLogger(__name__).debug("Checking target SW/HW match...")
        if not name in self.targetsData :
            errorMessage = "'{}' is not a recognized target.".format(name)
            return (False, errorMessage)

        target = self.targetsData[name]
        if target.hw != hw :
            errorMessage = "HW version: {} of '{}' target doesn't match.".format(target.hw, name)
            return (False, errorMessage)
        if target.sw != sw :
            errorMessage = "SW version: {} of '{}' target doesn't match.".format(target.sw, name)
            return (False, errorMessage)
        
        return (True, "")

    def targetTaskCheck(self, taskName, targetName) :
        logging.getLogger(__name__).debug("Target task check called...")
        target = self.targetsData[targetName]
        task = target.hasTask(taskName)
        if not task :
            errorMessage = "'{}' is not a task of '{}' target.".format(taskName, targetName)
            return (False, [], [], errorMessage)
        
        return (True, task.inputs, task.outputs, "")

    def addNewTargets(self, data) :
        logging.getLogger(__name__).debug("Adding targets {}...".format(data))
        target = self.targetsData[data["Target type"]]
        label = data["Target label"]
        canID = data["CAN ID"] if data["CAN ID"] != None else self.smallestCanIDAvailable
        count = data["Number of targets"]
        if count == 1 :
            self.tree.addTarget(target, label, canID)
        else :    
            for i in range(count) :
                self.tree.addTarget(target, label+"_"+str(i+1), canID+i)
        self.smallestCanIDAvailable = self.tree.updateSmallestCanIDAvailable(self.smallestCanIDAvailable)

#----------------------------------------------------------------------------#