#################################################
##      Filename:     scheduler_window.py      ##
##      Date:         18/04/2020               ##
##      Author:       Ayman HATOUM             ##
#################################################
"""This file contains the class definition of schedulerWindow(). A subclass
of QWidget() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
import pathlib
from PyQt5.QtCore import Qt
from PyQt5.QtCore import pyqtSignal
from PyQt5.QtWidgets import QWidget
from PyQt5.QtWidgets import QVBoxLayout
from src import tools
from src.constants import defaultBaudRate
from src.constants import defaultSlotBuff
from src.constants import defaultRoundBuff
from src.constants import defaultTimeUnit
from src.constants import deafaultSlotRoutine
from src.constants import defaultSlotOverlapBound
from src.windows.central.scheduler.tables.schedule_table import scheduleTable
from src.windows.central.scheduler.schedule.schedule_view import scheduleView
from src.dialogs.progress_dialog import progressDialog

#----------------------------------------------------------------------------#

class schedulerWindow(QWidget) :
    "Scheduler class window"
    generateFinish = pyqtSignal(tuple)
    def __init__(self, parent) :
        logging.getLogger(__name__).debug("Constructing scheduler window...")
        super().__init__(parent)

        self.mainWindow = parent

        self.baudRate = defaultBaudRate
        self.timeUnit = defaultTimeUnit
        self.slotRoutine = deafaultSlotRoutine
        self.slotBuff = defaultSlotBuff
        self.slotOverlapBound = defaultSlotOverlapBound
        self.roundBuff = defaultRoundBuff

        self.timeSlots = None

        self.createLayout()

    def restart(self) :
        logging.getLogger(__name__).debug("Restarting schedule window...")
        self.timeSlots = None
        self.schedTable.restart()
        self.schedule.restart()

    def createLayout(self) :
        self.layout = QVBoxLayout(self)

        self.schedTable = scheduleTable(self.mainWindow)
        self.schedule = scheduleView(self.mainWindow)

        self.layout.addWidget(self.schedTable)
        self.layout.addWidget(self.schedule, 1)

    def saveSettings(self) :
        logging.getLogger(__name__).debug("Saving preference settings...")
        self.setWindowModified(False)
        data = {}
        prefSettings = {}
        data["preferred settings"] = prefSettings
        prefSettings["baud rate"] = self.baudRate
        prefSettings["time unit"] = self.timeUnit
        prefSettings["slot routine"] = self.slotRoutine
        prefSettings["slot buffer"] = self.slotBuff
        prefSettings["slot overlap bound"] = self.slotOverlapBound
        prefSettings["round buffer"] = self.roundBuff

        return data

    def loadSettings(self, data) :
        logging.getLogger(__name__).debug("Loading preference settings...")
        self.baudRate = data["baud rate"]
        self.timeUnit = data["time unit"]
        self.slotRoutine = data["slot routine"]
        self.slotBuff = data["slot buffer"]
        self.slotOverlapBound = data["slot overlap bound"]
        self.roundBuff = data["round buffer"]

        self.setWindowModified(False)

    def updateSettings(self, data) :
        logging.getLogger(__name__).debug("Updating preference settings...")
        self.baudRate = data["Baud rate"]
        self.timeUnit = data["Time unit"]
        self.slotRoutine = data["Slot routine"]
        self.slotBuff = data["Slot buffer"]
        self.slotOverlapBound = data["Slot overlap bound"]
        self.roundBuff = data["Round buffer"]
        if self.isWindowModified() :
            return
        self.setWindowModified(True)
        self.mainWindow.setWindowModified(True)

    def updateDisplaySettings(self, data) :
        axisRes = data["Axis resolution"]
        self.schedule.scene.updateAxisRes(axisRes)

    def displaySchedTimeline(self, data, period, slotSeparators) :
        logging.getLogger(__name__).debug("Displaying schedule timeline...")
        timelineData = {ID:[] for ID in data.keys()}
        for target, taskSet in data.items() :
            for task in taskSet :
                taskItem = self.mainWindow.central.builder.scene.taskFromId(task.taskID)
                taskName = taskItem.name
                taskColor = taskItem.task.parent().color
                timelineData[target].append(tools.taskTimeline(taskName, taskColor, task.offset, task.deadline, task.WCET, task.Tx, task.taskType))
        
        self.schedule.displaySchedTimeline(timelineData, period, slotSeparators, self.baudRate, self.timeUnit)

    def displaySchedTable(self, data) :
        logging.getLogger(__name__).debug("Displaying schedule table: {}...".format(data))
        self.schedTable.restart()
        for slotID, slotData in data.items() :
            slotStart = slotData.start
            slotOverlapBound = slotData.overlapBound
            buffStart = slotData.buffStart
            slotEnd = slotData.end
            for task in slotData.tasks :
                targetName = self.mainWindow.targets.tree.getTargetName(task.targetID)
                taskName = self.mainWindow.central.builder.scene.taskFromId(task.taskID).name
                taskStart = task.offset
                WCET = task.WCET
                txTime = task.Tx
                self.schedTable.addItem(slotID, targetName, taskName, slotStart, slotOverlapBound, buffStart, slotEnd, taskStart, WCET, txTime)

    def generateSchedule(self, data) :
        optimizationThread = tools.optimizationThread(self.mainWindow, data, self.baudRate, self.timeUnit, self.slotBuff, self.slotRoutine, self.slotOverlapBound)
        optimizationThread.start()
        optimizationThread.optDone.connect(self.optimizationThreadFinished)
        
        self.progressDialog = progressDialog(self.mainWindow)
        self.progressDialog.exec()

    def optimizationThreadFinished(self, ret) :
        if not ret[0] :
            self.progressDialog.close()
            self.generateFinish.emit((False, ""))
            return

        self.timeSlots = ret[3]
        slotSeparators = [slot.end for slot in self.timeSlots.values()]
        self.extractScheduleDescription(ret[2])
        self.extractSchedule(ret[4])

        self.displaySchedTable(self.timeSlots)
        self.displaySchedTimeline(ret[1], ret[2], slotSeparators)
        self.progressDialog.close()
        self.generateFinish.emit((True, "{} {}".format(ret[2], self.timeUnit)))

    def extractScheduleDescription(self, period) :
        data = {}
        data["schedule name"] = pathlib.Path(self.mainWindow.currentProjectName).name
        data["baud rate"] = self.baudRate 
        data["time unit"] = self.timeUnit
        data["slot routine"] = self.slotRoutine
        data["slot buffer"] = self.slotBuff
        data["slot overlap bound"] = self.slotOverlapBound
        data["period"] = period
        data["round buffer"] = self.roundBuff
        data["slots"] = []

        for id, item in self.timeSlots.items() :
            slotData = {}
            slotData["id"] = id
            slotData["start"] = item.start
            slotData["overlap bound"] = item.overlapBound
            slotData["buffer start"] = item.buffStart
            slotData["end"] = item.end
            slotData["tasks"] = []

            for task in item.tasks :
                taskData = {}
                taskItem = self.mainWindow.central.builder.scene.taskFromId(task.taskID)
                taskData["name"] = taskItem.name
                taskData["target"] = self.mainWindow.targets.tree.getTargetName(task.targetID)
                if taskItem.inputs :
                    taskData["default inputs"] = {}
                    for port in taskItem.inputs :
                        taskData["default inputs"][port.label] = port.defaultValue
                if task.taskType == tools.ConsumerTask :
                    taskType = "consumer"
                elif task.taskType == tools.ProducerTask :
                    taskType = "producer"
                elif task.taskType == tools.FreeTask :
                    taskType = "free"
                else :
                    taskType = "consumer_producer"
                taskData["type"] = taskType
                taskData["start"] = task.offset
                taskData["wcet"] = task.WCET
                taskData["tx"] = task.Tx                
                slotData["tasks"].append(taskData)

            data["slots"].append(slotData)

        self.extractedScheduleDescription = data

    def extractSchedule(self, formulatedData) :
        logging.getLogger(__name__).debug("Extracting schedule data...")
        data = {}
        data["test name"] = pathlib.Path(self.mainWindow.currentProjectName).name
        data["targets"] = []
        data["variables"] = []
        data["tasks"] = []
        data["round"] = {"period" : {}, "buffer" : {}, "slots" : []}

        targetIDs = []
        for target, taskSet in formulatedData.items() :
            for task in taskSet :
                if task.consumerDependencies != [] or task.producerDependencies != [] :
                    targetIDs.append(target)
        targetIDs = list(set(targetIDs))
        if targetIDs == [] :
            self.extractedSchedule = data
            return

        for target in targetIDs :
            targetInfo = {}
            targetData = self.mainWindow.targets.tree.getTarget(target)
            targetInfo["name"] = targetData.name
            targetInfo["hw"] = targetData.hw
            targetInfo["sw"] = targetData.sw
            targetInfo["can id"] = target
            data["targets"].append(targetInfo)

        variables = []
        for taskSet in formulatedData.values() :
            for task in taskSet :
                variables.extend(task.consumerDependencies)
        variables = list(set(variables))

        data["variables"].extend(list(map(lambda x: x[0], variables)))

        taskIDs = []
        for target, taskSet in formulatedData.items() :
            for task in taskSet :
                if task.consumerDependencies != [] or task.producerDependencies != [] :
                    taskInfo = {}
                    taskInfo["target"] = targetIDs.index(target)
                    taskInfo["name"] = self.mainWindow.central.builder.scene.taskFromId(task.taskID).name
                    taskInfo["in"] = []
                    for param in task.consumerDependencies :
                        taskInfo["in"].append(variables.index(param))
                    taskInfo["out"] = []
                    for param in variables :
                        if task.taskID == param[1] :
                            taskInfo["out"].append(variables.index(param))
                    data["tasks"].append(taskInfo)
                    taskIDs.append(task.taskID)
        
        for slotID, slot in self.timeSlots.items() :
            slotInfo = {}
            slotInfo["assign"] = {"target" : None, "tasks" : []}
            slotInfo["time"] = {}
            ret = tools.convertTime(slot.start, self.timeUnit)
            slotInfo["time"]["start"] = {"sec" : ret[0], "nsec" : ret[1]}
            ret = tools.convertTime(slot.overlapBound, self.timeUnit)
            slotInfo["time"]["overlap"] = {"sec" : ret[0], "nsec" : ret[1]}
            ret = tools.convertTime(slot.buffStart, self.timeUnit)
            slotInfo["time"]["buffer"] = {"sec" : ret[0], "nsec" : ret[1]}
            ret = tools.convertTime(slot.end, self.timeUnit)
            slotInfo["time"]["end"] = {"sec" : ret[0], "nsec" : ret[1]}
            for task in slot.tasks :
                if task.taskID in taskIDs :
                    slotInfo["assign"]["target"] = targetIDs.index(task.targetID)
                    slotInfo["assign"]["tasks"].append(taskIDs.index(task.taskID))
            data["round"]["slots"].append(slotInfo)

        ret = tools.convertTime(slot.end, self.timeUnit)
        data["round"]["period"] = {"sec" : ret[0], "nsec" : ret[1]}
        ret = tools.convertTime(self.roundBuff, self.timeUnit)
        data["round"]["buffer"] = {"sec" : ret[0], "nsec" : ret[1]}

        self.extractedSchedule = data

#----------------------------------------------------------------------------#