##############################################
##      Filename:     schedule_view.py      ##
##      Date:         18/06/2020            ##
##      Author:       Ayman HATOUM          ##
##############################################
"""This file contains the class definition of scheduleView(). A subclass
of QGraphicsView() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
import math
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QGraphicsView
from src.constants import whatsThisMsg
from src.windows.central.scheduler.schedule.scene.schedule_scene import scheduleScene

#----------------------------------------------------------------------------#

class scheduleView(QGraphicsView) :
    "Schedule class view"
    def __init__(self, parent) :
        logging.getLogger(__name__).debug("Constructing schedule view...")
        super().__init__(parent)

        self.mainWindow = parent
        self.default = True
        self.zoom = 100

        self.scene = scheduleScene(self)
        self.setScene(self.scene)
        self.setDragMode(QGraphicsView.NoDrag)
        self.setTransformationAnchor(QGraphicsView.AnchorUnderMouse)
        self.setCacheMode(QGraphicsView.CacheBackground)

        self.setSizeAdjustPolicy(QGraphicsView.AdjustToContents)
        
        self.setWhatsThis(whatsThisMsg.format("Schedule View", "Here you can see the schedule timeline after a successful schedule has been generated. You can zoom using your <i>Mouse Wheel</i>. Each target is represented by a set of rows labeled from the left side <i>e.g: T0</i>. The first row of each target displays: non-connected to communication network <strong>Free Tasks</strong> <i>(dashed)</i> and <strong>Consumer Task</strong> <i>(non-dashed)</i>. The <strong>Producer Tasks</strong> are displayed each on a distinct row. You can see the <i>Legend</i> at the end of the schedule timeline."))
        # self.setInteractive(False)

    def displaySchedTimeline(self, data, period, slotSeparators, baudRate, timeUnit) :
        logging.getLogger(__name__).debug("Displaying schedule timeline...")
        self.default = False
        self.scene.displayTimeline(data, period, slotSeparators, baudRate, timeUnit)
        # self.fitInView(self.scene.sceneRect(), Qt.KeepAspectRatio)

    def restart(self) :
        logging.getLogger(__name__).debug("Restarting schedule view...")
        self.default = True
        self.scene.defaultTimeline()
        self.fitInView(self.scene.sceneRect(), Qt.KeepAspectRatio)

    def resizeEvent(self, event) :
        if self.default :
            self.scene.updateAxis(self.viewport().geometry())
        # else :
        #     self.fitInView(self.scene.sceneRect(), Qt.KeepAspectRatio)
        QGraphicsView.resizeEvent(self, event)

    def zoomIn(self) :
        logging.getLogger(__name__).debug("Zooming in...")
        if self.zoom > 300 :
            return 
        zoomInFactor = 1.1
        self.scale(zoomInFactor, zoomInFactor)
        self.zoom = self.zoom*zoomInFactor
        self.mainWindow.toolBar_.updateZoomButt(int(self.zoom))

    def zoomOut(self) :
        logging.getLogger(__name__).debug("Zooming out...")
        if self.zoom < 30 :
            return
        zoomOutFactor = 1 / 1.1
        self.scale(zoomOutFactor, zoomOutFactor)
        self.zoom = self.zoom*zoomOutFactor
        self.mainWindow.toolBar_.updateZoomButt(int(self.zoom))

    def resetZoom(self) :
        logging.getLogger(__name__).debug("Resetting zoom...")
        if self.zoom > 100 :
            zoomFactor = 1 / 1.1
        elif self.zoom < 100 :
            zoomFactor = 1.1
        else :
            return
        recursionFactor = math.log(100/self.zoom)/math.log(zoomFactor)
        self.scale(math.pow(zoomFactor, recursionFactor), math.pow(zoomFactor, recursionFactor))
        self.zoom = 100
        self.mainWindow.toolBar_.updateZoomButt(int(self.zoom))

    def wheelEvent(self, event) :
        logging.getLogger(__name__).debug("Wheel event invoked...")
        if event.modifiers() & Qt.ControlModifier :
            delta = event.angleDelta().y()
            if delta > 0 :
                self.zoomIn()
            elif delta < 0 :
                self.zoomOut()
            return                
        QGraphicsView.wheelEvent(self, event) 

#----------------------------------------------------------------------------#