###############################################
##      Filename:     schedule_scene.py      ##
##      Date:         18/06/2020             ##
##      Author:       Ayman HATOUM           ##
###############################################
"""This file contains the class definition of scheduleScene(). A subclass
of QGraphicsScene() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
from PyQt5.QtCore import QRectF
from PyQt5.QtCore import QRect
from PyQt5.QtWidgets import QGraphicsScene
from src import tools
from src.constants import targetTimelineHeight
from src.constants import tickLength
from src.constants import marginLength
from src.constants import defaultAxisRes
from src.constants import defaultTimeUnit
from src.constants import taskTimelineHeight
from src.windows.central.scheduler.schedule.scene.items.time_axis_item import timeAxisItem
from src.windows.central.scheduler.schedule.scene.items.target_timeline_item import targetTimelineItem
from src.windows.central.scheduler.schedule.scene.items.slot_separator_item import slotSeparatorItem
from src.windows.central.scheduler.schedule.scene.items.legend_item import legendItem

#----------------------------------------------------------------------------#

class scheduleScene(QGraphicsScene) :
    "Schedule class view"
    def __init__(self, parent) :
        logging.getLogger(__name__).debug("Constructing schedule scene...")
        super().__init__(parent)

        self.view = parent

        self.defaultTimeline()

    def defaultTimeline(self) :
        logging.getLogger(__name__).debug("Default time axis...")
        self.axisRes = defaultAxisRes
        self.timeUnit = defaultTimeUnit
        self.clear()
        self.viewPortRect = self.view.viewport().geometry()
        self.axis = timeAxisItem(self.viewPortRect, self.axisRes)
        self.setSceneRect(QRectF(self.viewPortRect))
        self.addItem(self.axis)

    def updateAxisRes(self, axisRes) :
        logging.getLogger(__name__).debug("Updating axis resolution...")
        self.axisRes = axisRes
        if self.view.default :
            self.axis.updateAxisRes(axisRes)
        else :
            self.drawTimeline()

    def updateAxis(self, rect) :
        logging.getLogger(__name__).debug("Updating time axis...")
        self.viewPortRect.setRect(rect.x(), rect.y(), rect.width(), rect.height())
        self.axis.prepareGeometryChange()
        self.setSceneRect(QRectF(rect))

    def computeRatio(self) :
        logging.getLogger(__name__).debug("Computing resolution ratio factor...")
        scale = int(self.axisRes.split(" ")[0])
        unit = self.axisRes.split(" ")[1]
        if unit == self.timeUnit :
            ratio = 1
        elif unit == "ns" :
            if self.timeUnit == "µs" :
                ratio = 1e3
            elif self.timeUnit == "ms" :
                ratio = 1e6
            elif self.timeUnit == "s" :
                ratio = 1e9
            else :
                ratio = 1
        elif unit == "ms" :
            if self.timeUnit == "ns" :
                ratio = 1e-6
            elif self.timeUnit == "µs" :
                ratio = 1e-3
            elif self.timeUnit == "s" :
                ratio = 1e3
            else :
                ratio = 1
        elif unit == "µs" :
            if self.timeUnit == "ns" :
                ratio = 1e-3
            elif self.timeUnit == "ms" :
                ratio = 1e3
            elif self.timeUnit == "s" :
                ratio = 1e6
            else :
                ratio = 1
        else :
            if self.timeUnit == "ns" :
                ratio = 1e-9
            elif self.timeUnit == "µs" :
                ratio = 1e-6
            elif self.timeUnit == "ms" :
                ratio = 1e-3
            else :
                ratio = 1
        return ratio/scale

    def displayTimeline(self, data, period, slotSeparators, baudRate, timeUnit) :
        logging.getLogger(__name__).debug("Displaying complete timeline: {}...".format(data))
        
        self.timelineData = data
        self.period = period
        self.timeUnit = timeUnit
        self.axisRes = "1 {}".format(self.timeUnit) 
        self.slotSeparators = slotSeparators
        self.baudRate = baudRate
        
        self.drawTimeline()

    def drawTimeline(self) :
        logging.getLogger(__name__).debug("Drawing timeline...")
        ratio = self.computeRatio()
        logging.getLogger(__name__).debug("Computed resolution ratio: [{}]...".format(ratio))
        targetsTimelineHeights = {}
        for target, taskSet in self.timelineData.items() :
            producers = len(list(filter(lambda x: x.taskType & tools.ProducerTask != 0, taskSet)))
            targetsTimelineHeights[target] = producers if producers == len(taskSet) else producers + 1      
        self.clear()
        self.setSceneRect(QRectF(0, - sum(targetsTimelineHeights.values())*targetTimelineHeight, (self.period*ratio + 2)*tickLength + 2*marginLength + 140, sum(targetsTimelineHeights.values())*targetTimelineHeight + 50))
        axis = timeAxisItem(QRect(0, 0, (self.period*ratio + 2)*tickLength + 2*marginLength, 30), self.axisRes)
        axis.setPos(0, 0)
        self.addItem(axis)
        yPos = 0
        yMargin = 0
        for item in self.timelineData.items() :
            timeline = targetTimelineItem(item, self.period, targetsTimelineHeights[item[0]], ratio)
            yPos = yPos + targetsTimelineHeights[item[0]]*targetTimelineHeight
            yMargin = yMargin + (targetsTimelineHeights[item[0]] - 1)*((targetTimelineHeight - taskTimelineHeight)/2)
            timeline.setPos(0, - yPos + yMargin)
            self.addItem(timeline)

        for slot in self.slotSeparators :
            separator = slotSeparatorItem(yPos - yMargin + 10)
            separator.setPos(marginLength + slot*ratio*tickLength, - (yPos - yMargin + 10))
            self.addItem(separator)

        leg = legendItem("{} {}".format(self.period, self.timeUnit), self.baudRate)
        leg.setPos((self.period*ratio + 2)*tickLength + 2*marginLength, -70)
        self.addItem(leg)

#----------------------------------------------------------------------------#