#####################################################
##      Filename:     target_timeline_item.py      ##
##      Date:         24/06/2020                   ##
##      Author:       Ayman HATOUM                 ##
#####################################################
"""This file contains the class definition of targetTimelineItem(). A subclass
of QGraphicsItem() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
from PyQt5.QtCore import QRectF
from PyQt5.QtCore import QMarginsF
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QGraphicsItem
from src import tools
from src.constants import targetTimelineHeight
from src.constants import taskTimelineHeight
from src.constants import tickLength
from src.constants import marginLength
from src.windows.central.scheduler.schedule.scene.items.task_timeline_item import taskTimelineItem

#----------------------------------------------------------------------------#

class targetTimelineItem(QGraphicsItem) :
    "Target timeline item class"
    def __init__(self, data, length, height, ratio) :
        logging.getLogger(__name__).debug("Constructing target timeline item...")
        super().__init__()

        self.targetID = data[0]
        self.tasks = data[1]
        self.ratio = ratio
        self.targetTimelineLength = (length*ratio + 2)*tickLength
        self.targetTimelineHeight = targetTimelineHeight*height - (height - 1)*((targetTimelineHeight - taskTimelineHeight)/2)

        self.rect = QRectF(0, 0, self.targetTimelineLength + marginLength, self.targetTimelineHeight)

        self.insertTasks()

    def insertTasks(self) :
        logging.getLogger(__name__).debug("Inserting task timelines...")
        yPos = (targetTimelineHeight - taskTimelineHeight)/2

        producers = len(list(filter(lambda x: x.taskType & tools.ProducerTask != 0, self.tasks)))
        if producers == len(self.tasks) :
            yPosProducer = - taskTimelineHeight
        else :
            yPosProducer = yPos

        for task in self.tasks :
            taskItem = taskTimelineItem(self, task.name, task.WCET*self.ratio, task.Tx*self.ratio, task.color, task.taskType)
            if task.taskType & tools.ProducerTask == 0 :
                taskItem.setPos(marginLength + tickLength*self.ratio*task.offset, yPos)
            else :
                yPosProducer = yPosProducer + yPos + taskTimelineHeight
                taskItem.setPos(marginLength + tickLength*self.ratio*task.offset, yPosProducer)

    def paint(self, painter, options, widget) :
        pen = painter.pen()
        pen.setStyle(Qt.DashLine)
        painter.setPen(pen)
        painter.drawLine(marginLength, 0, self.targetTimelineLength + marginLength, 0)

        font = painter.font()
        font.setPointSize(10)
        painter.setFont(font)
        painter.drawText(QRectF(0, 0, marginLength, self.targetTimelineHeight), Qt.AlignCenter | Qt.TextWordWrap, "T{}".format(self.targetID))

    def boundingRect(self) :
        return self.rect + QMarginsF(1, 1, 1, 1)

#----------------------------------------------------------------------------#