##########################################
##      Filename:     task_item.py      ##
##      Date:         27/04/2020        ##
##      Author:       Ayman HATOUM      ##
##########################################
"""This file contains the class definition of taskItem(). A subclass
of QGraphicsItem() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
from PyQt5.QtCore import Qt
from PyQt5.QtCore import QRectF
from PyQt5.QtCore import QMarginsF
from PyQt5.QtWidgets import QGraphicsItem
from PyQt5.QtWidgets import QGraphicsTextItem
from PyQt5.QtGui import QBrush
from PyQt5.QtGui import QColor
from src.constants import minTaskWidth
from src.constants import minTaskHeight
from src.constants import portRadius
from src.constants import spaceBetweenPorts
from src.constants import limitNumberOfPorts
from src.windows.central.builder.scene.items.task_port_item import taskPortItem

#----------------------------------------------------------------------------#

class taskItem(QGraphicsItem) :
    "Task item class"
    count = 0
    def __init__(self, task, load = False, id = None, label = None, defaults = None) :
        logging.getLogger(__name__).debug("Constructing task item ID: [{}]...".format(id if id != None else taskItem.count))
        super().__init__()
        
        self.task = task
        self.name = task.taskData.name
        self.inputs = []
        self.outputs = []
        
        self.setupRect()
        self.insertInputs()
        self.insertOutputs()
        
        if load :
            self.id = int(id)
            taskItem.count = self.id + 1
            if label :
                self.insertLabel(label)
            else :
                self.label = None
            if defaults :
                self.setInputsDefaultValues(defaults)
        else :
            self.id = taskItem.count
            taskItem.count = taskItem.count + 1
            self.label = None

        self.setCursor(Qt.SizeAllCursor)
        self.setFlag(QGraphicsItem.ItemIsMovable)
        self.setFlag(QGraphicsItem.ItemIsSelectable)

    def inputFromLabel(self, label) :
        logging.getLogger(__name__).debug("Getting '{}' input port from task...".format(label))
        for port in self.inputs :
            if port.label == label :
                return port
        return None

    def outputFromLabel(self, label) :
        logging.getLogger(__name__).debug("Getting '{}' output port from task...".format(label))
        for port in self.outputs :
            if port.label == label :
                return port
        return None

    def setInputsDefaultValues(self, values) :
        logging.getLogger(__name__).debug("Setting inputs default values {}...".format(values))
        for port in self.inputs :
            port.updateDefaultValue(values[port.label])
    
    def setupRect(self) :
        inputsCount = len(self.task.taskData.inputs)
        outputsCount = len(self.task.taskData.outputs)
        if inputsCount <= limitNumberOfPorts and outputsCount <= limitNumberOfPorts : 
            self.taskHeight = minTaskHeight
            self.rect = QRectF(0, 0, minTaskWidth, minTaskHeight)
        elif inputsCount <= outputsCount :
            self.taskHeight = (2*outputsCount + 1)*portRadius*2
            self.rect = QRectF(0, 0, minTaskWidth, self.taskHeight)
        else :
            self.taskHeight = (2*inputsCount + 1)*portRadius*2
            self.rect = QRectF(0, 0, minTaskWidth, self.taskHeight)
    
    def updateLabel(self, label) :
        logging.getLogger(__name__).debug("Updating task's label: '{}'...".format(label))
        if self.label :
            if label == "" :
                self.label.setParentItem(None)
                self.scene().removeItem(self.label)
                self.label = None
            else :
                self.label.setPlainText(label)
                self.label.setPos(0, - self.label.boundingRect().bottom())
        else :
            if label == "" :
                return
            else :
                self.insertLabel(label)
    
    def insertLabel(self, label) :
        logging.getLogger(__name__).debug("Inserting task's label: '{}'...".format(label))
        self.label = QGraphicsTextItem(label, self)
        self.label.setTextInteractionFlags(Qt.TextSelectableByMouse)
        self.label.setTextWidth(minTaskWidth)
        self.label.setCursor(Qt.IBeamCursor)
        font = self.label.font()
        font.setFamily("Courier")
        font.setItalic(True)
        self.label.setFont(font)
        self.label.setPos(0, - self.label.boundingRect().bottom())    
        
    def insertInputs(self) :
        inputs = self.task.taskData.inputs
        logging.getLogger(__name__).debug("Inserting task's inputs {}...".format(inputs))
        inputsCount = len(inputs)
        if inputsCount <= limitNumberOfPorts :
            if inputsCount == 2 :
                port = taskPortItem(self, taskPortItem.Input, inputs[0])
                port.setPos(- portRadius, (self.taskHeight/2) - spaceBetweenPorts - 2*portRadius)
                self.inputs.append(port)
                port = taskPortItem(self, taskPortItem.Input, inputs[1])
                port.setPos(- portRadius, (self.taskHeight/2) + spaceBetweenPorts)
                self.inputs.append(port)
            elif inputsCount == 1 :
                port = taskPortItem(self, taskPortItem.Input, inputs[0])
                port.setPos(- portRadius, (self.taskHeight/2) - portRadius)
                self.inputs.append(port)
            else :
                return
        else :
            inputsHeight = (2*inputsCount + 1)*portRadius*2
            if inputsHeight == self.taskHeight :
                offset = 0
            else :
                offset = (self.taskHeight - inputsHeight)/2
            for value, shift in zip(inputs, range(inputsCount)) :
                port = taskPortItem(self, taskPortItem.Input, value)
                port.setPos(- portRadius, (shift + 1)*spaceBetweenPorts + shift*(2*portRadius) + offset)
                self.inputs.append(port)

    def insertOutputs(self) :
        outputs = self.task.taskData.outputs
        logging.getLogger(__name__).debug("Inserting task's outputs {}...".format(outputs))
        outputsCount = len(outputs)
        if outputsCount <= limitNumberOfPorts :
            if outputsCount == 2 :
                port = taskPortItem(self, taskPortItem.Output, outputs[0])
                port.setPos(minTaskWidth - portRadius, (self.taskHeight/2) - spaceBetweenPorts - 2*portRadius)
                self.outputs.append(port)
                port = taskPortItem(self, taskPortItem.Output, outputs[1])
                port.setPos(minTaskWidth - portRadius, (self.taskHeight/2) + spaceBetweenPorts)
                self.outputs.append(port)
            elif outputsCount == 1 :
                port = taskPortItem(self, taskPortItem.Output, outputs[0])
                port.setPos(minTaskWidth - portRadius, (self.taskHeight/2) - portRadius)
                self.outputs.append(port)
            else :
                return
        else :
            outputsHeight = (2*outputsCount + 1)*portRadius*2
            if outputsHeight == self.taskHeight :
                offset = 0
            else :
                offset = (self.taskHeight - outputsHeight)/2        
            for value, shift in zip(outputs, range(outputsCount)) :
                port = taskPortItem(self, taskPortItem.Output, value)
                port.setPos(minTaskWidth - portRadius, (shift + 1)*spaceBetweenPorts + shift*(2*portRadius) + offset)
                self.outputs.append(port)
            
    def paint(self, painter, options, widget) :
        self.brush = QBrush(QColor(self.task.parent().color))
        font = painter.font()
        pen = painter.pen()
        font.setBold(True)
        pen.setWidth(2)
        painter.setFont(font)
        painter.setPen(pen)
        painter.setBrush(self.brush)
        painter.drawRoundedRect(self.rect, 5, 5)
        painter.drawText(self.rect, Qt.AlignCenter | Qt.TextWordWrap, self.task.parent().label.upper()+":\n"+self.name)
        
        if self.isSelected() :
            pen.setWidth(1)
            pen.setStyle(Qt.DashLine)
            painter.setPen(pen)
            painter.setBrush(Qt.NoBrush)
            rect = (self.rect + QMarginsF(1, 1, 1, 1))| self.childrenBoundingRect()
            painter.drawRect(rect)
        
    def boundingRect(self) :
        return (self.rect + QMarginsF(1, 1, 1, 1))| self.childrenBoundingRect()
            
    def itemChange(self, change, value) :
        return QGraphicsItem.itemChange(self, change, value)
        
    def mousePressEvent(self, event) :
        if event.button() == Qt.RightButton :
            if not self.isSelected() :
                self.scene().clearSelection()
                self.setSelected(True)
        QGraphicsItem.mousePressEvent(self, event)
        
    def mouseDoubleClickEvent(self, event) :
        if event.button() == Qt.LeftButton :
            self.scene().showTaskParametersDialog()
        QGraphicsItem.mouseDoubleClickEvent(self, event)
        
#----------------------------------------------------------------------------#