################################################
##      Filename:     connection_item.py      ##
##      Date:         30/04/2020              ##
##      Author:       Ayman HATOUM            ##
################################################
"""This file contains the class definition of connectionItem(). 
A subclass of QGraphicsItem() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
from PyQt5.QtCore import Qt
from PyQt5.QtCore import QMarginsF
from PyQt5.QtWidgets import QGraphicsItem
from PyQt5.QtGui import QBrush
from PyQt5.QtGui import QColor
from PyQt5.QtGui import QPainterPath
from PyQt5.QtGui import QPainterPathStroker
from src.constants import colorFault
from src.constants import colorSelection

#----------------------------------------------------------------------------#

class connectionItem(QGraphicsItem) :
    "Connection item class"
    def __init__(self, port, outToIn, load = False, loadPort = None) :
        logging.getLogger(__name__).debug("Constructing connection...")
        super().__init__()

        if load :
            self.loadConnection(port, loadPort, outToIn)
        else :
            self.createConnection(port, outToIn)

        self.setZValue(-1)
        self.setFlag(QGraphicsItem.ItemIsSelectable)
    
    def loadConnection(self, outPort, inPort, outToIn) :
        logging.getLogger(__name__).debug("Creating connection from {} to {}...".format(outPort, inPort))
        self.outToIn = outToIn
        self.outPort = outPort
        self.outPort.connections.append(self)
        self.inPort = inPort
        self.inPort.connection = self
        if self.outPort.matchType(inPort.typ) :
            self.valid = True
            self.penStyle = Qt.SolidLine
            self.penColor = inPort.brush.color()
        else :
            self.valid = False
            self.penStyle = Qt.DashLine
            self.penColor = QColor(colorFault)
        
        self.setupPath()

    def createConnection(self, port, outToIn) :
        logging.getLogger(__name__).debug("Creating connection from {}...".format(port))
        self.outToIn = outToIn                      #True if port.direction ==  taskPortItem.Output else False
        if self.outToIn :
            self.outPort = port
            self.inPort = None
            self.setTail(self.outPort.mapToScene(self.outPort.rect().center()))
        else :
            self.inPort = port
            self.outPort = None
            self.setTail(self.inPort.mapToScene(self.inPort.rect().center()))
        
        self.valid = True
        self.penStyle = Qt.SolidLine
        self.penColor = port.brush.color()
        
        self.setupPath()        
    
    def latch(self, port) :
        logging.getLogger(__name__).debug("Latching connection to {}...".format(port))
        if self.outToIn :
            self.inPort = port
            self.prepareGeometryChange()
            self.outPort.connections.append(self)
            if not self.outPort.matchType(port.typ) :
                self.setInValid()
        else :
            self.outPort = port
            self.prepareGeometryChange()
            self.inPort.connection = self
            if not self.inPort.matchType(port.typ) :
                self.setInValid()
    
    def sourcePort(self) :
        logging.getLogger(__name__).debug("Getting connection's source port...")
        return self.outPort if self.outToIn else self.inPort
        
    def setInValid(self) :
        self.valid = False
        self.penStyle = Qt.DashLine
        self.penColor = QColor(colorFault)
        self.update()
    
    def setTail(self, tail) :
        logging.getLogger(__name__).debug("Setting connection's tail...")
        self.tail = tail
        
    def setupPath(self) :
        if self.outToIn :    
            self.path = QPainterPath(self.outPort.mapToScene(self.outPort.rect().center()))
            if self.inPort :
                self.path.lineTo(self.inPort.mapToScene(self.inPort.rect().center()))
            else :
                self.path.lineTo(self.tail)
        else :
            self.path = QPainterPath(self.inPort.mapToScene(self.inPort.rect().center()))
            if self.outPort :
                self.path.lineTo(self.outPort.mapToScene(self.outPort.rect().center()))
            else :
                self.path.lineTo(self.tail)
        self.rect = self.path.boundingRect()
            
    def paint(self, painter, options, widget) :
        self.setupPath()
        painter.setBrush(Qt.NoBrush)
        pen = painter.pen()
        pen.setWidth(3)
        pen.setStyle(self.penStyle)
        if self.isSelected() :
            pen.setColor(QColor(colorSelection))
        else :
            pen.setColor(self.penColor)
        painter.setPen(pen)
        painter.drawPath(self.path)
        
    def boundingRect(self) :
        return self.rect + QMarginsF(200, 200, 200, 200)
        
    def shape(self) :
        path = QPainterPathStroker().createStroke(self.path)
        return path
            
    def itemChange(self, change, value) :
        return QGraphicsItem.itemChange(self, change, value)
        
    def type(self) :
        return QGraphicsItem.UserType + 1
        
#----------------------------------------------------------------------------#